=begin
/***************************************************************************
 *   Copyright (C) 2005,2006,2007 by Siraj Razick                          *
 *   siraj@kdemail.net                                                     *
 *                                                                         *
 *   Translated to Ruby by Richard Dale                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with self program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .        *
 ***************************************************************************/
=end

require 'clockConfig.rb'

class Clock < Plasma::Applet

  slots 'updated(QString,Plasma::DataEngine::Data)',
        :showConfigurationInterface,
        :configAccepted

  def initialize(parent, args)
    super
    @ui = Ui_ClockConfig.new
    @dialog = nil
    @size = Qt::SizeF.new
    @lastTimeSeen = Qt::Time.new
    setHasConfigurationInterface(true)

    cg = config()
    @showTimeString = cg.readEntry("showTimeString", Qt::Variant.new(false)).value
    @showSecondHand = cg.readEntry("showSecondHand", Qt::Variant.new(false)).value
    @pixelSize = cg.readEntry("size", Qt::Variant.new(250)).value
    @timezone = cg.readEntry("timezone", Qt::Variant.new("Local")).value
    @theme = Plasma::Svg.new("widgets/clock", self)
    @theme.setContentType(Plasma::Svg::SingleImage)
    @theme.resize(@pixelSize, @pixelSize)

    timeEngine = dataEngine("time")
    timeEngine.connectSource(@timezone, self)
    timeEngine.setProperty("reportSeconds", Qt::Variant.new(@showSecondHand))
    updated(@timezone, timeEngine.query(@timezone))
    constraintsUpdated()
  end

  def contentSize()
    return @size
  end

  def constraintsUpdated()
    prepareGeometryChange()

    if formFactor() == Plasma::Planar ||
      formFactor() == Plasma::MediaCenter
      @size = Qt::SizeF.new(@theme.size)
      update()
    else
      fm = Qt::FontMetrics.new(Qt::Application.font)
      @size = Qt::SizeF.new(fm.width("00:00:00") * 1.2, fm.height * 1.5)
    end
  end

  def updated(source, data)
    @time = data["Time"].toTime()
    if @time.minute == @lastTimeSeen.minute &&
      @time.second == @lastTimeSeen.second
      # avoid unnecessary repaints
      return
    end

    @lastTimeSeen = @time
    update()
  end

  def showConfigurationInterface() #TODO: Make the size settable
     if @dialog.nil?
       @dialog = KDE::Dialog.new
       @dialog.caption = "Configure Clock"

       @ui.setupUi(@dialog.mainWidget())
       @dialog.setButtons( KDE::Dialog::Ok | KDE::Dialog::Cancel | KDE::Dialog::Apply )
       connect( @dialog, SIGNAL(:applyClicked), self, SLOT(:configAccepted) )
       connect( @dialog, SIGNAL(:okClicked), self, SLOT(:configAccepted) )

       @ui.timeZones.setSelected(@timezone, true)
       @ui.spinSize.setValue(@size.width)
       @ui.showTimeStringCheckBox.setChecked(@showTimeString)
       @ui.showSecondHandCheckBox.setChecked(@showSecondHand)
    end
    @dialog.show
  end

  def configAccepted()
    cg = config()
    @showTimeString = @ui.showTimeStringCheckBox.checkState == Qt::Checked
    @showSecondHand = @ui.showSecondHandCheckBox.checkState == Qt::Checked
    cg.writeEntry("showTimeString", Qt::Variant.new(@showTimeString))
    cg.writeEntry("showSecondHand", Qt::Variant.new(@showSecondHand))
    dataEngine("time").setProperty("reportSeconds", Qt::Variant.new(@showSecondHand))
    update()
    cg.writeEntry("size", Qt::Variant.new(@ui.spinSize.value))
    @theme.resize(@ui.spinSize.value, @ui.spinSize.value)
    tzs = @ui.timeZones.selection

    dataEngine("time").connectSource(@timezone, self)
    constraintsUpdated()
    cg.config().sync()
  end

  def drawHand(p, rotation, handName)
  end

  def paintInterface(p, option, rect)
    tempRect = Qt::RectF.new(0, 0, 0, 0)

    boundSize = rect.size
    elementSize = Qt::Size.new

    p.renderHint = Qt::Painter::SmoothPixmapTransform

    seconds = 6.0 * @time.second - 180
    minutes = 6.0 * @time.minute - 180
    hours = 30.0 * @time.hour - 180 + ((@time.minute / 59.0) * 30.0)

    if formFactor() == Plasma::Horizontal ||
      formFactor() == Plasma::Vertical
      time = @time.toString()
      fm = Qt::FontMetrics.new(Qt::Application.font)

      if @showSecondHand
        p.drawText((rect.width * 0.1).to_i, (rect.height * 0.25).to_i, @time.toString())
      else
        p.drawText((rect.width * 0.1).to_i, (rect.height * 0.25).to_i, @time.toString("hh:mm"))
      end
      return
    end

    @theme.paint(p, Qt::RectF.new(rect), "ClockFace")

    p.save
    p.translate(boundSize.width/2, boundSize.height/2)
    p.rotate(hours)
    elementSize = @theme.elementSize("HourHand")

    p.translate(-elementSize.width/2, -elementSize.width)
    tempRect.size = Qt::SizeF.new(elementSize)
    @theme.paint(p, tempRect, "HourHand")
    p.restore


#     drawHand(p, hours, "SecondHand", 1)
    p.save
    p.translate(boundSize.width()/2, boundSize.height()/2)
    p.rotate(minutes)
    elementSize = @theme.elementSize("MinuteHand")
    elementSize = Qt::Size.new(elementSize.width, elementSize.height)
    p.translate(-elementSize.width/2, -elementSize.width)
    tempRect.size = Qt::SizeF.new(elementSize)
    @theme.paint(p, tempRect, "MinuteHand")
    p.restore

    # Make sure we paint the second hand on top of the others
    if @showSecondHand
      p.save
      p.translate(boundSize.width/2, boundSize.height/2)
      p.rotate(seconds)
      elementSize = @theme.elementSize("SecondHand")
      elementSize = Qt::Size.new(elementSize.width, elementSize.height)
      p.translate(-elementSize.width()/2, -elementSize.width)
      tempRect.size = Qt::SizeF.new(elementSize)
      @theme.paint(p, tempRect, "SecondHand")
      p.restore
    end

    p.save
    @theme.resize(Qt::SizeF.new(boundSize))
    elementSize = @theme.elementSize("HandCenterScrew")
    tempRect.size = Qt::SizeF.new(elementSize)
    p.translate(boundSize.width / 2 - elementSize.width / 2, boundSize.height / 2 - elementSize.height / 2)
    @theme.paint(p, tempRect, "HandCenterScrew")
    p.restore

    if @showTimeString
      if @showSecondHand
        #FIXME: temporary time output
        time = @time.toString()
        fm = Qt::FontMetrics.new(Qt::Application.font)
        p.drawText((rect.width/2 - fm.width(time) / 2),
                  ((rect.height/2) - fm.xHeight*3), @time.toString())
      else
        time = @time.toString("hh:mm")
        fm = Qt::FontMetrics.new(Qt::Application.font)
        p.drawText((rect.width/2 - fm.width(time) / 2),
                  ((rect.height/2) - fm.xHeight*3), @time.toString("hh:mm"))
      end
    end

    @theme.paint(p, Qt::RectF.new(rect), "Glass")
  end
end


class ClockFactory < KDE::PluginFactory
  def create(iface, parent, parentWidget, args, keyword)
    # Note that the global variable name should be unique, and really ought
    # to include the applet id
    $clock = Clock.new(parent, args)
    return $clock
  end
end
