/*
 *   Copyright (C) 2006
 *   Siraj Razick <siraj@kdemail.net>
 *   PhobosK <phobosk@mail.kbfx.org>
 *   see Also AUTHORS
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU Library General Public License version 2 as
 *   published by the Free Software Foundation
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details
 *
 *   You should have received a copy of the GNU Library General Public
 *   License along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include "kbfxbutton.h"

#include <QtGui/QMouseEvent>
#include <QtGui/QDesktopWidget>
#include <Qt3Support/Q3UriDrag>

KbfxButton::KbfxButton ( QWidget * parent, const char *name ) :
		QLabel ( parent,name,Qt::WStaticContents | Qt::WNoAutoErase |Qt::WPaintDesktop )
{
	setLineWidth ( 0 );
	setScaledContents ( false );
	setAcceptDrops ( true );
	loadSkins();
	m_toggle = false;
	QTimer * timer = new QTimer ( this,"Update Timer" );
	connect ( timer,SIGNAL ( timeout() ),this,SLOT ( update() ) );
	timer->start ( 25,false );
	m_tooltip = ConfigInit().m_ToolTip;
}

void
KbfxButton::loadSkins()
{
	this->readConfig();
	m_over_skin = QPixmap ( ConfigInit ().m_KbfxHoverButtonPath );
	m_normal_skin = QPixmap ( ConfigInit ().m_KbfxNormalButtonPath );
	m_pressed_skin = QPixmap ( ConfigInit ().m_KbfxPressedButtonPath );
	m_current_skin = m_normal_skin;
	this->resize ( m_current_skin.width(),m_current_skin.height() );
	this->repaint();
}

void
KbfxButton::loadSkins ( QPixmap hov, QPixmap norm, QPixmap press )
{
	m_over_skin = hov;
	m_normal_skin = norm;
	m_pressed_skin = press;
	m_current_skin = m_normal_skin;
	this->resize ( m_current_skin.size() );
	this->repaint();
}

void
KbfxButton::enterEvent ( QEvent * e )
{
	e = e;
	if ( m_toggle == true )
	{
		m_current_skin = m_pressed_skin;
	}
	else
	{
		m_current_skin = m_over_skin;
		showToolTip ();
	}
	this->resize ( m_current_skin.size() );
	this->update();
}

void
KbfxButton::leaveEvent ( QEvent * e )
{
	e = e;

	if ( m_toggle )
	{
		m_current_skin = m_pressed_skin;
	}
	else
	{
		m_current_skin = m_normal_skin;
	}
	this->resize ( m_current_skin.size() );
	this->update();
	emit leave ();
}

void
KbfxButton::toggle()
{

	m_current_skin = m_normal_skin;
	this->resize ( m_current_skin.size() );
	this->update();
	if ( !hasMouse() )
		m_toggle = false;
}

KbfxButton::~KbfxButton()
{}

void
KbfxButton::mousePressEvent ( QMouseEvent * e )
{
	if ( m_toggle == false )
	{
		m_current_skin = m_pressed_skin;
		m_toggle = true;
		this->resize ( m_current_skin.size() );
		this->update();
		if ( e->button() == Qt::LeftButton )
			emit pressed();
	}
	else
	{
		m_current_skin = m_normal_skin;
		m_toggle = false;
		this->resize ( m_current_skin.size() );
		this->update();
		emit hideMenu();
	}
	emit clicked ();
}

void
KbfxButton::readConfig()
{
	ConfigInit ().read ();
}

void
KbfxButton::paintEvent ( QPaintEvent * pe )
{

	QPainter p;
	if ( !m_current_skin.isNull () )
	{
		QRect r = QRect ( pe->rect().x(),pe->rect().y(),m_current_skin.width(),m_current_skin.height() );
		m_buffer = new QPixmap();
		m_buffer->resize ( r.size() );
		m_buffer->fill ( this, r.topLeft() );

		p.begin ( m_buffer,this );
		p.translate ( -r.x(), -r.y() );
		p.drawPixmap ( r,m_current_skin );
		p.end ();
		bitBlt ( this, r.x(), r.y(),m_buffer, 0, 0,m_current_skin.width(), m_current_skin.height() );
		delete m_buffer;
	}

}


void
KbfxButton::showToolTip ()
{
	if ( !m_tooltip )
		return;

	KbfxToolTip *tooltip = new KbfxToolTip ();
	QRect r = QDesktopWidget ().screenGeometry ( this );
	int popy = ( r.height () - ( tooltip->height () + height () ) );
	QPoint xpoint = ( this->mapToGlobal ( QPoint ( 0, 0 ) ) );
	int popx = xpoint.x ();

	connect ( this, SIGNAL ( leave () ), tooltip, SLOT ( hideToolTip () ) );
	connect ( this, SIGNAL ( clicked () ), tooltip, SLOT ( hideToolTip () ) );
	tooltip->setStartPos ( popx, popy - 2 );
	tooltip->show ();
}


void
KbfxButton::dragEnterEvent ( QDragEnterEvent * e )
{
	e->accept ( Q3UriDrag::canDecode ( e ) );
	kdDebug() << "Accepting KBFX button drag..." << endl;
}

void
KbfxButton::dropEvent ( QDropEvent * e )
{
	QStringList filelist;
	QString _hover, _normal, _pressed, _tmp;
	QPixmap _hover_pix, _normal_pix, _pressed_pix;
	Q3UriDrag::decodeLocalFiles ( e, filelist );

	for ( QStringList::Iterator it = filelist.begin(); it != filelist.end(); ++it )
	{
		_tmp = ( *it );
		kdDebug() << "KBFX button dropped file: " << _tmp << endl;
		if ( _tmp.contains ( "hover", FALSE ) > 0 )
		{
			_hover = _tmp;
			_hover_pix = QImage ( _tmp );
		}
		if ( _tmp.contains ( "normal", FALSE ) > 0 )
		{
			_normal = _tmp;
			_normal_pix = QImage ( _tmp );
		}
		if ( _tmp.contains ( "pressed", FALSE ) > 0 )
		{
			_pressed = _tmp;
			_pressed_pix = QImage ( _tmp );
		}
	}

	kdDebug() << "KBFX button hover: " << _hover << endl
			  << "KBFX button normal: " << _normal << endl
			  << "KBFX button pressed: " << _pressed << endl;

	if ( _hover_pix.isNull () ||
	        _normal_pix.isNull () ||
	        _pressed_pix.isNull () )
	{
		KMessageBox::error ( 0,
		                     tr2i18n ( "Invalid images were dropped!"
		                               "\nCannot set the KBFX button to use these images!" ),
		                     tr2i18n ( "Error setting KBFX button images" ) );
		return;
	}
	loadSkins ( _hover_pix, _normal_pix, _pressed_pix );

	ConfigInit ().m_KbfxNormalButtonPath = _normal;
	ConfigInit ().m_KbfxHoverButtonPath = _hover;
	ConfigInit ().m_KbfxPressedButtonPath = _pressed;
	ConfigInit().writeThemerc ( ConfigInit().m_SpinxThemeName );
}

#include "kbfxbutton.moc"
