/******************************************************************************
*   Copyright (C) 2006-2007 by Riccardo Iaconelli  <ruphy@fsfe.org>           *
*                                                                             *
*   This library is free software; you can redistribute it and/or             *
*   modify it under the terms of the GNU Library General Public               *
*   License as published by the Free Software Foundation; either              *
*   version 2 of the License, or (at your option) any later version.          *
*                                                                             *
*   This library is distributed in the hope that it will be useful,           *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of            *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU          *
*   Library General Public License for more details.                          *
*                                                                             *
*   You should have received a copy of the GNU Library General Public License *
*   along with this library; see the file COPYING.LIB.  If not, write to      *
*   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,      *
*   Boston, MA 02110-1301, USA.                                               *
*******************************************************************************/

#include <QtGui>

#include <KLocale>
#include <KUser>
#include <KDirSelectDialog>
#include <KMessage>
#include <KMessageBoxMessageHandler>
#include <KMessageBox>
#include <KIcon>
#include <KSeparator>
#include <KFileDialog>
#include <kstdguiitem.h>
#include <KStandardDirs>
#include <KConfigGroup>
#include <KDesktopFile>
#include <KMimeType>

#include <knewstuff2/core/entry.h>

#include "ui_createpackage.h"
#include "ui_description.h"
#include "ui_metadata.h"
#include "ui_welcomepage.h"

#include "plasma/package.h"
#include "plasma/packagemetadata.h"

#include "uploader.h"

#include <kdebug.h>

using namespace KNS;

class Uploader::Private
{
    private:
        Private(Uploader *parent);

        KMessageBoxMessageHandler *handler;

        QString m_appName;

        QStringList contentType;
        QStringListModel contentTypeModel;
        QStringList requiredVersion;
        QStringListModel requiredVersionModel;

        QString m_license;
        QString creationSuggestedFilename;

        bool alreadyAnsweredOverwrite;
        KUser user;

        QString mimeType;
        QString extension;
        QString packageType;
        QString rcFile;

        bool alreadyPackaged;
        bool askForMainFile;
        QWidgetItem *label, *urlRequester;

        Ui::WelcomePage welcomeUI;
        KPageWidgetItem* welcomePage;
        Ui::MetaDataInput metadataUI;
        KPageWidgetItem* metadataPage;
        Ui::DescriptionInput descriptionUI;
        KPageWidgetItem* descriptionPage;
        Ui::CreatePackage createPackageUI;
        KPageWidgetItem* createPackagePage;
        friend class Uploader;
};

Uploader::Private::Private(Uploader *p /*parent*/)
{
    QWidget* widget = new QWidget(p);
    welcomeUI.setupUi(widget);
    welcomePage = new KPageWidgetItem(widget);
    p->addPage(welcomePage);

    widget = new QWidget(p);
    metadataUI.setupUi(widget);
    metadataPage = new KPageWidgetItem(widget);
    p->addPage(metadataPage);

    widget = new QWidget(p);
    descriptionUI.setupUi(widget);
    descriptionPage = new KPageWidgetItem(widget);
    p->addPage(descriptionPage);

    widget = new QWidget(p);
    createPackageUI.setupUi(widget);
    createPackagePage = new KPageWidgetItem(widget);
    p->addPage(createPackagePage);

    contentTypeModel.setStringList(contentType);
    metadataUI.creationLanguage->setModel(&contentTypeModel);

    requiredVersionModel.setStringList(requiredVersion);
    metadataUI.creationRequiredVersion->setModel(&requiredVersionModel);

    askForMainFile = true;
    alreadyPackaged = false;
}

Uploader::Uploader(const QString &appName, QWidget *parent)
   : KAssistantDialog(parent),
     d(new Private(this))
{
    showButton(KDialog::Help, false);

    connectSignalsAndSlots();
    initializeSpecialWidgets();

    d->m_appName = appName;
    setExtension(".kns"); // Default extension for packages
    setMimeType("application/x-kns");

//     KMessageBoxMessageHandler *handler = new KMessageBoxMessageHandler;
//     KMessage::setMessageHandler(handler);

    setWindowIcon(KIcon("plasmagik"));
    adjustSize();
    //resize(minimumSizeHint());
}

Uploader::~Uploader()
{
    delete d;
}

#warning remove this function before release.
void Uploader::fillFields() //WARNING: remove this before release
{
    d->metadataUI.creationName->setText("FooPlasmoid");
    d->metadataUI.creationAuthor->setText("Mr. Konqui");
    d->metadataUI.creationVersion->setText("0.1");
    d->metadataUI.creationWebsite->setText("http://www.kde.org");
    d->metadataUI.creationEmail->setText("konqui@kde.org");
    d->metadataUI.creationLicense->setCurrentIndex(1); //GPL
    d->metadataUI.creationLanguage->setCurrentIndex(1); //C++
    d->descriptionUI.creationDescription->setText("The cooler creation I've ever written.");
    d->descriptionUI.creationNotes->setText("Just have fun! =D");
}

void Uploader::initializeSpecialWidgets()
{
    //Second page
    d->metadataUI.creationWebsite->setText("http://");
    d->metadataUI.creationAuthor->setText(d->user.fullName());
    d->metadataUI.creationIcon->setFilter("image/png image/svg+xml");

    //Third page
    d->createPackageUI.destFile->setUrl(d->user.homeDir());
    d->createPackageUI.firstFile->setUrl(d->user.homeDir());
    d->createPackageUI.sourcePathRequester->setUrl(d->user.homeDir());
    d->createPackageUI.sourcePathRequester->setMode(KFile::Directory);
    d->createPackageUI.destFile->fileDialog()->setOperationMode(KFileDialog::Saving);
    d->createPackageUI.destFile->setMode(KFile::LocalOnly);
}

void Uploader::connectSignalsAndSlots()
{
    connect(d->metadataUI.creationName, SIGNAL(textChanged(const QString &)),
            this, SLOT(creationNameChanged(const QString &)));
    connect(d->metadataUI.creationVersion, SIGNAL(textChanged(const QString &)),
            this, SLOT(creationVersionChanged(const QString &)));

    connect(d->createPackageUI.destFile, SIGNAL(urlSelected(const KUrl &)),
            this, SLOT(checkNotToBeOverwriting(const KUrl &)));

    connect(button(KDialog::User1), SIGNAL(clicked()),
            this, SLOT(checkInformation()));
/*    connect(d->metadataUI.uploadButton, SIGNAL(clicked()),
            this, SLOT(doUpload()));*/
}

void Uploader::doUpload()
{

}

void Uploader::creationNameChanged(const QString &newtext)
{
    d->creationSuggestedFilename = d->user.homeDir();
    d->creationSuggestedFilename += '/';
    d->creationSuggestedFilename += newtext;
    d->creationSuggestedFilename += '-';
    d->creationSuggestedFilename += d->metadataUI.creationVersion->text();
    d->creationSuggestedFilename += d->extension;
    d->createPackageUI.destFile->lineEdit()->setText(d->creationSuggestedFilename);
}

void Uploader::creationVersionChanged(const QString &newtext)
{
    d->creationSuggestedFilename = d->user.homeDir();
    d->creationSuggestedFilename += '/';
    d->creationSuggestedFilename += d->metadataUI.creationName->text();
    d->creationSuggestedFilename += '-';
    d->creationSuggestedFilename += newtext;
    d->creationSuggestedFilename += d->extension;
    d->createPackageUI.destFile->lineEdit()->setText(d->creationSuggestedFilename);
}

void Uploader::next()
{
    if (currentPage() == d->metadataPage && /* We are moving from the second to the third page */
        d->metadataUI.creationLicense->currentIndex() == 10 /* License is 'custom' */) {
        //TODO: ask for the file and add it for them!
        KMessage::message(KMessage::Information,
                          i18n("You choose to use a custom license. Please remember to add "
                               "a file in the root directory of your creation called "
                               "\"COPYING\" that contains your license"));
    }

    KAssistantDialog::next();
}

void Uploader::checkNotToBeOverwriting(const KUrl &url)
{
   QFile f(url.path());

    if (f.exists()) {
        warningOverwrite(url.path());
        d->alreadyAnsweredOverwrite = true;
    }
}

//This just displays a warning that the chosen output file qould be overwritten.
void Uploader::warningOverwrite(const QString &url)
{
    KMessageBox::information(this, i18n("The file %1 already exists. "
              "If you continue, this file will be silently overwritten.", url));
}

// Called in checkInfos()
bool Uploader::isOkToOverwrite()
{
    return ( (KMessageBox::warningContinueCancel(this, i18n("The file %1 already exists. "
            "Do you want to overwrite it?", d->createPackageUI.destFile->lineEdit()->text()),
              QString(), KGuiItem(i18n("Overwrite")), KStandardGuiItem::cancel(), QString())) 
              == KMessageBox::Ok);
}

bool Uploader::createPackage()
{
    QString pathToFirstFile;
    //FIXME check
    if (d->askForMainFile) {
        // use path() because we are not sure to parse things in the right way if
        // 'file://' (or anything else) is prepended.
        pathToFirstFile = d->createPackageUI.firstFile->url().path();
        QString basePath = d->createPackageUI.sourcePathRequester->url().path();

        // If the creation is C++ remove any extension to the name
        // stored in metadata.xml. We will NOT rename the file.
       if (d->metadataUI.creationLanguage->currentText() == "C++") {
           QString *tempFileName = new QString(d->createPackageUI.firstFile->url().fileName());
           QString *tempDirName = new QString(d->createPackageUI.firstFile->url().path());

            // remove the file name from the string
            tempDirName->truncate(tempDirName->lastIndexOf(*tempFileName));
            // remove any extesion from the filename (again, _we will NOT rename the file itself_,
            // the extension is just removed from the filename stored in metadata.xml)
            tempFileName->truncate(tempFileName->lastIndexOf('.'));

            pathToFirstFile = *tempDirName + *tempFileName;

            delete tempFileName;
            delete tempDirName;
        }

        // extract the relative path to the first file to be loaded from the absolute path
        // with a '/' as first char. At this moment / is the root directory of the sources of the creation.
        pathToFirstFile.remove(pathToFirstFile.indexOf(basePath),
                          basePath.size());

        if (pathToFirstFile.indexOf('/') != 0) {
            pathToFirstFile.prepend('/');
        }

        // to have the relative path form the point of view the file metadata.xml
        // prepend the name of the folder that contains the creation sources.
        // that is the name of the creation itself.
        pathToFirstFile.prepend(d->metadataUI.creationName->text());
    }

    Plasma::PackageMetadata metadata;
    metadata.setName(d->metadataUI.creationName->text());
    metadata.setDescription(d->descriptionUI.creationDescription->text());
    metadata.setAuthor(d->metadataUI.creationAuthor->text());
    metadata.setEmail(d->metadataUI.creationEmail->text());
    metadata.setVersion(d->metadataUI.creationVersion->text());
    metadata.setWebsite(d->metadataUI.creationWebsite->text());
    metadata.setLicense(d->m_license);
    metadata.setMainFile(pathToFirstFile);
    metadata.setApplication(d->m_appName);
    metadata.setRequiredVersion("FIXME");
    metadata.setReleaseNotes(d->descriptionUI.creationNotes->toPlainText());
    metadata.setIcon(d->metadataUI.creationIcon->lineEdit()->text());
    metadata.setType(d->packageType);

    kDebug() << metadata.isComplete();
    QString destination = d->createPackageUI.destFile->lineEdit()->text();
    QString directory = d->createPackageUI.sourcePathRequester->lineEdit()->text();
    if (!Plasma::Package::createPackage(metadata, destination, directory)) {
        KMessage::message(KMessage::Error, i18n("Some problem occoured during the "
                "packaging operation."));
        return false;
    }

    KMessage::message(KMessage::Information,
                      "<qt>"+i18nc("The argument is a filename", "Your creation has been "
                      "successfully packaged!<br>You can find it at: <b>%1</b>",
                      d->createPackageUI.destFile->lineEdit()->text())+"</qt>",
                      i18n("Plasmoid Packaged!") );
    d->createPackageUI.packageButton->setEnabled(false);
    d->alreadyPackaged = true;
    return true;
}

void Uploader::checkInformation()
{
    if (!fieldsNotEmpty()) 
    {
        KMessage::message(KMessage::Sorry, i18n("Sorry, you must fill in all the fields before we can "
                          "package your creation. Please fill in all the fields completely."),
                          i18n("Have you filled in all the fields?"));
        return;
    }
    QDir packageDir(d->createPackageUI.sourcePathRequester->lineEdit()->text());
    if (!packageDir.exists()) {
        KMessage::message(KMessage::Sorry, i18n("The folder that contains the content "
                                                "you want to package does not exist"));
        return;
    } else {
        // Always show this message if the dir exists.
        if (KMessageBox::warningContinueCancel(this,
            "<qt>"+i18n("You are going to package the %1 folder <br>Are you sure you want to continue?",
            d->createPackageUI.sourcePathRequester->lineEdit()->text())+"</qt>") == KMessageBox::Cancel) {
            return;
        }
    }
    if (!d->alreadyAnsweredOverwrite) {
        if (creationExists()) {
            if (!isOkToOverwrite()) {
                return;
            }
        }
    }
    if (!licenseIsAcceptable()) {
        KMessage::message(KMessage::Sorry, "<qt>"+i18nc("Do _NOT_ translate 'COPYING'",
                          "Sorry, we cannot create the package of your creation because we cannot find "
                          "any acceptable license.\n Have you specified a valid license? If you wish to "
                          "use a custom license, you must make sure to include it in a file called" 
                          "<b>COPYING</b>.")+"</qt>",
                          i18n("Could not find any acceptable license") );
        return;
    }
    if (!fileToLoadExists()) {
        KMessage::message(KMessage::Error, i18n("The file you have specified does not exists. Please "
                          "make sure that you've chosen the right file."));
        return;
    }

    if (packagingHomeDir()) {
        if ((KMessageBox::warningContinueCancel(this,"<qt>"+
            i18n("<b style=\"color: red\">***WARNING, WARNING, WARNING!!!***</b><br>"
            "It seems that you are going to package your ENTIRE home directory. This is not what "
            "you want in the 99.999% of the cases, and in any case it will take a lot of space "
            "on the disk and a lot of time to complete.<br>"
            "Are you sure you want to continue?")
            +"<qt>", QString(), KStandardGuiItem::cont(), KStandardGuiItem::cancel(), QString(),
            KMessageBox::Dangerous)) == KMessageBox::Cancel)
           return;
    }

    createPackage(); //If all the conditions are met, let's package.
}

bool Uploader::licenseIsAcceptable()
{
    int indexLicense = d->metadataUI.creationLicense->currentIndex();

    //m_license is a string that contains the name of the chosen license untraslated.
    //we store it to make use in the generateXML() function.
    switch (indexLicense) {
        case 0: { //Empty
            return false;
        }
        case 1: {
            d->m_license = "GPL";
            return true;
        }
        case 2: {
            d->m_license = "LGPL";
            return true;
        }
        case 3: {
            d->m_license = "BSD";
            return true;
        }
        case 4: {
            d->m_license = "MPL";
            return true;
        }
        case 5: {
            d->m_license = "Creative Commons BY";
            return true;
        }
        case 6: {
            d->m_license = "Creative Commons BY-SA";
            return true;
        }
        case 7: {
            d->m_license = "Creative Commons BY-ND";
            return true;
        }
        case 8: {
            d->m_license = "Creative Commons BY-ND-NC";
            return true;
        }
        case 9: {
            d->m_license = "Creative Commons BY-NC-SA";
            return true;
        }
        case 10: { //Custom license
            KUrl sourcePath = d->createPackageUI.sourcePathRequester->url();
            sourcePath.addPath("COPYING");
            QFile copyingFile(sourcePath.path());

            if (!copyingFile.exists()) {
                d->m_license = QString();
                return false;
            }

            d->m_license = "Custom";
            return true;
        }
        default: {//This shouldn't happen. Never.
            return false;
        }
    }
}

bool Uploader::fileToLoadExists()
{
    KUrl filePath = d->createPackageUI.firstFile->url();
    QFile f(filePath.path());
    if (f.exists()) {
        return true;
    } else {
        return false;
    }
}

inline bool Uploader::packagingHomeDir()
{
    //We clean the path wiht those 2 lines as much as we can to ensure
    //that the comparison happens in a right way.
    //(KUser::homeDir() is QString, not KUrl)
    d->createPackageUI.sourcePathRequester->url().adjustPath(KUrl::RemoveTrailingSlash);
    d->createPackageUI.sourcePathRequester->url().cleanPath();

    if (d->createPackageUI.sourcePathRequester->url() == KUrl(d->user.homeDir())) {
        return true;
    } else {
        return false;
    }
}

bool Uploader::fieldsNotEmpty()
{
    if (d->metadataUI.creationName->text().isEmpty() ||
        d->metadataUI.creationAuthor->text().isEmpty() ||
        d->metadataUI.creationVersion->text().isEmpty() ||
        d->metadataUI.creationEmail->text().isEmpty() ||
        d->metadataUI.creationLicense->currentIndex() == 0 ||
        d->metadataUI.creationLanguage->currentIndex() == 0 ||
        d->descriptionUI.creationDescription->text().isEmpty() ||
        d->createPackageUI.sourcePathRequester->url().prettyUrl().isEmpty() ||
        d->createPackageUI.destFile->url().fileName().isEmpty() ||
        (d->createPackageUI.firstFile->url() == d->user.homeDir()) ||
        (d->createPackageUI.firstFile->url().fileName().isEmpty() && d->askForMainFile)
        )
    return false;

    else return true;
}

bool Uploader::creationExists()
{
    QFile creation(d->createPackageUI.destFile->url().path());

    if (!creation.exists()) {
        return false;
    } else {
        return true;
    }
}

// ******************************************************************
// Public functions for the API
// ******************************************************************

void Uploader::setAskForMainFile(bool visible)
{
    d->createPackageUI.firstFileLabel->setVisible(visible);
    d->createPackageUI.firstFile->setVisible(visible);
    d->askForMainFile = visible;
}

bool Uploader::askForMainFile()
{
    return d->askForMainFile;
}

void Uploader::clearContentTypes()
{
    int count = d->contentTypeModel.rowCount();
    if (count > 0) {
        d->contentTypeModel.removeRows(0, count);
    }
}

void Uploader::addContentTypes(const QString &items)
{
//     QString temp_items = items;
//     if (position == 0 && temp_items.indexOf('|') != 1)
//     {
//         temp_items.prepend('|');
//     }
    addContentTypes(items.split('|'));
}

void Uploader::addContentTypes(const QStringList &items)
{
    if (items.size() > 0) {
        d->contentTypeModel.insertRows(d->contentTypeModel.rowCount(), items.size());
        QString item;
        int i = 0;
        QModelIndex index;

        foreach(item, items) {
            index = d->contentTypeModel.index(i);
            d->contentTypeModel.setData(index, item);
            i++;
        }

    } else {
        kWarning() << "No content type has been added";
    }
}

void Uploader::setContentTypes(const QString &items)
{
    setContentTypes(items.split('|'));
}

void Uploader::setContentTypes(const QStringList &items)
{
    clearContentTypes();
    addContentTypes(items);
}

void Uploader::setExtension(const QString &extension)
{
    d->extension = extension;
}

void Uploader::setMimeType(const QString &mimeType)
{
    d->mimeType = mimeType;
//     FIXME huh? why the line below makes the application crash?
//     d->createPackageUI.destFile->setFilter(mimeType);
}

void Uploader::setIcon(const QIcon &icon)
{
    setIcon(icon);
}

void Uploader::setText(Uploader::Label label, const QString &text)
{
    if(label.testFlag(Uploader::WelcomeText))  {
        d->welcomeUI.firstLabel->setText(text);
    } else if(label.testFlag(Uploader::ReleaseNotes)) {
        d->descriptionUI.releaseNotesLabel->setText(text);
    }
}

void Uploader::setRcFile(const QString &file)
{
    d->rcFile = file;
}

QString Uploader::rcFile()
{
    return d->rcFile;
}

//TODO: maybe add some more packages/proprieties to set?
void Uploader::setPackageType(const QString& type)
{
    d->packageType = type;
}

#include "uploader.moc"






