/***********************************************************************************
* Fancy Tasks: Plasmoid providing a fancy representation of your tasks and launchers.
* Copyright (C) 2009-2010 Michal Dutkiewicz aka Emdek <emdeck@gmail.com>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*
***********************************************************************************/

#include "FancyTasksToolTip.h"
#include "FancyTasksPreview.h"
#include "FancyTasksTask.h"
#include "FancyTasksLauncher.h"

#include <QTimer>
#include <QPainter>
#include <QVarLengthArray>

#include <Plasma/Theme>

#ifdef Q_WS_X11
#include <QX11Info>

#include <X11/Xlib.h>
#include <fixx11h.h>
#endif

namespace FancyTasks
{

ToolTip::ToolTip(QPixmap pixmap, const QPixmap &icon, const QString &description, Task *task, QWidget *parent) : QWidget(parent)
{
    setWindowFlags(Qt::ToolTip);
    setAttribute(Qt::WA_TranslucentBackground);

    init();

#ifdef Q_WS_X11
    if (KWindowSystem::compositingActive())
    {
        int number;
        Display *display = QX11Info::display();
        Atom atom = XInternAtom(display, "_KDE_WINDOW_PREVIEW", false);
        Atom *list = XListProperties(display, DefaultRootWindow(display), &number);

        if (list != NULL)
        {
            m_previewsAvailable = (qFind(list, (list + number), atom) != (list + number));

            XFree(list);
        }
    }
#endif

    if (task->group())
    {
        m_tasksLayout = new QGridLayout;

        m_mainLayout->addLayout(m_tasksLayout, (m_previewsAvailable?0:2), 0, 1, 2);

        TaskManager::ItemList tasks = task->group()->members();
        TaskManager::GroupManager *groupManager = new TaskManager::GroupManager(this);
        QList<QPixmap> pixmaps;

        int maxHeight = 0;

        for (int i = 0; i < tasks.count(); ++i)
        {
            TaskManager::TaskItem *task = static_cast<TaskManager::TaskItem*>(tasks.at(i));

            if (m_previewsAvailable)
            {
                if (task->task()->window())
                {
                    QSize windowSize = KWindowSystem::windowInfo(task->task()->window(), NET::WMGeometry | NET::WMFrameExtents).frameGeometry().size();

                    if (windowSize.isValid())
                    {
                        if (windowSize.width() > 200 || windowSize.height() > 200)
                        {
                            windowSize.scale(200, 200, Qt::KeepAspectRatio);
                        }

                        pixmap = QPixmap(windowSize);
                        pixmap.fill(Qt::transparent);
                    }
                }
            }

            if (!pixmap.isNull() && pixmap.height() > maxHeight)
            {
                maxHeight = pixmap.height();
            }

            pixmaps.append(pixmap);

            Preview *preview = new Preview(pixmaps.at(i), tasks.at(i)->icon().pixmap(16, 16), task->task()->visibleName(), (maxHeight - pixmaps.at(i).height()), new Task(task, groupManager, NULL), this);

            m_previews.append(preview);

            m_tasksLayout->addWidget(preview, (m_previewsAvailable?0:i), (m_previewsAvailable?i:0), Qt::AlignTop);
        }

        Preview *preview = new Preview(QPixmap(), icon, description, task, this);

        m_mainLayout->addWidget(preview, 1, 0, Qt::AlignLeft);
    }
    else
    {
        if (m_previews.count())
        {
            m_previews.at(0)->deleteLater();
            m_previews.removeAt(0);
        }

        if (m_previewsAvailable && task->windows().at(0))
        {
            QSize windowSize = KWindowSystem::windowInfo(task->windows().at(0), NET::WMGeometry | NET::WMFrameExtents).frameGeometry().size();

            if (windowSize.isValid())
            {
                if (windowSize.width() > 200 || windowSize.height() > 200)
                {
                    windowSize.scale(200, 200, Qt::KeepAspectRatio);
                }

                pixmap = QPixmap(windowSize);
                pixmap.fill(Qt::transparent);
            }
        }

        Preview *preview = new Preview(pixmap, icon, description, 0, task, this);

        m_previews.append(preview);

        m_mainLayout->addWidget(preview, 0, 0, Qt::AlignCenter);
    }

    adjustSize();
}

ToolTip::ToolTip(const QPixmap &pixmap, const QPixmap &icon, const QString &description, Launcher *launcher, QWidget *parent) : QWidget(parent)
{
    setWindowFlags(Qt::ToolTip);
    setAttribute(Qt::WA_TranslucentBackground);

    init();

    Preview *preview = new Preview(pixmap, icon, description, 0, launcher, this);

    m_previews.append(preview);

    m_mainLayout->addWidget(preview, 0, 0, Qt::AlignCenter);

    adjustSize();
}

ToolTip::ToolTip(const QPixmap &icon, const QString &description, QWidget *parent) : QWidget(parent)
{
    setWindowFlags(Qt::ToolTip);
    setAttribute(Qt::WA_TranslucentBackground);

    init();

    Preview *preview = new Preview(icon, description, this);

    m_previews.append(preview);

    m_mainLayout->addWidget(preview, 0, 0, Qt::AlignCenter);

    adjustSize();
}

void ToolTip::init()
{
    m_alreadyShown = false;

    m_previewsAvailable = false;

    m_background = new Plasma::FrameSvg(this);
    m_background->setImagePath("widgets/tooltip");
    m_background->setEnabledBorders(Plasma::FrameSvg::AllBorders);

    m_mainLayout = new QGridLayout(this);
    m_mainLayout->setSpacing(3);

    setLayout(m_mainLayout);

    updateTheme();

    connect(m_background, SIGNAL(repaintNeeded()), this, SLOT(updateTheme()));
}

void ToolTip::paintEvent(QPaintEvent *event)
{
    QPainter painter(this);
    painter.setRenderHint(QPainter::Antialiasing);
    painter.setClipRect(event->rect());
    painter.setCompositionMode(QPainter::CompositionMode_Source);
    painter.fillRect(rect(), Qt::transparent);

    m_background->paintFrame(&painter);
}

void ToolTip::resizeEvent(QResizeEvent *event)
{
    QWidget::resizeEvent(event);

    m_background->resizeFrame(size());

    setMask(m_background->mask());
}

void ToolTip::leaveEvent(QEvent *event)
{
    Q_UNUSED(event)

    QTimer::singleShot(500, this, SLOT(hide()));
}

void ToolTip::mousePressEvent(QMouseEvent *event)
{
    hide(true);

    event->accept();
}

void ToolTip::show()
{
#ifdef Q_WS_X11
    if (m_alreadyShown || (m_previews.count() == 1 && !m_previews[0]->task()) || !m_previewsAvailable)
    {
        QWidget::show();

        return;
    }

    m_alreadyShown = true;

    for (int i = 0; i < m_previews.count(); ++i)
    {
        m_previews.at(i)->show();
    }

    m_mainLayout->activate();

    adjustSize();

    Display *display = QX11Info::display();
    long unsigned int atom = XInternAtom(display, "_KDE_WINDOW_PREVIEW", false);
    QVarLengthArray<long, 1024> data(1 + (6 * m_previews.count()));
    data[0] = m_previews.count();

    for (int i = 0; i < m_previews.count(); ++i)
    {
        int index = (i * 6);

        qreal left, top, right, bottom;

        if (!m_previews.at(i)->background())
        {
            continue;
        }

        m_previews.at(i)->background()->getMargins(left, top, right, bottom);

        QPoint position = m_previews.at(i)->geometry().topLeft();
        position.setX(position.x() + ((m_previews.at(i)->size().width() - m_previews.at(i)->previewSize().width()) / 2));
        position.setY(position.y() + m_previews.at(i)->offset());

        QRect thumbnailRect = QRect(position, m_previews.at(i)->previewSize()).adjusted(left, top, -right, -bottom);

        data[index + 1] = 5;
        data[index + 2] = m_previews.at(i)->task()->windows().at(0);
        data[index + 3] = thumbnailRect.left();
        data[index + 4] = thumbnailRect.top();
        data[index + 5] = thumbnailRect.width();
        data[index + 6] = thumbnailRect.height();
    }

    XChangeProperty(display, winId(), atom, atom, 32, PropModeReplace, reinterpret_cast<unsigned char*>(data.data()), data.size());
#endif

    QWidget::show();
}

void ToolTip::hide(bool force)
{
    if (!underMouse() || force)
    {
        QWidget::hide();
    }
}

void ToolTip::setText(const QString &text)
{
    if (m_previews.count())
    {
        m_previews.at(0)->setText(text);

        m_alreadyShown = false;

        if (isVisible())
        {
            show();
        }
    }
}

void ToolTip::updateTheme()
{
    m_background->clearCache();

    m_mainLayout->setContentsMargins(m_background->marginSize(Plasma::LeftMargin), m_background->marginSize(Plasma::TopMargin), m_background->marginSize(Plasma::RightMargin), m_background->marginSize(Plasma::BottomMargin));

    QPalette plasmaPalette = QPalette();
    plasmaPalette.setColor(QPalette::Window, Plasma::Theme::defaultTheme()->color(Plasma::Theme::BackgroundColor));
    plasmaPalette.setColor(QPalette::WindowText, Plasma::Theme::defaultTheme()->color(Plasma::Theme::TextColor));

    setAutoFillBackground(true);

    setPalette(plasmaPalette);

    update();
}

}
