/********************************************************************************
 *   Copyright (C) 2008-2009 by Bram Schoenmakers <bramschoenmakers@kde.nl>     *
 *                                                                              *
 *   This program is free software; you can redistribute it and/or modify       *
 *   it under the terms of the GNU General Public License as published by       *
 *   the Free Software Foundation; either version 2 of the License, or          *
 *   (at your option) any later version.                                        *
 *                                                                              *
 *   This program is distributed in the hope that it will be useful,            *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the              *
 *   GNU General Public License for more details.                               *
 *                                                                              *
 *   You should have received a copy of the GNU General Public License          *
 *   along with this program; if not, write to the                              *
 *   Free Software Foundation, Inc.,                                            *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .             *
 ********************************************************************************/

#include <QAction>
#include <QCheckBox>
#include <QGridLayout>
#include <QGroupBox>
#include <QLabel>
#include <QPainter>
#include <QRadioButton>

#include <KColorButton>
#include <KComboBox>
#include <KConfigDialog>
#include <KFontRequester>
#include <KIntNumInput>
#include <KLineEdit>
#include <KMessageBox>
#include <KRun>
#include <Plasma/Theme>

#include "clusterdialog.h"
#include "photosetdialog.h"
#include "../engine/flickr_engine.h"

#include "flickrop.h"

using namespace Plasma;

// this Plasma:: should stay here, otherwise it breaks the applet!
Q_DECLARE_METATYPE( Plasma::DataEngine::Data )
 
FlickrOP::FlickrOP( QObject* p_parent, const QVariantList& p_args )
: Applet( p_parent, p_args )
, m_engine( 0 )
, m_authenticated( false )
, m_cluster_chosen( false )
, m_clusterdialog( 0 )
, m_photosetdialog( 0 )
, m_config_dialog( 0 )
, m_interval_combo( 0 )
, m_interval_value( 0 )
, m_size_combo( 0 )
, m_show_author_check( 0 )
, m_author_font_req( 0 )
, m_colorbutton( 0 )
{
  setBackgroundHints( Applet::NoBackground );
  setAspectRatioMode( KeepAspectRatio );
  setHasConfigurationInterface( true );
}

FlickrOP::~FlickrOP()
{
  KConfigGroup cfg = config();

  cfg.writeEntry( "token", m_config["token"] );

  delete m_clusterdialog;
  delete m_photosetdialog;
}

void FlickrOP::init()
{
  QAction *forceUpdate = new QAction( SmallIcon("system-run"), i18n("Ne&xt Photo"), this );
  m_actions.append( forceUpdate );
  connect( forceUpdate, SIGNAL( triggered( bool ) ), SLOT( forceUpdate() ) );

  m_markFavorite = new QAction( SmallIcon("emblem-favorite"), i18n("Mark As &Favorite"), this );
  m_markFavorite->setEnabled( false );
  m_actions.append( m_markFavorite );
  connect( m_markFavorite, SIGNAL( triggered( bool ) ), SLOT( markAsFavorite() ) );

  QAction *separator = new QAction( this );
  separator->setSeparator( true );
  m_actions.append( separator );

  QAction *visitAuthor = new QAction( i18n("Visit &Author's Profile"), this );
  m_actions.append( visitAuthor );
  connect( visitAuthor, SIGNAL( triggered( bool ) ), SLOT( visitAuthor() ) );

  QAction *visitAuthorPhotos = new QAction( i18n("Visit Aut&hor's Photos"), this );
  m_actions.append( visitAuthorPhotos );
  connect( visitAuthorPhotos, SIGNAL( triggered( bool ) ), SLOT( visitAuthorPhotos() ) );

  QAction *visitPhotoPage = new QAction( i18n("Visit &Photo Page"), this );
  m_actions.append( visitPhotoPage );
  connect( visitPhotoPage, SIGNAL( triggered( bool ) ), SLOT( visitPhoto() ) );

  separator = new QAction( this );
  separator->setSeparator( true );
  m_actions.append( separator );

  m_engine = dataEngine("flickrop");
  m_engine->connectAllSources( this );

  m_config["force_reload"] = true;
  m_config["check_token"] = true;
  readConfig(); // will also fire the engine
}

void FlickrOP::paintInterface ( QPainter *p_painter, const QStyleOptionGraphicsItem *p_option, const QRect &p_contentsRect )
{
  Q_UNUSED( p_option );

  if( !m_error.isEmpty() )
  {
    p_painter->drawText( p_contentsRect, Qt::AlignCenter, m_error );
  }
  else if( !m_photo.filename.isNull() )
  {
    // photo
    int y;
    int width;
    int height;
    if ( m_pixmap.width() > m_pixmap.height() )
    {
      width = qMin( p_contentsRect.width(), m_pixmap.width() );
      height = (int)( width * m_pixmap.height() / m_pixmap.width() );
      y = p_contentsRect.y() + (int)((p_contentsRect.height() / 2 ) - ( height / 2 ) );
    }
    else
    {
      height = qMin( p_contentsRect.height(), m_pixmap.height() );
      width = (int)( height * m_pixmap.width() / m_pixmap.height() );
      y = p_contentsRect.y();
      // if the author isn't shown, we can really center the image vertically
      if ( !m_show_author )
      {
	y += (int)((p_contentsRect.height() / 2 ) - ( height / 2 ) );
      }
    }
    int x = p_contentsRect.x() + (int)((p_contentsRect.width() / 2 ) - ( width / 2 ) );

    if ( m_show_author )
    {
      // text
      QString text;
      if ( m_photo.title.isEmpty() )
      {
	text = i18nc( "author", "By %1", m_photo.author.username );
      }
      else
      {
	text = i18nc( "title and author", "\"%1\" by %2", m_photo.title, m_photo.author.username );
      }
      p_painter->setPen( QPen( m_author_color ) );
      p_painter->setFont( m_author_font );

      QRect textArea( x, y + height, width, p_contentsRect.bottom() - y - height );
      QRect author_boundbox = p_painter->boundingRect( textArea, Qt::AlignRight | Qt::TextWordWrap, text );
      p_painter->drawText( author_boundbox, Qt::AlignRight | Qt::TextWordWrap, text );
    }

    p_painter->setRenderHint( QPainter::Antialiasing );
    p_painter->setRenderHint( QPainter::SmoothPixmapTransform );

    p_painter->drawPixmap( QRect( x, y, width, height ), m_pixmap );
  }
}

void FlickrOP::readConfig()
{
  KConfigGroup cfg = config();

  m_config["interval"] = cfg.readEntry( "interval", 3600 );

  QString token = cfg.readEntry( "token", "" );
  m_config["token"] = token;
  // don't check for token validity if there is none
  if ( token.isEmpty() )
  {
    m_config["check_token"] = false;
  }

  m_show_author = cfg.readEntry( "show_author", true );
  m_author_font = cfg.readEntry( "author_font", KGlobalSettings::generalFont() );
  QFontMetrics fm( m_author_font );
  m_font_height = fm.height();
  m_author_color = cfg.readEntry( "author_color", Theme::defaultTheme()->color( Theme::TextColor ) );
  QString size = cfg.readEntry( "size", "" );
  m_config["size"] = size;
  setSize( size );
  m_config["listtype"] = cfg.readEntry( "listtype", "interestingness" );

  // photosets
  m_config["photoset-id"] = cfg.readEntry( "photoset-id", "" );
  m_config["photoset-username"] = cfg.readEntry( "photoset-username", "" );
  m_config["photoset-nsid"] = cfg.readEntry( "photoset-nsid", "" );
  m_photoset_chosen = !m_config["photoset-id"].toString().isEmpty();

  // tags
  m_config["tag"] = cfg.readEntry( "tag", "" );
  m_config["cluster-id"] = cfg.readEntry( "cluster-id", "" );
  m_cluster_chosen = !m_config["cluster-id"].toString().isEmpty();

  sendConfigurationToEngine();
}

void FlickrOP::sendConfigurationToEngine()
{
  m_engine->setProperty( "config", QVariant::fromValue( m_config ) );
  m_config["force_reload"] = false;
  m_config["check_token"] = false;
}

void FlickrOP::writeConfig()
{
  KConfigGroup cfg = config();

  int index = m_interval_combo->currentIndex();
  int unit = m_interval_combo->itemData( index ).toInt();
  int interval = m_interval_value->value() * unit;

  cfg.writeEntry( "interval", interval );

  QString newsize = m_size_combo->itemData( m_size_combo->currentIndex() ).toString();
  cfg.writeEntry( "size", newsize );

  cfg.writeEntry( "show_author", m_show_author_check->isChecked() );
  cfg.writeEntry( "author_font", m_author_font_req->font() );
  cfg.writeEntry( "author_color", m_colorbutton->color() );

  QString listtype;
  QString clusterid;
  QString photosetid;
  if ( m_favorite_radio->isChecked() )
  {
    listtype = "favorites";
  }
  else if ( m_tag_radio->isChecked() && m_cluster_chosen )
  {
    listtype = "tag";
  }
  else if ( m_photoset_radio->isChecked() && m_photoset_chosen ) // the username is allowed to be empty
  {
    listtype = "photoset";
  }
  else
  {
    listtype = "interestingness";
  }

  cfg.writeEntry( "listtype", listtype );

  if ( m_photoset_chosen )
  {
    photosetid = m_photosetdialog ? m_photosetdialog->photoset() : m_config["photoset-id"].toString();
    cfg.writeEntry( "photoset-id", photosetid );
    cfg.writeEntry( "photoset-username", m_photoset_edit->text() );
    cfg.writeEntry( "photoset-nsid", m_nsid );
  }

  if ( m_cluster_chosen )
  {
    cfg.writeEntry( "tag", m_tag_edit->text() );
    clusterid = m_clusterdialog ? m_clusterdialog->cluster() : m_config["cluster-id"].toString();
    cfg.writeEntry( "cluster-id", clusterid );
  }

  cfg.writeEntry( "token", m_config["token"] );

  m_config["force_reload"] = ( m_config["listtype"] != listtype
			    || newsize != m_config["size"].toString()
			    || m_config["cluster-id"].toString() != clusterid
			    || m_config["photoset-id"].toString() != photosetid );
}

// http://www.flickr.com/services/api/misc.urls.html
// s	small square 75x75
// t	thumbnail, 100 on longest side
// m	small, 240 on longest side
// -	medium, 500 on longest side
// b	large, 1024 on longest side (only exists for very large original images)
// o	original image, either a jpg, gif or png, depending on source format
void FlickrOP::setSize( const QString &p_size )
{
  // some margin
  const int mw = 30;
  const int mh = 30 + ( m_show_author ? 2 * m_font_height : 0 );

  if ( p_size == "s" )
  {
    resize( 75 + mw, 75 + mh );
  }
  else if ( p_size == "t" )
  {
    resize( 100 + mw, 100 + mh );
  }
  else if ( p_size == "m" )
  {
    resize( 240 + mw, 240 + mh );
  }
  else if ( p_size.isEmpty() )
  {
    resize( 500 + mw, 500 + mh );
  }
  else if ( p_size == "b" )
  {
    resize( 1024 + mw, 1024 + mh );
  }
  // TODO: Original format
}

void FlickrOP::createConfigurationInterface( KConfigDialog *p_parent )
{
  // we need this in order to enable, disable the OK/Apply buttons from slots
  // in writeConfig and readConfig its already too late
  m_config_dialog = p_parent;

  p_parent->setButtons( KDialog::Ok | KDialog::Cancel | KDialog::Apply );
  connect( p_parent, SIGNAL( applyClicked() ), SLOT( writeConfig() ) );
  connect( p_parent, SIGNAL( applyClicked() ), SLOT( readConfig() ) );
  connect( p_parent, SIGNAL( okClicked() ), SLOT( writeConfig() ) );
  connect( p_parent, SIGNAL( okClicked() ), SLOT( readConfig() ) );


  p_parent->addPage( createGeneralPage(), i18n( "General" ), "configure" );
  p_parent->addPage( createStreamsPage(), i18n( "Streams" ), "camera-photo" );
  p_parent->addPage( createAuthenticationPage(), i18n( "Authentication" ), "preferences-desktop-user-password" );
}

QWidget *FlickrOP::createGeneralPage()
{
  QWidget *page = new QWidget();

  // command
  QGridLayout *layout = new QGridLayout( page );
  layout->setColumnStretch( 1, 1 );

  // interval
  QHBoxLayout *hbox = new QHBoxLayout();
  QLabel *interval_label = new QLabel( i18n( "&Refresh every:" ), page );
  m_interval_value = new KIntNumInput( page );
  m_interval_value->setMinimum( 0 );
  interval_label->setBuddy( m_interval_value );
  m_interval_combo = new KComboBox( page );

  const int second = 1;
  const int minute = 60;
  const int hour = 3600;
  const int day = 86400;

  m_interval_combo->insertItem( 0, i18n( "seconds"), second );
  m_interval_combo->insertItem( 1, i18n( "minutes"), minute );
  m_interval_combo->insertItem( 2, i18n( "hours" ), hour );
  m_interval_combo->insertItem( 3, i18n( "days" ), day );

  int interval = m_config.value("interval").toInt();
  if ( interval > 0 && interval % day == 0 )
  {
    m_interval_value->setValue( interval / day );
    m_interval_combo->setCurrentIndex( 3 );
  }
  else if ( interval > 0 && interval % hour == 0 )
  {
    m_interval_value->setValue( interval / hour );
    m_interval_combo->setCurrentIndex( 2 );
  }
  else if ( interval > 0 && interval % minute == 0 )
  {
    m_interval_value->setValue( interval / minute );
    m_interval_combo->setCurrentIndex( 1 );
  }
  else
  {
    m_interval_value->setValue( interval );
    m_interval_combo->setCurrentIndex( 0 );
  }

  hbox->addWidget( m_interval_value, 1 );
  hbox->addWidget( m_interval_combo );

  layout->addWidget( interval_label, 1, 0 );
  layout->addLayout( hbox, 1, 1 );

  QLabel *size_label = new QLabel( i18n("Photo Size:"), page );

  m_size_combo = new KComboBox();
  size_label->setBuddy( m_size_combo );

  // mapping sizes to an index
  QMap<QString,int> map;
  map["s"] = 0;
  map["t"] = 1;
  map["m"] = 2;
  map[""] = 3;
  map["b"] = 4;

  m_size_combo->insertItem( 0, i18n("Tiny"), "s" );
  m_size_combo->insertItem( 1, i18n("Thumbnail"), "t" );
  m_size_combo->insertItem( 2, i18n("Small"), "m" );
  m_size_combo->insertItem( 3, i18n("Medium"), "" );
  m_size_combo->insertItem( 4, i18n("Large"), "b" );
  // m_size_combo->insertItem( 5, i18n("Original"), "o" ); TODO: Maybe later, involves something more

  m_size_combo->setCurrentIndex( map[ m_config["size"].toString() ] );

  layout->addWidget( size_label, 2, 0 );
  layout->addWidget( m_size_combo, 2, 1 );

  m_show_author_check = new QCheckBox( i18n( "Show title and author" ), page );
  layout->addWidget( m_show_author_check, 3, 1 );

  QLabel *font_label = new QLabel( i18n("Font:"), page );

  m_author_font_req = new KFontRequester( page );
  m_author_font_req->setFont( m_author_font );
  font_label->setBuddy( m_author_font_req );

  layout->addWidget( font_label, 4, 0 );
  layout->addWidget( m_author_font_req, 4, 1 );

  QLabel *font_color_label = new QLabel( i18n( "&Color:" ), page );
  
  m_colorbutton = new KColorButton( page );
  m_colorbutton->setColor( m_author_color );
  font_color_label->setBuddy( m_colorbutton );

  layout->addWidget( font_color_label, 5, 0 );
  layout->addWidget( m_colorbutton, 5, 1 );

  m_show_author_check->setChecked( m_show_author );
  font_label->setEnabled( m_show_author );
  m_author_font_req->setEnabled( m_show_author );
  font_color_label->setEnabled( m_show_author );
  m_colorbutton->setEnabled( m_show_author );
  connect( m_show_author_check, SIGNAL( toggled( bool ) ), font_label, SLOT( setEnabled( bool ) ) );
  connect( m_show_author_check, SIGNAL( toggled( bool ) ), m_author_font_req, SLOT( setEnabled( bool ) ) );
  connect( m_show_author_check, SIGNAL( toggled( bool ) ), font_color_label, SLOT( setEnabled( bool ) ) );
  connect( m_show_author_check, SIGNAL( toggled( bool ) ), m_colorbutton, SLOT( setEnabled( bool ) ) );

  layout->addItem( new QSpacerItem( 0, 0 ), 3, 0 );
  layout->setRowStretch( 3, 1 );

  return page;
}

QWidget *FlickrOP::createStreamsPage()
{
  QWidget *page = new QWidget();
  QVBoxLayout *pageLayout = new QVBoxLayout( page );

  QGroupBox *groupBox = new QGroupBox( i18n("Which photos to show?" ), page );

  m_interesting_radio = new QRadioButton( i18n("&Interesting photos") );
  m_favorite_radio = new QRadioButton( i18n("My &favorite photos") );
  m_photoset_radio = new QRadioButton( i18n("Photo&set") );
  m_tag_radio = new QRadioButton( i18n( "By &tag" ) );

  connect( m_interesting_radio, SIGNAL( toggled( bool ) ), this, SLOT( radioChanged() ) );
  connect( m_favorite_radio, SIGNAL( toggled( bool ) ), this, SLOT( radioChanged() ) );
  connect( m_photoset_radio, SIGNAL( toggled( bool ) ), this, SLOT( radioChanged() ) );
  connect( m_tag_radio, SIGNAL( toggled( bool ) ), this, SLOT( radioChanged() ) );

  // photoset widgets
  m_photoset_edit = new KLineEdit();
  m_get_photoset_button = new QPushButton( i18n( "Get photosets" ) );
  m_get_photoset_button->setEnabled( m_photoset_chosen );
  m_photoset_edit->setEnabled( false );
  m_photoset_edit->setEnabled( false );
  m_photoset_edit->setText( m_config["photoset-username"].toString() );
  connect( m_get_photoset_button, SIGNAL( clicked( bool ) ), SLOT( getPhotosets() ) );
  connect( m_photoset_edit, SIGNAL( textChanged( const QString & ) ), SLOT( photosetUserChanged( const QString & ) ) );
  connect( m_photoset_edit, SIGNAL( textEdited( const QString & ) ), SLOT( photosetUserEdited( const QString & ) ) );

  // tag widgets
  m_get_cluster_button = new QPushButton( i18n( "Get clusters" ) );
  connect( m_get_cluster_button, SIGNAL( clicked( bool ) ), SLOT( getClusters() ) );
  m_get_cluster_button->setEnabled( m_cluster_chosen );
  m_tag_edit = new KLineEdit();
  connect( m_tag_edit, SIGNAL( textChanged( const QString & ) ), SLOT( tagTextChanged( const QString & ) ) );
  connect( m_tag_edit, SIGNAL( textEdited( const QString & ) ), SLOT( tagTextEdited( const QString & ) ) );
  m_tag_edit->setEnabled( false );

  QString type = m_config["listtype"].toString();
  m_interesting_radio->setChecked( type == "interestingness" );
  m_favorite_radio->setChecked( type == "favorites" );
  m_photoset_radio->setChecked( type == "photoset" );
  m_photoset_edit->setEnabled( type == "photoset" );
  m_tag_radio->setChecked( type == "tag" );
  m_tag_edit->setEnabled( type == "tag" );
  m_tag_edit->setText( m_config["tag"].toString() );

  QGridLayout *grid = new QGridLayout();
  grid->addWidget( m_interesting_radio, 0, 0 );
  grid->addWidget( m_favorite_radio, 1, 0 );
  grid->addWidget( m_photoset_radio, 2, 0 );
  grid->addWidget( m_photoset_edit, 3, 0 );
  grid->addWidget( m_get_photoset_button, 3, 1 );
  grid->addWidget( m_tag_radio, 4, 0 );
  grid->addWidget( m_tag_edit, 5, 0 );
  grid->addWidget( m_get_cluster_button, 5, 1 );
  groupBox->setLayout( grid );
  pageLayout->addWidget( groupBox );

  return page;
}

QWidget *FlickrOP::createAuthenticationPage()
{
  QWidget *page = new QWidget();

  QVBoxLayout *vlayout = new QVBoxLayout( page );

  QString text;
  if ( m_authenticated )
  {
    text = i18n( "You are already authenticated." );
  }
  else
  {
    text = i18n( "In order to use all functions in Flickr On Plasma, you have to authenticate with Flickr. Press the button below to start the authentication." );
  }
  QLabel *auth_guide = new QLabel( text );
  auth_guide->setWordWrap( true );
  auth_guide->setAlignment( Qt::AlignHCenter );
  QPushButton *auth_button = new QPushButton( i18n( "Authenticate" ), page );
  auth_button->setEnabled( !m_authenticated );
  connect( auth_button, SIGNAL( clicked( bool ) ), SLOT( authenticate() ) );

  vlayout->addWidget( auth_guide );
  vlayout->addWidget( auth_button );

  return page;
}

QList<QAction *> FlickrOP::contextualActions()
{
  return m_actions;
}

void FlickrOP::dataUpdated( const QString &p_sourceName, const DataEngine::Data &p_data )
{
  if ( p_sourceName == "error" )
  {
    m_error.clear();
    int type = p_data.value( "type" ).toInt();
    QString message = p_data.value( "message" ).toString();
    if ( type == ERR_APPLET )
    {
      // show on applet
      m_error = message;
      update();
    }
    else if ( type == ERR_MESSAGEBOX )
    {
      // delay it, such that we leave this function before it is called again
      QMetaObject::invokeMethod( this, "showErrorBox", Qt::QueuedConnection, Q_ARG( QString, message ) );
    }
    else if ( !message.isEmpty() )
    {
      kDebug() << message << endl;
    }
  }
  else if ( p_sourceName == "authentication" )
  {
    m_authenticated = p_data.value( "token_status" ).toBool();
    m_config["token"] = p_data.value( "token" ).toString();

    m_markFavorite->setEnabled( m_authenticated );
  }
  else if ( p_sourceName == "clusters" )
  {
    if ( m_clusterdialog )
    {
      m_clusterdialog->reset();
    }
    foreach( const QVariant &variant, p_data )
    {
      // filter out noise
      if ( variant.type() != QVariant::StringList )
      {
	continue;
      }
      if ( !m_clusterdialog )
      {
	m_clusterdialog = new ClusterDialog();
	connect( m_clusterdialog, SIGNAL( accepted() ), SLOT( clusterChosen() ) );
      }
      m_clusterdialog->addCluster( variant.toStringList() );
    }

    QTimer::singleShot( 0, this, SLOT( showClusterDialog() ) );
  }
  else if ( p_sourceName == "nsid" && !p_data["nsid"].toString().isEmpty() )
  {
    m_nsid = p_data["nsid"].toString();
    QMetaObject::invokeMethod( m_engine, "retrievePhotosets", Qt::DirectConnection, Q_ARG( QString, m_nsid ) );
  }
  else if ( p_sourceName == "photosets" )
  {
    if ( !p_data[ p_sourceName ].canConvert( QVariant::String ) )
    {
      if ( !m_photosetdialog )
      {
	m_photosetdialog = new PhotosetDialog();
	connect( m_photosetdialog, SIGNAL( accepted() ), SLOT( photosetChosen() ) );
      }
      m_photosetdialog->reset();
      m_photosetdialog->addPhotosets( p_data );
      QTimer::singleShot( 0, this, SLOT( showPhotosetDialog() ) );
    }
  }
  else // flickr stream
  {
    m_error.clear();
    m_photo = p_data.value( "flickr" ).value<Photo>();
    m_pixmap.load( m_photo.filename, "JPG" );
    update();
  }
}

void FlickrOP::showErrorBox( const QString &p_message )
{
  KMessageBox::error( 0, p_message );
}

void FlickrOP::forceUpdate()
{
  QMetaObject::invokeMethod( m_engine, "nextPhoto", Qt::DirectConnection );
}

void FlickrOP::visitAuthor()
{
  KRun::runUrl( m_photo.author.profileUrl, "text/html", 0 );
}

void FlickrOP::visitAuthorPhotos()
{
  KRun::runUrl( m_photo.author.photosUrl, "text/html", 0 );
}

void FlickrOP::visitPhoto()
{
  QString nsid = m_photo.author.nsid;
  QString photoid = m_photo.id;
  QString url = QString( "http://www.flickr.com/photos/%1/%2" ).arg( nsid ).arg( photoid );
  KRun::runUrl( KUrl( url ), "text/html", 0 );
}

void FlickrOP::authenticate()
{
  QMetaObject::invokeMethod( m_engine, "startAuthentication", Qt::QueuedConnection );
  if ( KMessageBox::warningContinueCancel( 0, i18n( "A browser will open now with the request to authenticate Flickr On Plasma. When you accepted this request, press Continue." ) ) == KMessageBox::Continue )
  {
    QMetaObject::invokeMethod( m_engine, "continueAuthentication", Qt::DirectConnection );
  }
}

void FlickrOP::markAsFavorite()
{
  QMetaObject::invokeMethod( m_engine, "markAsFavorite", Qt::DirectConnection );
}

void FlickrOP::getClusters()
{
  QMetaObject::invokeMethod( m_engine, "retrieveClusters", Qt::DirectConnection, Q_ARG( QString, m_tag_edit->text() ) );
}

void FlickrOP::showClusterDialog()
{
  if ( m_clusterdialog )
  {
    m_clusterdialog->exec();
  }
}

void FlickrOP::showPhotosetDialog()
{
  if ( m_photosetdialog )
  {
    m_photosetdialog->exec();
  }
}

void FlickrOP::tagTextChanged( const QString &p_tag )
{
  m_get_cluster_button->setEnabled( !p_tag.isEmpty() );
}

void FlickrOP::tagTextEdited( const QString &p_tag )
{
  Q_UNUSED( p_tag );
  m_cluster_chosen = false;
  setOkButton();
}

void FlickrOP::radioChanged()
{
  bool photosetsSelected = m_photoset_radio->isChecked();
  m_photoset_edit->setEnabled( photosetsSelected );
  m_get_photoset_button->setEnabled( photosetsSelected );
  bool tagsSelected = m_tag_radio->isChecked();
  m_tag_edit->setEnabled( tagsSelected );
  m_get_cluster_button->setEnabled( tagsSelected && !m_tag_edit->text().isEmpty() );
  setOkButton();
}

void FlickrOP::setOkButton()
{
  if ( m_config_dialog )
  {
    m_config_dialog->enableButtonOk( ( !m_tag_radio->isChecked() || m_cluster_chosen ) && ( !m_photoset_radio->isChecked() || m_photoset_chosen ) );
  }
}

void FlickrOP::clusterChosen()
{
  m_cluster_chosen = true;
  setOkButton();
}

void FlickrOP::photosetUserChanged( const QString &p_username )
{
  m_get_photoset_button->setEnabled( !p_username.isEmpty() );
}

void FlickrOP::photosetUserEdited( const QString &p_username )
{
  Q_UNUSED( p_username );
  m_photoset_chosen = false;
  setOkButton();
}

void FlickrOP::photosetChosen()
{
  m_photoset_chosen = true;
  setOkButton();
}

void FlickrOP::getPhotosets()
{
  QMetaObject::invokeMethod( m_engine, "retrieveNSID", Qt::DirectConnection, Q_ARG( QString, m_photoset_edit->text() ) );
}

K_EXPORT_PLASMA_APPLET(flickrop, FlickrOP)
