/*/***************************************************************************
 *   Copyright (C) 2009 by Björn Ruberg <bjoern@ruberg-wegener.de>    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .        *
 ***************************************************************************/


#include "widget.h"
#include "BoardKey.h"
#include "AlphaNumKey.h"
#include "FuncKey.h"
#include "EnterKey.h"
#include "BackspaceKey.h"
#include "TabKey.h"
#include "ShiftKey.h"
#include "CapsKey.h"
#include "ArrowTopKey.h"
#include "ArrowBottomKey.h"
#include "ArrowLeftKey.h"
#include "ArrowRightKey.h"
#include <QPainter>
#include <QGraphicsGridLayout>
#include <QLabel>
#include <KConfigGroup>


#define BACKSPACEKEY 0
#define TABKEY 1
#define ENTERKEY 2
#define CAPSKEY 3
#define SHIFT_L_KEY 4
#define SHIFT_R_KEY 5
#define CTLKEY 6
#define META_L_KEY 7
#define ALT_L_KEY 8
#define SPACE 9
#define ALTGRKEY 10
#define META_R_KEY 11

#define MENU 12
#define CONTROL_LEFT 13

#define HOMEKEY 16
#define ENDKEY 17
#define INSKEY 18
#define DELKEY 19
#define PGUPKEY 20
#define PGDOWNKEY 21

#define ARROWTOPKEY 	22
#define ARROWLEFTKEY	23
#define ARROWBOTTOMKEY	24
#define ARROWRIGHTKEY	25

PlasmaboardWidget::PlasmaboardWidget(QGraphicsWidget *parent)
    : Plasma::Containment(parent)
{
    setPreferredSize(500, 200);
    setMinimumSize(300,150);
    setWindowFlags(Qt::FramelessWindowHint | Qt::WindowStaysOnTopHint | Qt::X11BypassWindowManagerHint);

    isLevel2 = false;
    isAlternative = false;
    isLocked = false;
    m_layout = new QGraphicsGridLayout(this);
    extendedKeys = false;
    basicKeys = false;

    tooltipTimer = new QTimer(this);
    connect(tooltipTimer, SIGNAL(timeout()), this, SLOT(clearTooltip()));
    tooltip = new Tooltip("");
}


PlasmaboardWidget::~PlasmaboardWidget(){
    delete tooltipTimer;
}

void PlasmaboardWidget::resetKeyboard(){

	int i = 0;

	if( basicKeys ){
		i = m_layout->count() - 1;
		while(i >= 0){
			m_layout->removeAt(i--);
		}

                qDeleteAll(funcKeys);
                funcKeys.clear();

                qDeleteAll(alphaKeys);
                alphaKeys.clear();

		basicKeys = false;
	}

	i = 0;
	if( extendedKeys ) {
                qDeleteAll(extKeys);
                extKeys.clear();
		extendedKeys = false;
	}
}

void PlasmaboardWidget::initExtendedKeyboard(){

	extendedKeys = true;

	int i = 0;
	while ( i < 42 ) {
                extKeys << new FuncKey(this);
		QObject::connect(extKeys[i], SIGNAL( clicked() ), this, SLOT( clear() ) );
		i++;
	}

	delete extKeys[ARROWTOPKEY];
	delete extKeys[ARROWLEFTKEY];
	delete extKeys[ARROWBOTTOMKEY];
	delete extKeys[ARROWRIGHTKEY];
	extKeys[ARROWTOPKEY] = new ArrowTopKey(this);
	extKeys[ARROWLEFTKEY] = new ArrowLeftKey(this);
	extKeys[ARROWBOTTOMKEY] = new ArrowBottomKey(this);
	extKeys[ARROWRIGHTKEY] = new ArrowRightKey(this);

	m_layout->addItem(extKeys[0], 0, 0, 1, 2, Qt::AlignLeft);
	i = 1;
	for(i = 1; i < 5; i++){
		m_layout->addItem(extKeys[i], 0, (i*2)+1, 1, 2, Qt::AlignLeft);
	}
	for(i = 5; i < 9; i++){
		m_layout->addItem(extKeys[i], 0, (i*2)+2, 1, 2, Qt::AlignLeft);
	}
	for(i = 9; i < 13; i++){
		m_layout->addItem(extKeys[i], 0, (i*2)+3, 1, 2, Qt::AlignLeft);
	}

	for(i = 13; i < 16; i++){
		m_layout->addItem(extKeys[i], 0, (i*2)+4, 1, 2, Qt::AlignLeft);
	}

        extKeys[0]->setKey(XK_Escape, true, i18n("Esc"));
        extKeys[1]->setKey(XK_F1, true, i18n("F1"));
        extKeys[2]->setKey(XK_F2, true, i18n("F2"));
        extKeys[3]->setKey(XK_F3, true, i18n("F3"));
        extKeys[4]->setKey(XK_F4, true, i18n("F4"));
        extKeys[5]->setKey(XK_F5, true, i18n("F5"));
        extKeys[6]->setKey(XK_F6, true, i18n("F6"));
        extKeys[7]->setKey(XK_F7, true, i18n("F7"));
        extKeys[8]->setKey(XK_F8, true, i18n("F8"));
        extKeys[9]->setKey(XK_F9, true, i18n("F9"));
        extKeys[10]->setKey(XK_F10, true, i18n("F10"));
        extKeys[11]->setKey(XK_F11, true, i18n("F11"));
        extKeys[12]->setKey(XK_F12, true, i18n("F12"));
        extKeys[13]->setKey(XK_Print, true, i18n("Print"));
        extKeys[14]->setKey(XK_Num_Lock, true, i18n("Num"));
        extKeys[15]->setKey(XK_Pause, true, i18n("Pause"));

	initBasicKeyboard(1);

	m_layout->addItem(extKeys[ARROWLEFTKEY], 5, 29, 1, 2, Qt::AlignLeft);

	int col = 31;

	m_layout->addItem(extKeys[HOMEKEY], 1, col, 1, 2, Qt::AlignLeft);
        extKeys[HOMEKEY]->setKey(XK_Home, true, i18n("Home"));

	m_layout->addItem(extKeys[ENDKEY], 2, col, 1, 2, Qt::AlignLeft);
        extKeys[ENDKEY]->setKey(XK_End, true, i18n("End"));

	m_layout->addItem(extKeys[INSKEY], 3, col, 1, 2, Qt::AlignLeft);
        extKeys[INSKEY]->setKey(XK_Insert, true, i18n("Ins"));

	m_layout->addItem(extKeys[ARROWTOPKEY], 4, col, 1, 2, Qt::AlignLeft);
	m_layout->addItem(extKeys[ARROWBOTTOMKEY], 5, col, 1, 2, Qt::AlignLeft);

	col+=2;

	m_layout->addItem(extKeys[ARROWRIGHTKEY], 5, col, 1, 2, Qt::AlignLeft);
	m_layout->addItem(extKeys[DELKEY], 1, col, 1, 2, Qt::AlignLeft);
        extKeys[DELKEY]->setKey(XK_Delete, true, i18n("Del"));
	m_layout->addItem(extKeys[PGUPKEY], 2, col, 1, 2, Qt::AlignLeft);
        extKeys[PGUPKEY]->setKey(XK_Page_Up, true, i18n("PgUp"));
	m_layout->addItem(extKeys[PGDOWNKEY], 3, col, 1, 2, Qt::AlignLeft);
        extKeys[PGDOWNKEY]->setKey(XK_Page_Down, true, i18n("PgDn"));

	col+=3;

	m_layout->addItem(extKeys[26], 2, col, 1, 2);
        extKeys[26]->setKey(XK_KP_7, true, i18n("7"));
	m_layout->addItem(extKeys[27], 3, col, 1, 2);
        extKeys[27]->setKey(XK_KP_4, true, i18n("4"));
	m_layout->addItem(extKeys[28], 4, col, 1, 2);
        extKeys[28]->setKey(XK_KP_1, true, i18n("1"));
	m_layout->addItem(extKeys[29], 5, col, 1, 4);
        extKeys[29]->setKey(XK_KP_0, true, i18n("0"));

	col+=2;

	m_layout->addItem(extKeys[30], 1, col, 1, 2);
        extKeys[30]->setKey(XK_KP_Divide, true, i18n("/"));
	m_layout->addItem(extKeys[31], 2, col, 1, 2);
        extKeys[31]->setKey(XK_KP_8, true, i18n("8"));
	m_layout->addItem(extKeys[32], 3, col, 1, 2);
        extKeys[32]->setKey(XK_KP_5, true, i18n("5"));
	m_layout->addItem(extKeys[33], 4, col, 1, 2);
        extKeys[33]->setKey(XK_KP_2, true, i18n("2"));

	col+=2;

	m_layout->addItem(extKeys[34], 1, col, 1, 2);
        extKeys[34]->setKey(XK_KP_Multiply, true, i18n("*"));
	m_layout->addItem(extKeys[35], 2, col, 1, 2);
        extKeys[35]->setKey(XK_KP_9, true, i18n("9"));
	m_layout->addItem(extKeys[36], 3, col, 1, 2);
        extKeys[36]->setKey(XK_KP_6, true, i18n("6"));
	m_layout->addItem(extKeys[37], 4, col, 1, 2);
        extKeys[37]->setKey(XK_KP_3, true, i18n("3"));
	m_layout->addItem(extKeys[38], 5, col, 1, 2);
        extKeys[38]->setKey(XK_KP_Separator, true, i18n("."));

	col+=2;

	m_layout->addItem(extKeys[39], 1, col, 1, 2);
        extKeys[39]->setKey(XK_KP_Subtract, true, i18n("-"));
	m_layout->addItem(extKeys[40], 2, col, 2, 2);
        extKeys[40]->setKey(XK_KP_Add, true, i18n("+"));
	m_layout->addItem(extKeys[41], 4, col, 2, 2);
        extKeys[41]->setKey(XK_KP_Enter, true, i18n("Enter"));

	for(int i = 30; i < 44; i++){
                m_layout->setColumnPreferredWidth(i,size().width()/30);
	}
}

void PlasmaboardWidget::initBasicKeyboard(int offset){

	basicKeys = true;

	// create objects

        funcKeys.append( new BackspaceKey(this) );
        funcKeys << new TabKey(this);
        funcKeys << new EnterKey(this);
        funcKeys << new CapsKey(this);
        funcKeys << new ShiftKey(this);
        funcKeys << new ShiftKey(this);

        int i = 6;
        while ( i < 14 ) {
                funcKeys << new FuncKey(this);
                i++;
        }

        funcKeys[CTLKEY]->setKey(XK_Control_L, false, i18n("Ctrl"));
        funcKeys[META_L_KEY]->setKey(XK_Meta_L, false, i18n("Meta"));
        funcKeys[ALT_L_KEY]->setKey(XK_Alt_L, false, i18n("Alt"));
        funcKeys[SPACE]->setKeycode(XK_space, true);
        funcKeys[ALTGRKEY]->setKey(XK_ISO_Level3_Shift, false, i18n("Alt Gr"));
        funcKeys[META_R_KEY]->setKey(XK_Meta_L, false, i18n("Meta"));
        funcKeys[MENU]->setKey(XK_Menu, true, i18n("Menu"));
        funcKeys[CONTROL_LEFT]->setKey(XK_Control_L, false, i18n("Ctrl"));

        // set Keymap

        alphaKeys << new AlphaNumKey(this, 49);

        for ( i = 10; i < 22; i++ ){
               alphaKeys << new AlphaNumKey(this, i);
        }

        for ( i = 24; i < 36; i++ ){
               alphaKeys << new AlphaNumKey(this, i);
        }

        for ( i = 38; i < 49; i++ ){
               alphaKeys << new AlphaNumKey(this, i);
        }

        alphaKeys << new AlphaNumKey(this, 51);
        alphaKeys << new AlphaNumKey(this, 94);

        for ( i = 52; i < 62; i++ ){
               alphaKeys << new AlphaNumKey(this, i);
        }


	// create layout

	int numberOfCols = 30;

	int x;
	int row = offset;
	for(x = 0; x<=12; x++){
		m_layout->addItem(alphaKeys[x], row, x*2, 1, 2, Qt::AlignLeft);
	}
	m_layout->addItem(funcKeys[BACKSPACEKEY], row, x*2, 1, 4, Qt::AlignLeft);

	row++;

	m_layout->addItem(funcKeys[TABKEY], row, 0, 1, 3, Qt::AlignLeft);
	int t = 3;
	for(x = 13; x<=24; x++){
		m_layout->addItem(alphaKeys[x], row, t, 1, 2, Qt::AlignLeft);
		t+=2;
	}
	m_layout->addItem(funcKeys[ENTERKEY], row, t+1, 2, 2, Qt::AlignLeft);

	row++;

	m_layout->addItem(funcKeys[CAPSKEY], row, 0, 1, 4, Qt::AlignLeft);
	t = 4;
	for(x = 25; x<=36; x++){
		m_layout->addItem(alphaKeys[x], row, t, 1, 2, Qt::AlignLeft);
		t+=2;
	}

	row++;

	m_layout->addItem(funcKeys[SHIFT_L_KEY], row, 0, 1, 3, Qt::AlignLeft);
	t = 3;
	for(x = 37; x<=47; x++){
		m_layout->addItem(alphaKeys[x], row, t, 1, 2, Qt::AlignLeft);
		t+=2;
	}
	m_layout->addItem(funcKeys[SHIFT_R_KEY], row, t, 1, 5, Qt::AlignLeft);

	row++;

	m_layout->addItem(funcKeys[CTLKEY], row, 0, 1, 2, Qt::AlignLeft);
	m_layout->addItem(funcKeys[META_L_KEY], row, 2, 1, 2, Qt::AlignLeft);
	m_layout->addItem(funcKeys[ALT_L_KEY], row, 4, 1, 2, Qt::AlignLeft);
        m_layout->addItem(funcKeys[SPACE], row, 6, 1, 14, Qt::AlignLeft);
	m_layout->addItem(funcKeys[ALTGRKEY], row, 20, 1, 2, Qt::AlignLeft);
	m_layout->addItem(funcKeys[META_R_KEY], row, 22, 1, 2, Qt::AlignLeft);
        m_layout->addItem(funcKeys[MENU], row, 24, 1, 2, Qt::AlignLeft);
        m_layout->addItem(funcKeys[CONTROL_LEFT], row, 26, 1, 2, Qt::AlignLeft);

	for(int i = 0; i<numberOfCols; i++){
                m_layout->setColumnPreferredWidth(i,size().width()/numberOfCols);
	}

	for(int i = 30; i < 44; i++){
		m_layout->setColumnPreferredWidth(i, 0);
	}

	m_layout->setSpacing(0);
	m_layout->setContentsMargins(0,0,0,0);


	QObject::connect(funcKeys[SHIFT_L_KEY], SIGNAL( clicked() ), this, SLOT( switchLevel() ) );
	QObject::connect(funcKeys[SHIFT_R_KEY], SIGNAL( clicked() ), this, SLOT( switchLevel() ) );
	QObject::connect(funcKeys[CAPSKEY], SIGNAL( clicked() ), this, SLOT( switchCaps() ) );
	QObject::connect(funcKeys[ALTGRKEY], SIGNAL( clicked() ), this, SLOT( switchAlternative() ) );

}

void PlasmaboardWidget::paint(QPainter *p,
                const QStyleOptionGraphicsItem *option,
                 QWidget*)
{
    p->setRenderHint(QPainter::SmoothPixmapTransform);
    p->setRenderHint(QPainter::Antialiasing);
}

void PlasmaboardWidget::switchLevel(){
	if(!isLevel2){
		isLevel2 = true;
	}
	else {
		isLevel2 = false;
		Helpers::fakeKeyRelease(Helpers::keysymToKeycode(XK_Shift_L));
		funcKeys[SHIFT_L_KEY]->toggleOff();
		funcKeys[SHIFT_R_KEY]->toggleOff();
	}
	foreach (AlphaNumKey* key, alphaKeys){
	    key->switchKey(isLevel2, isAlternative, isLocked);
	}
}

void PlasmaboardWidget::switchAlternative(){
	if(!isAlternative){
		isAlternative = true;
	}
	else {
		isAlternative = false;
		Helpers::fakeKeyRelease(Helpers::keysymToKeycode(XK_ISO_Level3_Shift));
		funcKeys[ALTGRKEY]->toggleOff();
	}
        foreach (AlphaNumKey* key, alphaKeys){
	    key->switchKey(isLevel2, isAlternative, isLocked);
	}
}

void PlasmaboardWidget::switchCaps(){
	if(isLocked){
		isLocked = false;
	}
	else {
		isLocked = true;
	}
        foreach (AlphaNumKey* key, alphaKeys){
            key->switchKey(isLevel2, isAlternative, isLocked);
	}
}

void PlasmaboardWidget::clearAnything(){
	if ( isLocked ) {
		switchCaps();
		funcKeys[CAPSKEY]->sendKeycodePress();
		funcKeys[CAPSKEY]->sendKeycodeRelease();
	}
	clear();
}

void PlasmaboardWidget::clear(){

	bool change = false;
	if(isLevel2){
		Helpers::fakeKeyRelease(Helpers::keysymToKeycode(XK_Shift_L));
		funcKeys[SHIFT_L_KEY]->toggleOff();
		funcKeys[SHIFT_R_KEY]->toggleOff();
		isLevel2 = false;
		change = true;
	}
	if(isAlternative){
		Helpers::fakeKeyRelease(Helpers::keysymToKeycode(XK_ISO_Level3_Shift));
		funcKeys[ALTGRKEY]->toggleOff();
		isAlternative = false;
		change = true;
	}

	if(change){

                foreach (AlphaNumKey* key, alphaKeys){
                    key->switchKey(isLevel2, isAlternative, isLocked);
                }
	}

	Helpers::fakeKeyRelease(Helpers::keysymToKeycode(XK_Control_L));
	funcKeys[CTLKEY]->toggleOff();
        funcKeys[CONTROL_LEFT]->toggleOff();
	Helpers::fakeKeyRelease(Helpers::keysymToKeycode(XK_Meta_L));
	funcKeys[META_L_KEY]->toggleOff();
	funcKeys[META_R_KEY]->toggleOff();
	Helpers::fakeKeyRelease(Helpers::keysymToKeycode(XK_Meta_L));
	funcKeys[ALT_L_KEY]->toggleOff();
}


void PlasmaboardWidget::setTooltip(QString text, QSizeF buttonSize, QPointF position){
    tooltip -> setText( text );
    tooltip -> move( popupPosition(buttonSize.toSize()) + QPoint(position.x() - buttonSize.width() / 2 , position.y() - buttonSize.height()) );
    tooltip -> resize( (buttonSize * 2).toSize() );
    tooltip -> show();

    tooltipTimer->start(300);
}

void PlasmaboardWidget::clearTooltip(){

    tooltip->hide();
}
