/***********************************************************************************
* Spell Check: Plasmoid for fast spell checking.
* Copyright (C) 2008 - 2009 Michal Dutkiewicz aka Emdek <emdeck@gmail.com>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*
***********************************************************************************/

#include "SpellCheck.h"

#include <QFile>
#include <QClipboard>
#include <QVBoxLayout>
#include <QHBoxLayout>
#include <QTextStream>
#include <QApplication>
#include <QGraphicsLinearLayout>

#include <Plasma/IconWidget>
#include <Plasma/ToolTipManager>

#include <KIcon>
#include <KLocale>
#include <KIconLoader>

K_EXPORT_PLASMA_APPLET(spellcheck, SpellCheck)

SpellCheck::SpellCheck(QObject *parent, const QVariantList &args) : Plasma::Applet(parent, args)
{
    KGlobal::locale()->insertCatalog("spellcheck");

    setAspectRatioMode(Plasma::ConstrainedSquare);

    setAcceptDrops(true);

    int iconSize = IconSize(KIconLoader::Desktop);

    resize((iconSize * 2), (iconSize * 2));
}

void SpellCheck::init()
{
    KConfigGroup configuration = config();

    m_dragTimer = new QTimer(this);
    m_dragTimer->setInterval(500);
    m_dragTimer->setSingleShot(true);

    Plasma::IconWidget *icon = new Plasma::IconWidget(KIcon("tools-check-spelling"), QString(), this);

    registerAsDragHandle(icon);

    Plasma::ToolTipManager::self()->setContent(this, Plasma::ToolTipContent(i18n("Spell checking"), i18n("Check spelling of clipboard contents."), icon->icon().pixmap(IconSize(KIconLoader::Desktop))));

    QGraphicsLinearLayout *layout = new QGraphicsLinearLayout(this);
    layout->setContentsMargins(0, 0, 0, 0);
    layout->setSpacing(0);
    layout->addItem(icon);

    m_dialog = new Plasma::Dialog(NULL, Qt::Tool);
    m_dialog->setFocusPolicy(Qt::NoFocus);
    m_dialog->setWindowTitle(i18n("Spell checking"));
    m_dialog->setResizeHandleCorners(Plasma::Dialog::All);
    m_dialog->resize(configuration.readEntry("dialogSize", m_dialog->size()));

    m_textEdit = new KTextEdit(m_dialog);
    m_textEdit->enableFindReplace(false);
    m_textEdit->setCheckSpellingEnabled(true);
    m_textEdit->createHighlighter();

    m_dictionaryComboBox = new Sonnet::DictionaryComboBox(m_dialog);
    m_dictionaryComboBox->setToolTip(i18n("Language"));

    m_copyButton = new QToolButton(m_dialog);
    m_copyButton->setIcon(KIcon("edit-copy"));
    m_copyButton->setToolTip(i18n("Copy"));

    m_spellingButton = new QToolButton(m_dialog);
    m_spellingButton->setIcon(KIcon("tools-check-spelling"));
    m_spellingButton->setToolTip(i18n("Spell checking"));

    QHBoxLayout *horizontalLayout = new QHBoxLayout;
    horizontalLayout->addWidget(m_dictionaryComboBox);
    horizontalLayout->addWidget(m_spellingButton);
    horizontalLayout->addWidget(m_copyButton);

    QVBoxLayout *verticalLayout = new QVBoxLayout(m_dialog);
    verticalLayout->setSpacing(0);
    verticalLayout->setMargin(0);
    verticalLayout->addWidget(m_textEdit);
    verticalLayout->addLayout(horizontalLayout);

    setLanguage(configuration.readEntry("dictionary", m_textEdit->highlighter()->currentLanguage()));

    connect(this, SIGNAL(activate()), this, SLOT(toggleDialog()));
    connect(this, SIGNAL(destroyed()), m_dialog, SLOT(deleteLater()));
    connect(icon, SIGNAL(clicked()), this, SLOT(toggleDialog()));
    connect(m_dialog, SIGNAL(dialogResized()), this, SLOT(dialogResized()));
    connect(m_dragTimer, SIGNAL(timeout()), this, SLOT(toggleDialog()));
    connect(m_copyButton, SIGNAL(clicked()), this, SLOT(copyToClipboard()));
    connect(m_spellingButton, SIGNAL(clicked()), m_textEdit, SLOT(checkSpelling()));
    connect(m_dictionaryComboBox, SIGNAL(dictionaryChanged(QString)), this, SLOT(setLanguage(QString)));
}

void SpellCheck::dragEnterEvent(QGraphicsSceneDragDropEvent *event)
{
    if (!m_dialog->isVisible())
    {
        m_dragTimer->start();
    }

    Plasma::Applet::dragEnterEvent(event);
}

void SpellCheck::dragLeaveEvent(QGraphicsSceneDragDropEvent *event)
{
    m_dragTimer->stop();

    Plasma::Applet::dragLeaveEvent(event);
}

void SpellCheck::dropEvent(QGraphicsSceneDragDropEvent *event)
{
    if (!KUrl::List::canDecode(event->mimeData()))
    {
        event->ignore();

        return;
    }

    m_dragTimer->stop();

    if (!m_dialog->isVisible())
    {
        toggleDialog(false);
    }

    KUrl::List droppedUrls = KUrl::List::fromMimeData(event->mimeData());
    QTextCursor cursor = m_textEdit->textCursor();

    for (int i = 0; i < droppedUrls.count(); ++i)
    {
        QFile file(droppedUrls.at(i).toLocalFile());

        if (file.open(QIODevice::ReadOnly | QIODevice::Text))
        {
            QTextStream contents(file.readAll());
            contents.setAutoDetectUnicode(true);

            cursor.insertText(contents.readAll());
        }

        file.close();
    }
}

void SpellCheck::toggleDialog(bool pasteText)
{
    if (m_dialog->isVisible())
    {
        m_dialog->hide();

        m_textEdit->clear();
    }
    else
    {
        m_dialog->move(popupPosition(m_dialog->sizeHint()));

        m_dialog->show();

        if (pasteText)
        {
            m_textEdit->setText(QApplication::clipboard()->text());
        }

        m_textEdit->setFocus();
    }
}

void SpellCheck::dialogResized()
{
    KConfigGroup configuration = config();
    configuration.writeEntry("dialogSize", m_dialog->size());

    emit configNeedsSaving();
}

void SpellCheck::copyToClipboard()
{
    if (!m_textEdit->textCursor().selectedText().isEmpty())
    {
        QApplication::clipboard()->setText(m_textEdit->textCursor().selectedText());
    }
    else
    {
        QApplication::clipboard()->setText(m_textEdit->toPlainText());
    }
}

void SpellCheck::setLanguage(QString language)
{
    KConfigGroup configuration = config();
    configuration.writeEntry("dictionary", language);

    m_textEdit->setSpellCheckingLanguage(language);

    m_dictionaryComboBox->setCurrentByDictionary(language);

    emit configNeedsSaving();
}

#include "SpellCheck.moc"
