/***********************************************************************************
* STasks
* Copyright (C) 2009 Marcin Baszczewski <marcin.baszczewski@gmail.com>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*
***********************************************************************************/

//OWN
#include "stasks_icon.h"
#include "stasks_applet.h"
//QT
#include <QTimer>
#include <QPainter>
#include <QStyleOptionGraphicsItem>
#include <QFont>
//PLASMA
#include <Plasma/Applet>
#include <Plasma/PaintUtils>
//KDE
#include <KIcon>
#include <KIconEffect>
#include <KIconLoader>

STasksIcon::STasksIcon(QObject *parent,STasksApplet *applet) : QObject(parent),
    m_applet(applet),
    m_animationRepeater(new QTimer),
    m_animation(0),
    m_repeater(true)
{
    m_count=0;
}
STasksIcon::~STasksIcon()
{
    if (m_animation) 
    {
        Plasma::Animator::self()->stopCustomAnimation(m_animation);
    }
}
QPointF STasksIcon::getPos() 
{
    QSize iconSize = m_icon.actualSize(QSize(size(),size()));
    QPointF pos;
    if (m_applet->formFactor() == Plasma::Vertical)
    {
	if (m_rect.height()<m_rect.width())
	{
	    pos = QPointF(m_rect.width()/2 - iconSize.height()/2,m_rect.height()-m_rect.height()/2-iconSize.height()/2);
	}
	else
	{
	    qreal mheight=m_rect.width();
	    if (m_applet->iconShape() == 0)
	    {
		if ( ( qreal(m_rect.height()/m_rect.width()) >= 0.79) )
		mheight*=0.8;
	    }
	    pos = QPointF(m_rect.width()/2 - iconSize.height()/2,mheight/2 - iconSize.height()/2);
	}
    }
    else
    {
	if (m_rect.height()<m_rect.width())
	{
	    qreal mwidth=m_rect.height();
	    if (m_applet->iconShape() == 0)
	    {
		if ( ( qreal(m_rect.width()/m_rect.height()) >= 1.199) )
		mwidth*=1.2;
		else
		mwidth=m_rect.width();
	    }
	    pos = QPointF(mwidth/2 - iconSize.height()/2,m_rect.height()/2 - iconSize.height()/2);
	}
	else
	{
	    pos = QPointF(m_rect.width()/2-iconSize.width()/2,m_rect.height()/2 - iconSize.height()/2);
	}
    }
    return pos;
}

void STasksIcon::paint(QPainter *p, const QStyleOptionGraphicsItem *option, QWidget *widget)
{
    Q_UNUSED(option);
    Q_UNUSED(widget);
    m_pixmap = m_icon.pixmap(size());    
    if (m_pixmap.isNull())
    {
	kDebug() << "null";
	return;
    }
    if (m_currentAnimation=="animationMouseInIcon")
    animationMouseInIcon(m_progress);
    if (m_currentAnimation=="animationMouseOutIcon")
    animationMouseOutIcon(m_progress);    
    if (m_currentAnimation=="animationStartup")
    animationStartup(m_progress);    

    if (type() == STasksIcon::Group)
    {
	QPainter iconPainter(&m_pixmap);
	QPixmap icon;
	icon = KIcon("document-multiple").pixmap((m_pixmap.width() * 0.45), (m_pixmap.height() * 0.45));
	iconPainter.drawPixmap((m_pixmap.width() - icon.width()), (m_pixmap.height() - icon.height()), icon);
	iconPainter.end();
    }
    QPointF pos = getPos();
    p->drawPixmap(pos.x(),pos.y(), m_pixmap);
}
void STasksIcon::setRect(QRectF rect)
{
    if (rect!=m_rect)
    {
	m_rect=rect;
	updateIcon(m_icon);
    }
}
void STasksIcon::setIcon(QIcon icon)
{
    emit updateIcon(icon);
}
void STasksIcon::setType(IconType type)
{
    m_type = type;
}
STasksIcon::IconType STasksIcon::type()
{
    return m_type;
}
void STasksIcon::updateIcon(QIcon icon)
{
    if (icon.isNull())
    {
	return;
    }
    m_icon = icon;
}
qreal STasksIcon::size()
{
    qreal temp;
    if (m_rect.height()<m_rect.width())
    temp = m_rect.height()*m_applet->iconScale();
    else
    temp = m_rect.width()*m_applet->iconScale();
    if (temp<1) temp = 1;
    return temp;
}
void STasksIcon::startAnimation(QString animation,int duration,bool repeater)
{
    m_count=0;
    m_currentAnimationDuration = duration;
    m_currentAnimation = animation;

    m_animationRepeater->setInterval(duration);
    m_animationRepeater->start();
    
    
    repeatAnimation();

    m_repeater = repeater;
    connect(m_animationRepeater, SIGNAL(timeout()), this, SLOT(repeatAnimation()));
}
void STasksIcon::stopAnimation()
{
    m_animationRepeater->stop();
    m_currentAnimation = "";
    Plasma::Animator::self()->stopCustomAnimation(m_animation);
    m_animation = 0;
}
void STasksIcon::repeatAnimation()
{    
    if ( (!m_repeater) && (m_count!=0) )
    return;

    if (m_animation)
    {
        Plasma::Animator::self()->stopCustomAnimation(m_animation);
    }
    m_animation = Plasma::Animator::self()->customAnimation(m_applet->fps() / (1000 / m_currentAnimationDuration), m_currentAnimationDuration, Plasma::Animator::LinearCurve, this, "animation");
    m_count++;
}
void STasksIcon::animation(qreal progress)
{
    m_progress = progress;  
    emit update();
}
void STasksIcon::animationMouseInIcon(qreal progress)
{
    KIconEffect *effect = KIconLoader::global()->iconEffect();
    if (effect->hasEffect(KIconLoader::Desktop, KIconLoader::ActiveState)) 
    {
	if (qFuzzyCompare(qreal(1.0), progress)) 
	{
	    m_pixmap = effect->apply(m_pixmap, KIconLoader::Desktop, KIconLoader::ActiveState);
	} 
	else 
	{
	    m_pixmap = Plasma::PaintUtils::transition(m_pixmap,effect->apply(m_pixmap, KIconLoader::Desktop,KIconLoader::ActiveState), progress);
	}
    }
}
void STasksIcon::animationMouseOutIcon(qreal progress)
{
    KIconEffect *effect = KIconLoader::global()->iconEffect();
    if (effect->hasEffect(KIconLoader::Desktop, KIconLoader::ActiveState)) 
    {
	m_pixmap = Plasma::PaintUtils::transition(m_pixmap,effect->apply(m_pixmap, KIconLoader::Desktop,KIconLoader::ActiveState), 1-progress);
    }
}
void STasksIcon::animationStartup(qreal progress)
{
    QPixmap pixmap = QPixmap(m_pixmap.width(), m_pixmap.height());
    pixmap.fill(Qt::transparent);
    int width = (m_pixmap.width() * ((progress < 0.5)?((progress * 0.5) + 0.5):(1 - (progress / 2))));
    int height = (m_pixmap.height() * ((progress < 0.5)?(1 - (progress / 2)):((progress * 0.5) + 0.5)));
    QPixmap scaled = m_pixmap.scaled(width, height, Qt::IgnoreAspectRatio, Qt::SmoothTransformation);
    if (!scaled.isNull())
    {
	QPainter pixmapPainter(&pixmap);
	pixmapPainter.drawPixmap(((m_pixmap.width() - width) / 2), ((m_pixmap.height() - height) / 2), scaled);


	pixmapPainter.end();
    }
    m_pixmap = pixmap;

    QPixmap pixmap2 = QPixmap(m_pixmap.width(), m_pixmap.height());
    pixmap2.fill(Qt::transparent);
    m_pixmap = Plasma::PaintUtils::transition(pixmap2, m_pixmap, 0.85);
}
#include "stasks_icon.moc"