/***********************************************************************************
* STasks
* Copyright (C) 2009 Marcin Baszczewski <marcin.baszczewski@gmail.com>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*
***********************************************************************************/

//OWN
#include "stasks_item.h"
#include "stasks_frame.h"
#include "stasks_task.h"
#include "stasks_applet.h"
#include "stasks_icon.h"
#include "stasks_light.h"
#include "stasks_tooltip.h"
//QT
#include <QApplication>
#include <QPainter>
#include <QFontMetrics>
#include <QSizeF>
#include <QGraphicsLinearLayout>
#include <QGraphicsSceneHoverEvent>
#include <QGraphicsSceneContextMenuEvent>
#include <QStyleOptionGraphicsItem>
#include <QGraphicsSceneMouseEvent>
#include <QTextLayout>
#include <QGraphicsScene>
#include <QGraphicsView>
#include <QMimeData>
#include <QTextDocument>
//KDE
#include <KIcon>
#include <KIconEffect>
#include <KIconLoader>
#include <KGlobalSettings>
#include <KColorUtils>
//PLASMA
#include <Plasma/FrameSvg>
#include <Plasma/Theme>
#include <Plasma/PaintUtils>
#include <Plasma/ToolTipManager>
#include <Plasma/Containment>
#include <Plasma/Corona>
#include <QTime>
#include <sys/time.h>

STasksItem::STasksItem(TaskManager::AbstractGroupableItem *abstractItem, STasksApplet *parent) : QGraphicsWidget(parent),
    m_applet(parent),
    m_backgroundPrefix("normal"),
    m_animId(0)
{    
    m_toolTipTimer = new QTimer;
    //m_toolTip = new STasksToolTip(QPixmap(), QString(), 200, qobject_cast<QWidget*>(this));
//     m_previewToolTip = new STasksToolTip(qobject_cast<QWidget*>(this));

    m_abstractItem = abstractItem;
    m_task = new STasksTask(abstractItem,this,m_applet);
    m_icon = new STasksIcon(this,m_applet);
    m_icon->setIcon(m_task->icon());

    m_mouseIn = false;
    m_activateTimer = 0;
    m_size=0;
    m_fadeIn = true;
    m_frame = new STasksFrame(this,m_applet);
    m_light = new STasksLight(this,m_applet);
    m_alpha = 1;
    m_additionalSize = 0;
    m_lastUpdate = 0;
    m_updateTimer = new QTimer();
    m_updateTimer->setInterval(1000/m_applet->fps());
    connect(m_updateTimer, SIGNAL(timeout()), this, SLOT(update()));


    setAcceptsHoverEvents(true);
    setAcceptDrops(true);
    //task signals
    connect(m_task, SIGNAL(update()),this, SLOT(update()));
    connect(m_task, SIGNAL(updateState()),this, SLOT(updateState()));
    connect(m_task, SIGNAL(updateToolTip()),this, SLOT(updateToolTip()));
    connect(m_task, SIGNAL(gotTask()),this,SLOT(publishIconGeometry()));
    //icon
    connect(m_icon, SIGNAL(update()),this, SLOT(update()));
    connect(m_task, SIGNAL(updateIcon(QIcon)), m_icon, SLOT(updateIcon(QIcon)));
   
    updateState();
    //frame
    connect(m_frame, SIGNAL(update()),this, SLOT(update()));
    //light
    connect(m_light, SIGNAL(update()),this, SLOT(update()));
    //additional
   
    //updateToolTip();
    if (m_task->type()==STasksTask::Startup)
    {
	m_icon->startAnimation("animationStartup",500,true);
	m_light->startAnimation("animationStartup",500,true);
    }
}
STasksItem::~STasksItem()
{
    foreach(int id,m_animations ) 
    {
	Plasma::Animator::self()->stopCustomAnimation(id);
    } 
    Plasma::ToolTipManager::self()->unregisterWidget(this);
    m_updateTimer->deleteLater();
}
qreal STasksItem::size() const
{
    return m_size;
}
void STasksItem::setSize(qreal size)
{
    m_size = size;
    if (m_applet->iconShape() == 0)
    {
	if (m_applet->formFactor() == Plasma::Vertical)
	{
	    m_width = size;
	    m_height = size*0.8;
	}
	else
	{
	    m_width = size*1.2;
	    m_height = size;
	}
    }
    else
    {
	m_width = size;
	m_height = size;
    }
    m_expanded=false;
    setPreferredWidth(m_width);
    setPreferredHeight(m_height);
    if (m_applet->expandTasks())
    {
	if (m_applet->keepExpanded()==3)
	{
	    if (m_applet->formFactor() == Plasma::Vertical)
	    setPreferredHeight(m_height+m_applet->expandingSize());
	    else
	    setPreferredWidth(m_width+m_applet->expandingSize());
	    m_expanded=true;
	}
	else if (m_applet->keepExpanded()==2)
	{
	    if ( (m_task->isOnCurrentDesktop()) || (m_task->isOnAllDesktops()) )
	    {
		if (m_applet->formFactor() == Plasma::Vertical)
		setPreferredHeight(m_height+m_applet->expandingSize());
		else
		setPreferredWidth(m_width+m_applet->expandingSize());
		m_expanded = true;
	    }
	    else
	    {
		m_expanded=false;
		setPreferredWidth(m_width);
		setPreferredHeight(m_height);
	    }
	}
    }
}
void STasksItem::update()
{
    if ( (QTime().elapsed()-m_lastUpdate) > (1000/m_applet->fps()) )
    {
	QGraphicsWidget::update();
	m_lastUpdate = QTime().elapsed();
	m_updateTimer->stop();
    }
    else
    m_updateTimer->start();
}
void STasksItem::updateToolTip()
{
    if (m_applet->m_toolTip->isVisible())
    {
        m_applet->m_toolTip->hide(true);
    }

    delete m_applet->m_toolTip;

    m_applet->m_toolTip = new STasksToolTip( 200, qobject_cast<QWidget*>(this));

    if (m_task->type()==STasksTask::Group)
    {
        m_applet->m_toolTip->setTasks(m_task->group()->members());
    }
    if (m_task->type()==STasksTask::Task)
    {
	TaskManager::ItemList task;
	task.append(m_task->taskItem());
	m_applet->m_toolTip->setTasks(task);
    }
   m_applet->m_toolTip->move(popupPosition(m_applet->m_toolTip->size(), true));

}
QPoint STasksItem::popupPosition(QSize size, bool center)
{
    QRect geometry = iconGeometry();
    QPoint position(geometry.left(), geometry.top());

    switch (m_applet->location())
    {
        case Plasma::BottomEdge:
            position = QPoint((position.x() - (center?((size.width() - boundingRect().size().width()) / 2):0)), (position.y() - size.height()));
        break;
        case Plasma::TopEdge:
            position = QPoint((position.x() - (center?((size.width() - boundingRect().size().width()) / 2):0)), (position.y() + boundingRect().size().height()));
        break;
        case Plasma::LeftEdge:
            position = QPoint((position.x() + boundingRect().size().width()), (position.y() - (center?((size.height() - boundingRect().size().height()) / 2):0)));
        break;
        case Plasma::RightEdge:
            position = QPoint((position.x() - size.width()), (position.y() - (center?((size.height() - boundingRect().size().height()) / 2):0)));
        break;
        default:
            if (position.y() - size.height() > 0)
            {
                position = QPoint((position.x() - (center?((size.width() - boundingRect().size().width()) / 2):0)), (position.y() - size.height()));
            }
            else
            {
                position = QPoint((position.x() - (center?((size.width() - boundingRect().size().width()) / 2):0)), (position.y() + boundingRect().size().height()));
            }
    }

    QRect screenRect = m_applet->containment()->corona()->screenGeometry(m_applet->containment()?m_applet->containment()->screen():-1);
    
    if (m_applet->location() != Plasma::LeftEdge && position.rx() + size.width() > screenRect.right())
    {
        position.rx() -= ((position.rx() + size.width()) - screenRect.right());
    }

    if (m_applet->location() != Plasma::TopEdge && position.ry() + size.height() > screenRect.bottom())
    {
       position.ry() -= ((position.ry() + size.height()) - screenRect.bottom());
    }

    position.rx() = qMax(0, position.rx());
    position.ry() = qMax(0, position.ry());
    return position;
}
QRect STasksItem::iconGeometry() const
{
    if (!scene() || !boundingRect().isValid()) {
        return QRect();
    }

    QGraphicsView *parentView = 0;
    QGraphicsView *possibleParentView = 0;
    // The following was taken from Plasma::Applet, it doesn't make sense to make the item an applet, and this was the easiest way around it.
    foreach (QGraphicsView *view, scene()->views()) {
        if (view->sceneRect().intersects(sceneBoundingRect()) ||
            view->sceneRect().contains(scenePos())) {
            if (view->isActiveWindow()) {
                parentView = view;
                break;
            } else {
                possibleParentView = view;
            }
        }
    }

    if (!parentView) {
        parentView = possibleParentView;

        if (!parentView) {
            return QRect();
        }
    }

    QRect rect = parentView->mapFromScene(mapToScene(boundingRect())).boundingRect().adjusted(0, 0, 1, 1);
    rect.moveTopLeft(parentView->mapToGlobal(rect.topLeft()));
    return rect;
}
void STasksItem::publishIconGeometry()
{
    if (m_task->type()==STasksTask::Task)
    {
	QRect rect = iconGeometry();
	m_task->task()->publishIconGeometry(rect);
	//m_toolTip->setWindow(m_task->task()->window());
    }
}
STasksTask *STasksItem::task()
{
    return m_task;
}
void STasksItem::toggleToolTip()
{
    if (m_mouseIn && isVisible())
    {
        if (!m_applet->m_toolTip->isVisible())
        {
            updateToolTip();
            m_applet->m_toolTip->show();
        }
    }
    else
    {
        m_applet->m_toolTip->hide();
    }
    m_toolTipTimer->stop();
}
void STasksItem::blockToolTip()
{
    m_toolTipTimer->stop();

    if (m_applet->m_toolTip->isVisible())
    {
        m_applet->m_toolTip->hide();
    }
}
void STasksItem::updateState()
{
    //m_applet->sorting(this);
    publishIconGeometry();
    m_icon->stopAnimation();
    m_light->stopAnimation();

    QString newBackground;
    if (m_task->demandsAttention())
    {
        newBackground = "attention";
	m_light->startAnimation("animationAttention",900,true);
    } 
    else if (m_task->isMinimized())
    {
        newBackground = "minimized";
    } 
    else if (m_task->isActive())
    {
	emit itemActive(this);
        newBackground = "focus";
	if (m_applet->expandTasks())
	switch (m_applet->keepExpanded())
	{
	    case 1:
		if (!m_mouseIn)
		if (m_backgroundPrefix!="focus") 
		expandTask(m_applet->animationDuration(),true);
		break;
	    case 3:
		break;
	}
    } 
    else 
    {
        newBackground = "normal";
    }
    if (m_applet->expandTasks() && m_applet->keepExpanded()==2 && (m_task->task()))
    {
	if (m_task->isOnCurrentDesktop() || m_task->isOnAllDesktops() )
	expandTask(m_applet->animationDuration(),true);
	else
	expandTask(m_applet->animationDuration(),false);
    }
    if (newBackground != m_backgroundPrefix) 
    {
	if (m_applet->expandTasks())
	switch (m_applet->keepExpanded())
	{
	    case 1:
		if (m_backgroundPrefix=="focus") 
		if (!m_mouseIn)
		if (newBackground!="focus")
		expandTask(m_applet->animationDuration(),false);
		break;
	    case 3:
		break;
	}
	
	m_backgroundPrefix = newBackground;
	m_frame->fade(newBackground,100,true);
    }
}

void STasksItem::contextMenuEvent(QGraphicsSceneContextMenuEvent *event)
{
    blockToolTip();
    m_task->popup(QCursor::pos());
    event->accept();
}
void STasksItem::mousePressEvent(QGraphicsSceneMouseEvent *event)
{
    update();
    event->accept();
}
void STasksItem::mouseMoveEvent(QGraphicsSceneMouseEvent *event)
{
//     if (m_applet->sortingStrategy()!=TaskManager::GroupManager::ManualSorting)
//     {
// 	return;
//     }
    if (QPoint(event->screenPos() - event->buttonDownScreenPos(Qt::LeftButton)).manhattanLength() < QApplication::startDragDistance()) 
    {
        return;
    }

    QByteArray data;
    data.resize(sizeof(STasksItem*));
    STasksItem *selfPtr = this;
    memcpy(data.data(), &selfPtr, sizeof(STasksItem*));

    QMimeData* mimeData = new QMimeData();
    mimeData->setData("STasksItem",data);
    //mimeData->setText("STasksItem");
    //setAdditionalMimeData(mimeData);
    

    QDrag *drag = new QDrag(event->widget());
    drag->setMimeData(mimeData);
    drag->setPixmap(m_task->icon().pixmap(20));
   // drag->setDragCursor( set the correct cursor //TODO
    drag->exec();
}
void STasksItem::mouseReleaseEvent(QGraphicsSceneMouseEvent *event)
{
    if (event->button() == Qt::LeftButton)
    {
	m_toolTipTimer->stop();

	if (m_applet->m_toolTip->isVisible())
	{
	    m_applet->m_toolTip->hide();
	}

	if (m_applet->m_toolTip->isVisible())
	{
	    m_applet->m_toolTip->hide();
	}


	if (m_task->type()==STasksTask::Task)
	{
	    publishIconGeometry();
	    m_task->task()->activateRaiseOrIconify();
	}
	if (m_task->type()==STasksTask::Group)
	{
	    updateToolTip();
	    m_applet->m_toolTip->show();
// 	    TaskManager::GroupPopupMenu *groupMenu;
// 	    groupMenu = new TaskManager::GroupPopupMenu(qobject_cast<QWidget*>(this), m_task->group(), m_applet->m_groupManager);
// 	    groupMenu->exec(QCursor::pos());
// 
// 	    delete groupMenu;
	}
    }
}
void STasksItem::activate()
{
    if (m_task->type()==STasksTask::Task)
    {
	m_task->task()->activate();
    }
}
void STasksItem::dragEnterEvent(QGraphicsSceneDragDropEvent *event)
{
    if (event->mimeData()->hasFormat("STasksItem"))
    {
        //event->ignore(); //ignore it so the taskbar gets the event
	event->acceptProposedAction();
        return;
    }
    event->accept();
    if (!m_activateTimer) 
    {
	m_activateTimer = new QTimer(this);
        m_activateTimer->setSingleShot(true);
        m_activateTimer->setInterval(300);
        connect(m_activateTimer, SIGNAL(timeout()), this, SLOT(activate()));
    }
    m_activateTimer->start();
}
void STasksItem::dragMoveEvent(QGraphicsSceneDragDropEvent *event)
{
    Q_UNUSED(event);
    if (m_activateTimer) 
    {
        m_activateTimer->start();
    }
}
void STasksItem::dragLeaveEvent(QGraphicsSceneDragDropEvent *event)
{
    Q_UNUSED(event);
    if (m_activateTimer) 
    {
        delete m_activateTimer;
        m_activateTimer = 0;
    }
}
void STasksItem::dropEvent(QGraphicsSceneDragDropEvent *event)
{
    if (m_activateTimer) 
    {
        delete m_activateTimer;
        m_activateTimer = 0;
    }
    m_applet->dropEvent(event);
}

void STasksItem::hoverMoveEvent(QGraphicsSceneHoverEvent* e)
{
    m_light->setPoint(e->pos());
    update();
    QGraphicsWidget::hoverMoveEvent(e);
}
void STasksItem::hoverEnterEvent(QGraphicsSceneHoverEvent *)
{	
    m_mouseIn = true;
    m_light->setMouse(true);
    m_frame->fade("hover",m_applet->animationDuration(),true);
    if (m_applet->expandTasks())
    switch (m_applet->keepExpanded())
    {
	case 0:
	    expandTask(m_applet->animationDuration(),true);
	    break;
	case 1:
	    if ( !m_task->isActive() && m_backgroundPrefix!="focus" )
	    {
		expandTask(m_applet->animationDuration(),true);
	    }
	    break;
	case 2:
	    if (!m_task->isOnCurrentDesktop())
	    expandTask(m_applet->animationDuration(),true);
	    break;
	case 3:
	    break;
    }
    if (m_task->type()!=STasksTask::Startup)
    m_icon->startAnimation("animationMouseInIcon",m_applet->animationDuration(),false);
    //m_light->startAnimation("animationAttention",900,true);
    m_toolTipTimer->setInterval(500);
    m_toolTipTimer->start();
    connect(m_toolTipTimer, SIGNAL(timeout()), this, SLOT(toggleToolTip()));
}

void STasksItem::hoverLeaveEvent(QGraphicsSceneHoverEvent *event)
{
    Q_UNUSED(event)

    m_mouseIn = false;
    m_light->setMouse(false);
    QString backgroundPrefix;

    if (m_task->demandsAttention())
    {
	backgroundPrefix = "attention";
    } 
    else if (m_task->isMinimized())
    {
	backgroundPrefix = "minimized";
    } 
    else if (m_task->isActive())
    {
	backgroundPrefix = "focus";
    } 
    else 
    {
	backgroundPrefix = "normal";
    }


    
    m_frame->fade(backgroundPrefix,m_applet->animationDuration(),false);
    if (m_applet->expandTasks())
    switch (m_applet->keepExpanded())
    {
	case 0:
	    expandTask(m_applet->animationDuration(),false);
	    break;
	case 1:
	    if (!m_task->isActive())
	    expandTask(m_applet->animationDuration(),false);
	    break;
	case 2:
	    if (!m_task->isOnCurrentDesktop())
	    expandTask(m_applet->animationDuration(),false);
	    break;
	case 3:
	    break;
    }  
    if (m_task->type()!=STasksTask::Startup)
    m_icon->startAnimation("animationMouseOutIcon",m_applet->animationDuration(),false);

    m_toolTipTimer->setInterval(200);
    m_toolTipTimer->start();

    connect(m_toolTipTimer, SIGNAL(timeout()), this, SLOT(toggleToolTip()));
}
void STasksItem::expandTask(int duration,bool expandIn)
{
    if (m_expanded==expandIn)
    return;
    if (m_expand)
    {
	Plasma::Animator::self()->stopCustomAnimation(m_expand);
    }   
    if (expandIn)
    m_expand = Plasma::Animator::self()->customAnimation(m_applet->fps() / (1000 / duration), duration,Plasma::Animator::LinearCurve, this, "animationExpandTask1");
    else
    m_expand = Plasma::Animator::self()->customAnimation(m_applet->fps() / (1000 / duration), duration,Plasma::Animator::LinearCurve, this, "animationExpandTask2");
    m_expanded = expandIn;
}
void STasksItem::animationExpandTask1(qreal progress)
{
    if (qFuzzyCompare(qreal(1.0), progress)) 
    {
         m_expand = 0;
    }
    qreal m_newWidth = 0;
    m_newWidth = m_applet->expandingSize();

    m_additionalSize = progress*m_newWidth;
    if (m_applet->formFactor() == Plasma::Vertical)
    setPreferredHeight(m_height+m_additionalSize);
    else
    setPreferredWidth(m_width+m_additionalSize);
    update();
}
void STasksItem::animationExpandTask2(qreal progress)
{
    if (qFuzzyCompare(qreal(1.0), progress)) 
    {
         m_expand = 0;
    }
    qreal m_newWidth = 0;
    m_newWidth = m_applet->expandingSize();
    //m_additionalSize = m_newWidth*(1-progress);
    if (m_newWidth*(1-progress)>m_additionalSize)
    m_additionalSize *= (1-progress);
    else
    m_additionalSize = m_newWidth*(1-progress);

    if (m_applet->formFactor() == Plasma::Vertical)
    setPreferredHeight(m_height+m_additionalSize);
    else
    setPreferredWidth(m_width+m_additionalSize);

    update();

}
void STasksItem::paint(QPainter *p, const QStyleOptionGraphicsItem *option, QWidget *widget)
{
    p->setRenderHint(QPainter::Antialiasing);

    m_frame->resize(boundingRect().size());
    m_frame->paint(p,option,widget);
    m_icon->setRect(boundingRect());
    //draw ligt
    if (m_applet->lights())
    {
	m_light->setRect(m_frame->contentsRect());
	m_light->paint(p,option,widget);
    }
    //draw text
    if (m_applet->expandTasks())
    drawText(p,option,widget);  
    //draw icon
    if (m_applet->formFactor() == Plasma::Vertical)
    {
	p->save();
	if ((boundingRect().height()-m_height)>0)
	p->translate(0,(boundingRect().height()-m_height));
    }
    if (m_task->type()==STasksTask::Group)
    m_icon->setType(STasksIcon::Group);
    else
    m_icon->setType(STasksIcon::Other);
    m_icon->paint(p,option,widget);
    if (m_applet->formFactor() == Plasma::Vertical)
    {
	p->restore();
    }
}
void STasksItem::drawText(QPainter *p, const QStyleOptionGraphicsItem *option, QWidget *widget)
{
    Q_UNUSED(option);
    Q_UNUSED(widget);
//     if (!m_expand && !m_mouseIn)
//     {
// 	return;
//     }
//     if (m_additionalSize==0)
//     return;
    p->setPen(QPen(textColor(), 1.0));

    QTextLayout layout;
    layout.setFont(KGlobalSettings::taskbarFont());
    QRect rect;
    if (m_applet->formFactor() == Plasma::Vertical)
    rect = QRect(m_height/2 + m_icon->size()/2+3,0 ,boundingRect().height()-m_height-6,m_width);
    else
    rect = QRect(m_width/2 + m_icon->size()/2+3,0 ,boundingRect().width()-m_width-6,m_height);
   
    QSize temp = layoutText(layout, m_task->text(), rect.size());
    if (m_applet->formFactor() == Plasma::Vertical)
    {
	p->save();
	p->rotate(-90);
	p->translate(-(boundingRect().height()),0);
    }
    drawTextLayout(p, layout, rect);
    if (m_applet->formFactor() == Plasma::Vertical)
    {
	p->restore();
    }
}

QSize STasksItem::layoutText(QTextLayout &layout, const QString &text,const QSize &constraints) const
{
    Q_UNUSED(constraints);
    QFontMetrics metrics(layout.font());
    int leading     = metrics.leading();
    int height      = 0;
    int widthUsed   = 0;
    QTextLine line;
    layout.setText(text);

    layout.beginLayout();
    line = layout.createLine();
    height += leading;
    line.setPosition(QPoint(0, height));

    widthUsed =metrics.width (text) ;

    layout.endLayout();
    return QSize(widthUsed, height);
}

void STasksItem::drawTextLayout(QPainter *painter, const QTextLayout &layout, const QRect &rect) const
{
    if (rect.width() < 1 || rect.height() < 1) 
    {
        return;
    }
    QPixmap pixmap(rect.size());
    pixmap.fill(Qt::transparent);

    QPainter p(&pixmap);
    p.setPen(painter->pen());

    // Create the alpha gradient for the fade out effect
    QLinearGradient alphaGradient(0, 0, 1, 0);
    alphaGradient.setCoordinateMode(QGradient::ObjectBoundingMode);
    if (layout.textOption().textDirection() == Qt::LeftToRight)
    {
        alphaGradient.setColorAt(0, QColor(0, 0, 0, 255));
        alphaGradient.setColorAt(1, QColor(0, 0, 0, 0));
    } 
    else
    {
        alphaGradient.setColorAt(0, QColor(0, 0, 0, 0));
        alphaGradient.setColorAt(1, QColor(0, 0, 0, 255));
    }
    QFontMetrics fm(layout.font());
    int textHeight = layout.lineCount() * fm.lineSpacing();

    QPointF position = textHeight < rect.height() ? QPointF(0, (rect.height() - textHeight) / 2 + (fm.tightBoundingRect("M").height() - fm.xHeight())) : QPointF(0, 0);
    QList<QRect> fadeRects;
    int fadeWidth = 30;
    // Draw each line in the layout
    for (int i = 0; i < layout.lineCount(); i++)
    {
        QTextLine line = layout.lineAt(i);
        line.draw(&p, position);
        // Add a fade out rect to the list if the line is too long
        if (line.naturalTextWidth() > rect.width())
        {
            int x = int(qMin(line.naturalTextWidth(), (qreal)pixmap.width())) - fadeWidth;
            int y = int(line.position().y() + position.y());
            QRect r = QStyle::visualRect(layout.textOption().textDirection(), pixmap.rect(),
                                         QRect(x, y, fadeWidth, int(line.height())));
            fadeRects.append(r);
        }
    }
    // Reduce the alpha in each fade out rect using the alpha gradient
    if (!fadeRects.isEmpty())
    {
        p.setCompositionMode(QPainter::CompositionMode_DestinationIn);
        foreach (const QRect &rect, fadeRects) 
	{
            p.fillRect(rect, alphaGradient);
        }
    }
    p.end();
    QColor shadowColor;
    if (textColor().value() < 128) 
    {
        shadowColor = Qt::white;
    } 
    else 
    {
        shadowColor = Qt::black;
    }
    QImage shadow = pixmap.toImage();
    Plasma::PaintUtils::shadowBlur(shadow, 2, shadowColor);
    painter->drawImage(rect.topLeft() + QPoint(1,2), shadow);
    painter->drawPixmap(rect.topLeft(), pixmap);
}
QColor STasksItem::textColor() const
{
    QColor color;
    qreal bias;
    Plasma::Theme *theme = Plasma::Theme::defaultTheme();
    if ((m_oldBackgroundPrefix == "attention" || m_backgroundPrefix == "attention")) 
    {
        if (!m_animId && m_backgroundPrefix != "attention") 
	{
            color = theme->color(Plasma::Theme::TextColor);
        } 
	else if (!m_animId) 
	{
            color = theme->color(Plasma::Theme::ButtonTextColor);
        } 
	else 
	{
            if (m_oldBackgroundPrefix == "attention") 
	    {
                bias = 1 - m_alpha;
            } 
	    else 
	    {
                bias = m_alpha;
            }

            color = KColorUtils::mix(theme->color(Plasma::Theme::TextColor),
            theme->color(Plasma::Theme::ButtonTextColor), bias);
        }
    } 
    else 
    {
        color = theme->color(Plasma::Theme::TextColor);
    }
    if (m_task->isMinimized())
    {
        color.setAlphaF(0.85);
    }
    return color;
}
#include "stasks_item.moc"