/*
 *  Player - One Hell of a Robot Server
 *  Copyright (C) 2006
 *     Raymond Sheh, Luke Gumbley
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <pthread.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <errno.h>
#include <sys/mman.h>
#include <stdio.h>
#include <poll.h>

#include "UvcInterface.h"

int UvcInterface::Open(void)
{
	int ret=0;
	int i;

	if(fd!=-1) Close();

	try
	{
		fd=open(device,O_RDWR|O_NONBLOCK);
		if(fd==-1)
			throw "Error opening device";

		//retrieve device capabilities
	    memset(&cap, 0, sizeof(v4l2_capability));
	    ret=ioctl(fd,VIDIOC_QUERYCAP,&cap);
	    if(ret==-1)
	    	throw "Error querying capabilities";

	    //check for video capture capability
	    if(!cap.capabilities&V4L2_CAP_VIDEO_CAPTURE)
	    	throw "Device does not support video capture";

		//check for streaming i/o capability (unnecessary?)
	    if(!cap.capabilities&V4L2_CAP_STREAMING)
	    	throw "Device does not support streaming i/o";

	     //check for read/write capability
		if(!cap.capabilities&V4L2_CAP_READWRITE)
			throw "Device does not support read/write i/o";

		//set capture format
		memset(&fmt, 0, sizeof(v4l2_format));
		fmt.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
		fmt.fmt.pix.width = width;
		fmt.fmt.pix.height = height;
		fmt.fmt.pix.pixelformat = V4L2_PIX_FMT_MJPEG;
		fmt.fmt.pix.field = V4L2_FIELD_ANY;
		ret=ioctl(fd,VIDIOC_S_FMT,&fmt);
		if(ret==-1)
			printf("Unable to set format. (%d, %d)\r\n",ret,errno);

		ret=ioctl(fd,VIDIOC_G_FMT,&fmt);
		if(ret==-1)
			printf("Unable to retrieve format. (%d, %d)\r\n",ret,errno);

		//request memory buffers
		v4l2_requestbuffers rb;
		memset(&rb,0,sizeof(v4l2_requestbuffers));
		rb.count=2; //two buffers; one for the device to fill and one for the driver to read
		rb.type=V4L2_BUF_TYPE_VIDEO_CAPTURE;
		rb.memory=V4L2_MEMORY_MMAP;
		ret = ioctl(fd,VIDIOC_REQBUFS,&rb);
		if(ret==-1)
			throw "Unable to allocate buffers";

		//general-purpose buffer info struct
		v4l2_buffer buf;

		buffer[0]=0;
		buffer[1]=0;

		for(i=0;i<2;i++)
		{
			//query a buffer
			memset(&buf,0,sizeof(v4l2_buffer));
			buf.index=i;
			buf.type=V4L2_BUF_TYPE_VIDEO_CAPTURE;
			buf.memory=V4L2_MEMORY_MMAP;
			ret=ioctl(fd,VIDIOC_QUERYBUF,&buf);
			if(ret==-1)
				throw "Unable to query frame";

			//retrieve the buffer memory location
			buffer[i]=(unsigned char*)mmap(0,buf.length,PROT_READ,MAP_SHARED,fd,buf.m.offset);
			length[i]=buf.length;

			if(buffer[i]==MAP_FAILED)
			{
				buffer[i]=0;
				throw "Unable to map frame";
			}
		}

		for(i=0;i<2;i++)
		{
			memset(&buf,0,sizeof(v4l2_buffer));
			buf.index=i;
			buf.type=V4L2_BUF_TYPE_VIDEO_CAPTURE;
			buf.memory=V4L2_MEMORY_MMAP;
			ret=ioctl(fd,VIDIOC_QBUF,&buf);
			if(ret==-1)
				throw "Unable to queue frame";
		}

		int type=V4L2_BUF_TYPE_VIDEO_CAPTURE;
		ret=ioctl(fd,VIDIOC_STREAMON,&type);
		if(ret==-1)
			throw "Unable to initiate video stream";

		int bufLength=length[0];
		if(length[1]>bufLength) bufLength=length[1];
		frame=new unsigned char[bufLength+dht_size];
	}
	catch(char const *msg)
	{
		printf("%s. (%d, %d)\r\n",msg,ret,errno);
		Close();
		return -1;
	}
	return 0;
}

int UvcInterface::Close(void)
{
	int ret,i;

	if(fd==-1) return 0;

	int type=V4L2_BUF_TYPE_VIDEO_CAPTURE;
	ret=ioctl(fd,VIDIOC_STREAMOFF,&type);
	if(ret==-1)
		printf("Unable to terminate video stream. (%d, %d)\r\n",ret,errno);

	for(i=0;i<2;i++)
	{
		if(buffer[i]!=0)
		{
			ret=munmap(buffer[i],length[i]);
			if(ret==-1)
				printf("Unable to unmap frame. (%d, %d)\r\n",ret,errno);
			buffer[i]=0;
		}
	}

	ret=close(fd);
	if(ret==-1)
		printf("Unable to close video device. (%d, %d)\r\n",ret,errno);
	fd=-1;

	if(frame) delete [] frame;
	frame=0;
	frameSize=0;

	return 0;
}

int UvcInterface::Read()
{
	int ret=0;
	const int hdr=0xaf;

	try
	{
		pollfd pfd;
		pfd.fd=fd;
		pfd.events=POLLIN;
		do
		{
			poll(&pfd,1,100);
			pthread_testcancel();

		} while(pfd.revents==0);

		v4l2_buffer buf;
	    memset(&buf,0,sizeof(v4l2_buffer));
	    buf.type=V4L2_BUF_TYPE_VIDEO_CAPTURE;
	    buf.memory=V4L2_MEMORY_MMAP;
	    ret=ioctl(fd,VIDIOC_DQBUF,&buf);
	    if(ret==-1)
			throw "Unable to dequeue frame";

		memcpy(frame,buffer[buf.index],hdr);
		memcpy(frame+hdr,dht_data,dht_size);
		memcpy(frame+hdr+dht_size,buffer[buf.index]+hdr,(buf.bytesused-hdr));
		frameSize=buf.bytesused+dht_size;

		ret=ioctl(fd,VIDIOC_QBUF,&buf);
		if(ret==-1)
			throw "Unable to requeue frame";
	}
	catch(char const *msg)
	{
		printf("%s. (%d, %d)\r\n",msg,ret,errno);
		return -1;
	}
	return 0;
}

int UvcInterface::GetWidth(void) const
{
	if(fd==-1) return -1;
	return fmt.fmt.pix.width;
}

int UvcInterface::GetHeight(void) const
{
	if(fd==-1) return -1;
	return fmt.fmt.pix.height;
}

// Default Huffman Table (DHT) for the MJPG frames

// table size
const int UvcInterface::dht_size=432;

// table data
const unsigned char UvcInterface::dht_data[dht_size] = {
    0xff, 0xc4, 0x00, 0x1f, 0x00, 0x00, 0x01, 0x05, 0x01, 0x01, 0x01, 0x01,
	0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a,
    0x0b, 0xff, 0xc4, 0x00, 0xb5, 0x10, 0x00, 0x02,
    0x01, 0x03, 0x03, 0x02, 0x04, 0x03, 0x05, 0x05, 0x04, 0x04, 0x00, 0x00,
    0x01, 0x7d, 0x01, 0x02, 0x03, 0x00, 0x04, 0x11,
    0x05, 0x12, 0x21, 0x31, 0x41, 0x06, 0x13, 0x51, 0x61, 0x07, 0x22, 0x71,
    0x14, 0x32, 0x81, 0x91, 0xa1, 0x08, 0x23, 0x42,
    0xb1, 0xc1, 0x15, 0x52, 0xd1, 0xf0, 0x24, 0x33, 0x62, 0x72, 0x82, 0x09,
    0x0a, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x25, 0x26,
    0x27, 0x28, 0x29, 0x2a, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x43,
    0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x53,
    0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x63, 0x64, 0x65, 0x66, 0x67,
    0x68, 0x69, 0x6a, 0x73, 0x74, 0x75, 0x76, 0x77,
    0x78, 0x79, 0x7a, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x92,
    0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a,
    0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xb2, 0xb3, 0xb4,
    0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xc2, 0xc3,
    0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6,
    0xd7, 0xd8, 0xd9, 0xda, 0xe1, 0xe2, 0xe3, 0xe4,
    0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6,
    0xf7, 0xf8, 0xf9, 0xfa, 0xff, 0xc4, 0x00, 0x1f,
    0x01, 0x00, 0x03, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x02,
    0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0xff, 0xc4, 0x00,
    0xb5, 0x11, 0x00, 0x02, 0x01, 0x02, 0x04, 0x04,
    0x03, 0x04, 0x07, 0x05, 0x04, 0x04, 0x00, 0x01, 0x02, 0x77, 0x00, 0x01,
    0x02, 0x03, 0x11, 0x04, 0x05, 0x21, 0x31, 0x06,
    0x12, 0x41, 0x51, 0x07, 0x61, 0x71, 0x13, 0x22, 0x32, 0x81, 0x08, 0x14,
    0x42, 0x91, 0xa1, 0xb1, 0xc1, 0x09, 0x23, 0x33,
    0x52, 0xf0, 0x15, 0x62, 0x72, 0xd1, 0x0a, 0x16, 0x24, 0x34, 0xe1, 0x25,
    0xf1, 0x17, 0x18, 0x19, 0x1a, 0x26, 0x27, 0x28,
    0x29, 0x2a, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x43, 0x44, 0x45, 0x46,
    0x47, 0x48, 0x49, 0x4a, 0x53, 0x54, 0x55, 0x56,
    0x57, 0x58, 0x59, 0x5a, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a,
    0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7a,
    0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x92, 0x93, 0x94,
    0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0xa2, 0xa3,
    0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6,
    0xb7, 0xb8, 0xb9, 0xba, 0xc2, 0xc3, 0xc4, 0xc5,
    0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8,
    0xd9, 0xda, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7,
    0xe8, 0xe9, 0xea, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa
};
