/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file sdl_music.cpp
 * \brief Implementation of the bear::audio::sdl_music class.
 * \author Julien Jorge
 */
#include "audio/sdl_music.hpp"
#include <SDL/SDL.h>
#include <claw/exception.hpp>
#include <claw/logger.hpp>

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param file A stream containing a wave, mod, midi, ogg or mp3 file.
 */
bear::audio::sdl_music::sdl_music( std::istream& file )
  : m_music(NULL)
{
  file.seekg( 0, std::ios::end );
  unsigned int file_size = file.tellg();
  file.seekg( 0, std::ios::beg );

  m_data = new char[file_size];

  file.read( m_data, file_size );

  m_sdl_rw = SDL_RWFromMem(m_data, file_size);

  if (m_sdl_rw)
    m_music = Mix_LoadMUS_RW( m_sdl_rw );

  if (!m_music)
    throw CLAW_EXCEPTION( Mix_GetError() );
} // sdl_music::sdl_music()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
bear::audio::sdl_music::~sdl_music()
{
  stop();

  Mix_FreeMusic( m_music );

  delete[] m_data;
} // sdl_music::~sdl_music()

/*----------------------------------------------------------------------------*/
/**
 * \brief Start to play the music.
 */
void bear::audio::sdl_music::play()
{
  stop(); // stop any playing music

  int channel = Mix_PlayMusic(m_music, -1);

  if (channel == -1)
    claw::logger << claw::log_error << "sdl_music::play(): " << Mix_GetError()
                 << claw::lendl;
} // sdl_music::play()

/*----------------------------------------------------------------------------*/
/**
 * \brief Stop the music.
 */
void bear::audio::sdl_music::stop()
{
  Mix_HaltMusic();
} // sdl_music::stop()
