/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file static_map.tpp
 * \brief Implementation of the bear::concept::static_map class.
 * \author Julien Jorge.
 */
#include <limits>
#include <claw/assert.hpp>

/*----------------------------------------------------------------------------*/
template <class ItemType, class ItemTraits>
const ItemTraits
bear::concept::static_map<ItemType, ItemTraits>::s_item_traits = ItemTraits();

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param width Width of the whole map.
 * \param height Height of the whole map.
 * \param box_size Size of the boxes.
 *
 * \remark Big values for box_size increase the compression but also the work
 *         done by get_area(). In an other hand, small values decrease the
 *         work but also the compression. You should find a good compromize
 *         (the empty_cells() and cells_load() methods are made to help you).
 */
template <class ItemType, class ItemTraits>
bear::concept::static_map<ItemType, ItemTraits>::static_map
( unsigned int width, unsigned int height, unsigned int box_size )
  : m_box_size(box_size),
    m_size(width / m_box_size + 1, height / m_box_size + 1),
    m_map( m_size.x, column(m_size.y) )
{
  CLAW_PRECOND( width > 0 );
  CLAW_PRECOND( height > 0 );

  // in fact, floating point exception had already been thrown
  CLAW_PRECOND( box_size > 0 );
} // static_map::static_map()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add an item in the map.
 * \param item The item to add.
 */
template <class ItemType, class ItemTraits>
void bear::concept::static_map<ItemType, ItemTraits>::insert
( const item_type& item )
{
  claw::math::coordinate_2d<unsigned int> top_left, top_right;
  claw::math::coordinate_2d<unsigned int> bottom_left, bottom_right;

  item_box_to_local_coordinates( item, top_left, top_right, bottom_left,
                                 bottom_right );

  CLAW_PRECOND( top_left.x < m_size.x );
  CLAW_PRECOND( top_left.y < m_size.y );

  m_map[top_left.x][top_left.y].push_front(item);

  if ( (top_right != top_left) && (top_right.x < m_size.x) )
    m_map[top_right.x][top_right.y].push_front(item);

  if ( (bottom_left != top_left) && (bottom_left.y < m_size.y) )
    m_map[bottom_left.x][bottom_left.y].push_front(item);

  if ( (bottom_right != top_right) && (bottom_right != bottom_left)
       && (bottom_right.x < m_size.x) && (bottom_right.y < m_size.y) )
    m_map[bottom_right.x][bottom_right.y].push_front(item);
} // static_map::insert()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get all items inside a rectangular region of the map.
 * \param area The area from which to take the items.
 * \param items (out) The founded items.
 */
template <class ItemType, class ItemTraits>
void bear::concept::static_map<ItemType, ItemTraits>::get_area
( const claw::math::rectangle<unsigned int>& area, item_set& items ) const
{
  unsigned int x, y;
  typename item_box::const_iterator it;

  unsigned int min_x = area.position.x / m_box_size;
  unsigned int max_x = area.right() / m_box_size;
  unsigned int min_y = area.position.y / m_box_size;
  unsigned int max_y = area.bottom() / m_box_size;

  if ( max_x >= m_size.x )
    max_x = m_size.x - 1;

  if ( max_y >= m_size.y )
    max_y = m_size.y - 1;

  for (x=min_x; x<=max_x; ++x)
    for (y=min_y; y<=max_y; ++y)
      for (it=m_map[x][y].begin(); it!=m_map[x][y].end(); ++it)
        {
          claw::math::coordinate_2d<unsigned int> local_position(x, y);

          if ( can_be_added( local_position, *it, area ) )
            items.insert( *it );
        }
} // static_map::get_area()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get all items.
 * \param items (out) The founded items.
 */
template <class ItemType, class ItemTraits>
void bear::concept::static_map<ItemType, ItemTraits>::get_all
( item_set& items ) const
{
  unsigned int x, y;
  typename item_box::const_iterator it;

  for (x=0; x!=m_map.size(); ++x)
    for (y=0; y!=m_map[x].size(); ++y)
      items.insert( m_map[x][y].begin(), m_map[x][y].end() );
} // static_map::get_all()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell the number of empty cells in the compressed map.
 */
template <class ItemType, class ItemTraits>
unsigned int
bear::concept::static_map<ItemType, ItemTraits>::empty_cells() const
{
  unsigned int cells=0;
  unsigned int x, y;

  for (x=0; x!=m_map.size(); ++x)
    for (y=0; y!=m_map[x].size(); ++y)
      if ( m_map[x][y].empty() )
        ++cells;

  return cells;
} // static_map::empty_cells()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get some statistics about the cells content.
 * \param min (out) Minimum number of items found in a cell.
 * \param max (out) Maximum number of items found in a cell.
 * \param avg (out) Average number of items found in not empty cells.
 */
template <class ItemType, class ItemTraits>
void bear::concept::static_map<ItemType, ItemTraits>::cells_load
( unsigned int& min, unsigned int& max, double& avg ) const
{
  unsigned int not_empty_cells=0;
  unsigned int x, y;
  unsigned int load=0;

  min = std::numeric_limits<unsigned int>::max();
  max = 0;
  avg = 0;

  for (x=0; x!=m_map.size(); ++x)
    for (y=0; y!=m_map[x].size(); ++y)
      {
        unsigned int size = m_map[x][y].size();

        if ( size > max )
          max = size;

        if ( size < min )
          min = size;

        if (size != 0)
          {
            load += size;
            ++not_empty_cells;
          }
      }

  if ( (load != 0) && (not_empty_cells!=0) )
    avg = (double)load / (double)not_empty_cells;
} // static_map::cells_load()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the four points of an item's bounding box in the local
 *        coordinates.
 * \param item The item from which to take the box.
 * \param top_left (out) Local coordinates of the top left corner of the item.
 * \param top_right (out) Local coordinates of the top right corner of the item.
 * \param bottom_left (out) Local coordinates of the top left corner of the
 *        item.
 * \param bottom_right (out) Local coordinates of the top right corner of the
 *        item.
 */
template <class ItemType, class ItemTraits>
void
bear::concept::static_map<ItemType, ItemTraits>::item_box_to_local_coordinates
( const item_type& item,
  claw::math::coordinate_2d<unsigned int>& top_left,
  claw::math::coordinate_2d<unsigned int>& top_right,
  claw::math::coordinate_2d<unsigned int>& bottom_left,
  claw::math::coordinate_2d<unsigned int>& bottom_right ) const
{
  claw::math::rectangle<unsigned int> box;

  box = s_item_traits.get_bounding_box( item );

  top_left = box.position / m_box_size;
  top_right.set( box.right() / m_box_size, box.position.y / m_box_size );
  bottom_right.set( box.right() / m_box_size, box.bottom() / m_box_size );
  bottom_left.set( box.position.x / m_box_size, box.bottom() / m_box_size );
} // static_map::item_box_to_local_coordinates()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if an item can be added in the list of items.
 *
 * An item can be added in the list if he had not been already added. He is
 * already in the list if we have alfready found one of his corners.
 *
 * \param local_position Coordinates of the item in the local map.
 * \param item The item we would like to add.
 * \param area The real area from which we take items.
 */
template <class ItemType, class ItemTraits>
bool bear::concept::static_map<ItemType, ItemTraits>::can_be_added
( const claw::math::coordinate_2d<unsigned int>& local_position,
  const item_type& item, const claw::math::rectangle<unsigned int>& area ) const
{
  bool result;
  claw::math::coordinate_2d<unsigned int> top_left, top_right;
  claw::math::coordinate_2d<unsigned int> bottom_left, bottom_right;
  claw::math::rectangle<unsigned int> box;

  item_box_to_local_coordinates( item, top_left, top_right, bottom_left,
                                 bottom_right );

  box = s_item_traits.get_bounding_box( item );

  claw::math::coordinate_2d<unsigned int> real_top_right;
  claw::math::coordinate_2d<unsigned int> real_bottom_left;
  claw::math::coordinate_2d<unsigned int> real_bottom_right;

  real_top_right.set( box.right(), box.position.y );
  real_bottom_left.set( box.position.x, box.bottom() );
  real_bottom_right.set( box.right(), box.bottom() );

  if ( !area.includes( box.position ) )
    if ( !area.includes( real_top_right ) )
      if ( ! area.includes( real_bottom_left ) )
        if ( ! area.includes( real_bottom_right ) )
          result = false;
        else
          result = local_position == bottom_right;
      else
        result = local_position == bottom_left;
    else
      result = local_position == top_right;
  else
    result = local_position == top_left;

  return result;
} // static_map::can_be_added()

