/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file base_item.hpp
 * \brief The basic physical item of our game.
 * \author Julien Jorge
 */
#ifndef __ENGINE_BASE_ITEM_HPP__
#define __ENGINE_BASE_ITEM_HPP__

#include <string>

#include "engine/class_export.hpp"

#include "universe/base_entity.hpp"
#include "engine/item_handle.hpp"
#include "engine/scene_visual.hpp"
#include "visual/sprite.hpp"
#include "visual/animation.hpp"

namespace bear
{
  namespace engine
  {
    class world;

    /**
     * \brief The basic physical item of our game.
     * \author Julien Jorge
     *  - \a pos_x: \c The x-coordinate,
     *  - \a pos_y: \c The y-coordinate,
     *  - \a height: \c The height of the box,
     *  - \a width: \c The width of the box,
     *  - \a mass: \c Mass of the item,
     *  - \a angle: \c Angle of the item.
     */
    class ENGINE_EXPORT base_item:
      public universe::base_entity
    {
    public:
      /** \brief type of the unique identifier of items. */
      typedef unsigned int id_type;

      /** \brief The type of the class that stores our fields. */
      typedef physical_item_state state_type;

    public:
      static void print_allocated();

      base_item();
      base_item( const base_item& that );
      ~base_item();

      virtual void start();

      virtual void get_visual( std::list<scene_visual>& visuals ) const;

      virtual bool is_player() const;

      virtual bool set_u_integer_field
      ( const std::string& name, unsigned int value );
      virtual bool set_integer_field( const std::string& name, int value );
      virtual bool set_real_field( const std::string& name, double value );
      virtual bool set_bool_field( const std::string& name, bool value );
      virtual bool set_string_field
      ( const std::string& name, const std::string& value );
      virtual bool set_item_field
      ( const std::string& name, base_item* value );
      virtual bool set_sprite_field
      ( const std::string& name, visual::sprite* value );
      virtual bool set_animation_field
      ( const std::string& name, visual::animation* value );

      virtual bool set_u_integer_list_field
      ( const std::string& name, const std::vector<unsigned int>& value );
      virtual bool set_integer_list_field
      ( const std::string& name, const std::vector<int>& value );
      virtual bool set_real_list_field
      ( const std::string& name, const std::vector<double>& value );
      virtual bool set_bool_list_field
      ( const std::string& name, const std::vector<bool>& value );
      virtual bool set_string_list_field
      ( const std::string& name, const std::vector<std::string>& value );
      virtual bool set_item_list_field
      ( const std::string& name, const std::vector<base_item*>& value );
      virtual bool set_sprite_list_field
      ( const std::string& name, const std::vector<visual::sprite*>& value );
      virtual bool set_animation_list_field
      ( const std::string& name, const std::vector<visual::animation*>& value );

      virtual bool is_valid() const;

      void add_handle( item_handle* h );
      void remove_handle( item_handle* h );

      id_type get_id() const;
   
      virtual void kill();
      void set_owner( world& w );
      world* get_owner() const;

      virtual void hit
      ( base_item& that, const state_type& old_self,
        const state_type& old_that );

      virtual void collision
      ( physical_item& that, const physical_item_state& old_self,
        const physical_item_state& old_that );

    protected:
      void to_string( std::string& str ) const;

    private:
      /**
       * \brief Unique identifier.
       *
       * Valid identifiers are strictly greater than zero.
       */
      const id_type m_id;

      /** \brief The world in which we are. */
      world* m_owner;

      /** \brief The handles on me. */
      std::list<item_handle*> m_handles;

      /** \brief Next item identifier. */
      static id_type s_next_id;

      /** \brief A list in which we store the reference of the items, so we can
          detect memory leaks. */
      static std::list<base_item*> s_allocated;

    }; // class base_item

  } // namespace engine
} // namespace bear

#endif // __ENGINE_BASE_ITEM_HPP__
