/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file base_item.cpp
 * \brief Implementation of the bear::engine::base_item class.
 * \author Julien Jorge.
 */
#include <algorithm>

#include "engine/base_item.hpp"

#include <claw/logger.hpp>
#include "engine/world.hpp"

/*----------------------------------------------------------------------------*/
bear::engine::base_item::id_type bear::engine::base_item::s_next_id = 1;
std::list<bear::engine::base_item*> bear::engine::base_item::s_allocated;

/*----------------------------------------------------------------------------*/
/**
 * \brief Print the address of the items that were not deleted.
 */
void bear::engine::base_item::print_allocated()
{
  unsigned int s = s_allocated.size();

  if ( s == 0 )
    claw::logger << claw::log_verbose << "All base_item have been deleted."
                 << claw::lendl;
  else
    {
      claw::logger << claw::log_verbose << s
                   << " base_item have NOT been deleted." << claw::lendl;

      std::list<base_item*>::const_iterator it;

      for (it=s_allocated.begin(); it!=s_allocated.end(); ++it)
        {
          std::string str;
          (*it)->to_string( str );

          claw::logger << claw::log_verbose << "-- Item\n" << str
                       << claw::lendl;
        }
    }
} // base_item::print_allocated()

/*----------------------------------------------------------------------------*/
/**
 * \brief Contructor.
 */
bear::engine::base_item::base_item()
  : m_id( s_next_id ), m_owner(NULL)
{
  ++s_next_id;
  s_allocated.push_front(this);
} // base_item::base_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Copy contructor.
 */
bear::engine::base_item::base_item( const base_item& that )
  : m_id( s_next_id ), m_owner(that.m_owner)
{
  ++s_next_id;
  s_allocated.push_front(this);
} // base_item::base_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
bear::engine::base_item::~base_item()
{
  // force a call to remove_handle, that will remove the handle from the list
  while ( !m_handles.empty() )
    *m_handles.front() = NULL;

  s_allocated.erase( std::find(s_allocated.begin(), s_allocated.end(), this) );
} // base_item::~base_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Initialize the item.
 */
void bear::engine::base_item::start()
{
  // nothing to do
} // base_item::start()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the sprite representing the item.
 * \param visuals (out) The sprites of the item, and their positions.
 */
void
bear::engine::base_item::get_visual( std::list<scene_visual>& visuals ) const
{
  // nothing to do
} // base_item::get_visual()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the heiress class is a player.
 */
bool bear::engine::base_item::is_player() const
{
  return false;
} // base_item::is_player()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type \c unsigned \c integer.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool bear::engine::base_item::set_u_integer_field
( const std::string& name, unsigned int value )
{
  claw::logger << claw::log_warning
               << "base_item::set_u_integer_field(): '" << name
               << "' hasn't been set." << claw::lendl;
  
  return false;
} // base_item::set_u_integer_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type \c integer.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool
bear::engine::base_item::set_integer_field( const std::string& name, int value )
{
  claw::logger << claw::log_warning
               << "base_item::set_integer_field(): '" << name
               << "' hasn't been set." << claw::lendl;
  return false;
} // base_item::set_integer_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type \c real.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool
bear::engine::base_item::set_real_field( const std::string& name, double value )
{
  bool ok = true;

  if (name == "pos_x")
    {
      universe::position_type pos(get_position());

      pos.x = value;
      set_position(pos);
    }
  else if (name == "pos_y")
    {
      universe::position_type pos(get_position());

      pos.y = value;
      set_position(pos);
    }
  else if (name == "height")
    set_height(value);
  else if (name == "width")
    set_width(value);
  else if (name == "mass")
    set_mass(value);
  else if (name == "angle")
    set_angle((double)value);    
  else
    {
      claw::logger << claw::log_warning
                   << "base_item::set_real_field(): '" << name
                   << "' hasn't been set." << claw::lendl;
      ok = false;
    }

  return ok;
} // base_item::set_real_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type \c bool.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool
bear::engine::base_item::set_bool_field( const std::string& name, bool value )
{
  if (name == "can_move_items")
    set_can_move_items(value);
  else 
    claw::logger << claw::log_warning
                 << "base_item::set_bool_field(): '" << name
                 << "' hasn't been set." << claw::lendl;

  return false;
} // base_item::set_bool_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type \c string.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool bear::engine::base_item::set_string_field
( const std::string& name, const std::string& value )
{
  claw::logger << claw::log_warning
               << "base_item::set_string_field(): '" << name
               << "' hasn't been set." << claw::lendl;

  return false;
} // base_item::set_string_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type \c base_item.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool bear::engine::base_item::set_item_field
( const std::string& name, base_item* value )
{
  claw::logger << claw::log_warning
               << "base_item::set_string_field(): '" << name
               << "' hasn't been set." << claw::lendl;

  return false;
} // base_item::set_item_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type \c visual::sprite.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool bear::engine::base_item::set_sprite_field
( const std::string& name, visual::sprite* value )
{
  claw::logger << claw::log_warning
               << "base_item::set_sprite_field(): '" << name
               << "' hasn't been set." << claw::lendl;

  return false;
} // base_item::set_sprite_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type \c visual::animation.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool bear::engine::base_item::set_animation_field
( const std::string& name, visual::animation* value )
{
  claw::logger << claw::log_warning
               << "base_item::set_animation_field():  '" << name
               << "' hasn't been set." << claw::lendl;

  return false;
} // base_item::set_animation_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type \c list of unsigned integer.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool bear::engine::base_item::set_u_integer_list_field
( const std::string& name, const std::vector<unsigned int>& value )
{
  claw::logger << claw::log_warning
               << "base_item::set_u_integer_list_field(): '" << name
               << "' hasn't been set." << claw::lendl;
  
  return false;
} // base_item::set_u_integer_list_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type \c list of integer.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool
bear::engine::base_item::set_integer_list_field
( const std::string& name, const std::vector<int>& value )
{
  claw::logger << claw::log_warning
               << "base_item::set_integer_list_field(): '" << name
               << "' hasn't been set." << claw::lendl;
  return false;
} // base_item::set_integer_list_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type \c list of real.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool
bear::engine::base_item::set_real_list_field
( const std::string& name, const std::vector<double>& value )
{
  claw::logger << claw::log_warning
               << "base_item::set_real_list_field(): '" << name
               << "' hasn't been set." << claw::lendl;
  
  return false;
} // base_item::set_real_list_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type \c list of bool.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool
bear::engine::base_item::set_bool_list_field
( const std::string& name, const std::vector<bool>& value )
{
  claw::logger << claw::log_warning
               << "base_item::set_bool_list_field(): '" << name
               << "' hasn't been set." << claw::lendl;

  return false;
} // base_item::set_bool_list_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type \c list of string.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool bear::engine::base_item::set_string_list_field
( const std::string& name, const std::vector<std::string>& value )
{
  claw::logger << claw::log_warning
               << "base_item::set_string_list_field(): '" << name
               << "' hasn't been set." << claw::lendl;

  return false;
} // base_item::set_string_list_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type \c list of item.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool bear::engine::base_item::set_item_list_field
( const std::string& name, const std::vector<base_item*>& value )
{
  claw::logger << claw::log_warning
               << "base_item::set_string_list_field(): '" << name
               << "' hasn't been set." << claw::lendl;

  return false;
} // base_item::set_item_list_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type \c list of visual::sprite.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool bear::engine::base_item::set_sprite_list_field
( const std::string& name, const std::vector<visual::sprite*>& value )
{
  claw::logger << claw::log_warning
               << "base_item::set_sprite_list_field(): '" << name
               << "' hasn't been set." << claw::lendl;

  return false;
} // base_item::set_sprite_list_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type \c list of visual::animation.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool bear::engine::base_item::set_animation_list_field
( const std::string& name, const std::vector<visual::animation*>& value )
{
  claw::logger << claw::log_warning
               << "base_item::set_animation_list_field():  '" << name
               << "' hasn't been set." << claw::lendl;

  return false;
} // base_item::set_animation_list_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the item is correctly initialized.
 */
bool bear::engine::base_item::is_valid() const
{
  return true;
} // base_item::is_valid()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add an handle to the item.
 * \pre (h != NULL)
 *      && (std::find(m_handles.begin(), m_handles.end(), h) == m_handles.end())
 */
void bear::engine::base_item::add_handle( item_handle* h )
{
  CLAW_PRECOND( h != NULL );
  CLAW_PRECOND( std::find(m_handles.begin(), m_handles.end(), h)
                == m_handles.end() );

  m_handles.push_front( h );
} // base_item::add_handle()

/*----------------------------------------------------------------------------*/
/**
 * \brief Remove an handle to the item.
 * \pre (h != NULL)
 *      && (std::find(m_handles.begin(), m_handles.end(), h) != m_handles.end())
 */
void bear::engine::base_item::remove_handle( item_handle* h )
{
  CLAW_PRECOND( h != NULL );
  CLAW_PRECOND( std::find(m_handles.begin(), m_handles.end(), h)
                != m_handles.end() );

  m_handles.erase( std::find( m_handles.begin(), m_handles.end(), h ) );
} // base_item::remove_handle()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get item's identifier.
 */
bear::engine::base_item::id_type bear::engine::base_item::get_id() const
{
  return m_id;
} // base_item::get_id()

/*----------------------------------------------------------------------------*/
/**
 * \brief Kill the item.
 * \pre m_owner != NULL
 */
void bear::engine::base_item::kill()
{
  CLAW_PRECOND( m_owner != NULL );

  m_owner->release_item(this);
} // base_item::kill()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the item in a world.
 * \param w The world in which we are.
 * \pre !m_owner
 */
void bear::engine::base_item::set_owner( world& w )
{
  CLAW_PRECOND( m_owner == NULL );

  m_owner = &w;
} // base_item::set_owner()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the world in which we are.
 * \param w The world in which we are.
 */
bear::engine::world* bear::engine::base_item::get_owner() const
{
  return m_owner;
} // base_item::get_owner()

/*----------------------------------------------------------------------------*/
/**
 * \brief A collision occured, the strongest item has the control.
 * \param that The other item of the collision.
 * \param old_self The state to consider for the current item.
 * \param old_that The state to consider for the item "that".
 */
void bear::engine::base_item::hit
( base_item& that, const physical_item_state& old_self,
  const physical_item_state& old_that )
{
  default_collision_event(that, old_self, old_that);
} // base_item::hit()

/*----------------------------------------------------------------------------*/
/**
 * \brief When a collision between two items occurs, hit() is called from the
 *        strongest item.
 * \param that The other item of the collision.
 * \param old_self The state to consider for the current item.
 * \param old_that The state to consider for the item "that".
 */
void bear::engine::base_item::collision
( physical_item& that, const physical_item_state& old_self,
  const physical_item_state& old_that )
{
  base_item* o = dynamic_cast<base_item*>(&that);

  if (o)
    hit(*o, old_self, old_that);
  else
    claw::logger << claw::log_error
		 << "bear::engine::base_item::collision(): Collision with an "
		 << "item of type different of bear::engine::base_item."
		 << claw::lendl;
} // base_item::collision()

/*----------------------------------------------------------------------------*/
/**
 * \brief Give a string representation of the item.
 * \param str (out) The result of the convertion.
 */
void bear::engine::base_item::to_string( std::string& str ) const
{
  std::ostringstream oss;
  oss << "id: " << m_id << "\n";

  universe::base_entity::to_string(str);

  str = oss.str() + str;
} // base_item::to_string()
