/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file camera.cpp
 * \brief Implementation of the bear::engine::camera class.
 * \author Sebastien Angibaud
 */
#include "engine/camera.hpp"

#include "additional_stream_formatting.hpp"

/*----------------------------------------------------------------------------*/
const std::string bear::engine::camera::s_default_name( "camera" );

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param mode The mode to se the camera to.
 */
bear::engine::camera::set_placement_message::set_placement_message
( placement_mode mode )
  : m_mode(mode)
{

} // set_placement_message::set_placement_message()

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the message to a camera.
 * \param that The camera to apply the message to.
 */
bool bear::engine::camera::set_placement_message::apply_to
( communication::messageable& that )
{
  CLAW_PRECOND( dynamic_cast<camera*>(&that) != NULL );

  camera* cam = (camera*)&that;
  cam->set_mode( m_mode );

  return true;
} // set_placement_message::apply_to()




/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the message to a camera.
 * \param that The camera to apply the message to.
 */
bool bear::engine::camera::msg_get_focus::apply_to
( communication::messageable& that )
{
  CLAW_PRECOND( dynamic_cast<camera*>(&that) != NULL );

  camera* cam = (camera*)&that;
  focus = cam->get_focus();

  return true;
} // msg_get_focus::apply_to()





/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param name The name of the camera.
 * \param size The size of the camera.
 */
bear::engine::camera::camera
( const std::string& name, const claw::math::coordinate_2d<unsigned int>& size,
  unsigned int max_move_length )
  : communication::messageable(name), m_box(0, 0, size.x, size.y),
    m_valid_area(0, 0, size.x, size.y), m_placement(shared),
    m_first_player( player::player_name(1) ),
    m_second_player( player::player_name(2) ),
    m_max_move_length(max_move_length)
{

} // camera::camera()

/*----------------------------------------------------------------------------*/
/**
 * \brief Ajust the position according to the current mode.
 */
void bear::engine::camera::auto_position()
{
  switch(m_placement)
    {
    case lock_first_player : set_first_player(); break;
    case lock_second_player : set_second_player(); break;
    case shared : set_shared(); break;
    case do_nothing : break;
    }
} // camera::auto_position()

/*----------------------------------------------------------------------------*/
/**
 * \brief Force the position of the camera, like a first time.
 */
void bear::engine::camera::initialise_position()
{
  // we allow a movement of any length, this time only
  const universe::coordinate_type saved_max_move = m_max_move_length;
  m_max_move_length = std::numeric_limits<universe::coordinate_type>::max();

  auto_position();

  m_max_move_length = saved_max_move;
} // camera::initialise_position()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the position of the camera.
 * \param position New position of the camera.
 */
void bear::engine::camera::set_position
( const universe::position_type& position )
{
  m_box.position = position;
  stay_valid();
} // camera::set_position()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the area in which the camera can be set.
 * \param area The area.
 */
void bear::engine::camera::set_valid_area
( const universe::rectangle_type& area )
{
  m_valid_area = area;
  stay_valid();
} // camera::set_valid_area()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the focus of the camera.
 */
const bear::universe::rectangle_type& bear::engine::camera::get_focus() const
{
  return m_box;
} // camera::get_focus()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the center of the camera.
 */
bear::universe::position_type bear::engine::camera::get_center() const
{
  return m_box.position + m_box.size() / 2;
} // camera::get_center()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the placement mode.
 * \param mode The new mode.
 */
void bear::engine::camera::set_mode( placement_mode mode )
{
  if ( mode == lock_first_player )
    {
      if ( m_placement == lock_first_player )
        m_placement = shared;
      else
        m_placement = mode;
    }
  else if ( mode == lock_second_player )
    {
      if ( m_placement == lock_second_player )
        m_placement = shared;
      else
        m_placement = mode;
    }
  else
    m_placement = mode;
} // camera::set_mode()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the default name ofthe cameras.
 */
const std::string& bear::engine::camera::default_name()
{
  return s_default_name;
} // camera::default_name()

/*----------------------------------------------------------------------------*/
/**
 * \brief Center the camera on the first player.
 */
void bear::engine::camera::set_first_player()
{
  if ( m_first_player )
    adjust_position( m_first_player->hot_spot() );
  else if ( m_second_player )
    set_second_player();
  else
    m_placement = do_nothing;
} // camera::set_first_player()

/*----------------------------------------------------------------------------*/
/**
 * \brief Center the camera on the second player.
 */
void bear::engine::camera::set_second_player()
{
  if ( m_second_player )
    adjust_position( m_second_player->hot_spot() );
  else if ( m_first_player )
    set_first_player();
  else
    m_placement = do_nothing;
} // camera::set_second_player()

/*----------------------------------------------------------------------------*/
/**
 * \brief Center the camera between the two players.
 */
void bear::engine::camera::set_shared()
{
  if ( m_first_player )
    {
      if ( m_second_player )
        {
          universe::position_type center1;
          universe::position_type center2;
	
          center1 = m_first_player->hot_spot();
          center2 = m_second_player->hot_spot();
	
          adjust_position( (center1 + center2) / 2 );
        }
      else
        set_first_player();
    }
  else if ( m_second_player )
    set_second_player();
  else
    m_placement = do_nothing;
} // camera::set_shared()

/*----------------------------------------------------------------------------*/
/**
 * \brief Adjust the position of the center of the camera.
 * \param center_position New position for the center of the camera.
 */
void bear::engine::camera::adjust_position
( const universe::position_type& center_position )
{
  adjust_position_x(center_position.x);
  adjust_position_y(center_position.y);
} // camera::adjust_position()

/*----------------------------------------------------------------------------*/
/**
 * \brief Adjust the X position of the center of the camera.
 * \param center_position X-coordinate to set to the center of the camera.
 */
void bear::engine::camera::adjust_position_x
( universe::coordinate_type center_position )
{
  const universe::coordinate_type current_x =
    m_box.position.x + m_box.width / 2;
  universe::coordinate_type distance;

  if ( center_position < current_x )
    {
      distance = std::min(current_x - center_position, m_max_move_length);

      if ( m_box.position.x - m_valid_area.position.x >= distance )
        m_box.position.x -= distance;
      else
        m_box.position.x = m_valid_area.position.x;
    }
  else if ( center_position > current_x )
    {
      distance = std::min(center_position - current_x, m_max_move_length);

      if ( m_box.right() + distance <= m_valid_area.right() )
        m_box.position.x += distance;
      else
        m_box.position.x = m_valid_area.right() - m_box.width;
    }
} // camera::adjust_position_x()

/*----------------------------------------------------------------------------*/
/**
 * \brief Adjust the Y position of the center of the camera.
 * \param center_position Y-coordinate to set to the center of the camera.
 */
void bear::engine::camera::adjust_position_y
( universe::coordinate_type center_position )
{
  const universe::coordinate_type current_y =
    m_box.position.y + m_box.height / 2;

  universe::coordinate_type distance;

    if ( center_position < current_y )
    {
      distance = std::min(current_y - center_position, m_max_move_length);

      if ( m_box.position.y - m_valid_area.position.y >= distance )
        m_box.position.y -= distance;
      else
        m_box.position.y = m_valid_area.position.y;
    }
  else if ( center_position > current_y )
    {
      distance = std::min(center_position - current_y, m_max_move_length);

      if ( m_box.bottom() + distance <= m_valid_area.bottom() )
        m_box.position.y += distance;
      else
        m_box.position.y = m_valid_area.bottom() - m_box.height;
    }
} // camera::adjust_position_y()

/*----------------------------------------------------------------------------*/
/**
 * \brief Check if the camera is in its valid area and adjust its position if
 *        needed.
 */
void bear::engine::camera::stay_valid()
{
  if ( m_box.position.x < m_valid_area.position.x )
    m_box.position.x = m_valid_area.position.x;

  if ( m_box.position.y < m_valid_area.position.y )
    m_box.position.y = m_valid_area.position.y;

  if ( m_box.right() > m_valid_area.right() )
    {
      if ( m_valid_area.right() > m_box.width )
        m_box.position.x = m_valid_area.right() - m_box.width;
      else
        m_box.position.x = 0;
    }

  if ( m_box.bottom() > m_valid_area.bottom() )
    {
      if ( m_valid_area.bottom() > m_box.height )
        m_box.position.y = m_valid_area.bottom() - m_box.height;
      else
        m_box.position.y = 0;
    }
} // camera::stay_valid()

