/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file font_factory.cpp
 * \brief Implementation of the bear::engine::font_factory class.
 * \author Julien Jorge
 */
#include "engine/font_factory.hpp"
#include "engine/game.hpp"

#include <sstream>
#include <claw/exception.hpp>

/*----------------------------------------------------------------------------*/
/**
 * \brief Instanciate a font, given an image resource.
 * \param name The name of the image resource.
 *
 * The type of the created font is determined by its name. If \a font_name ends
 * by '.ttf' (not yet supported), the returned font will be the corresponding
 * True Type font (found in the resources). Otherwise \a font_name (in the image
 * resources) must looks like 'anything_WxH', where \e anything is any string of
 * text, then follows an underscore '_'. \e W is the width of the characters
 * and \e H is their height. \e W and \e H are separated by an 'x'.
 *
 * \todo Implement the support of true type fonts.
 */
bear::engine::font_factory::font_type
bear::engine::font_factory::create( const std::string& font_name )
{
  bool valid = false;
  const visual::image& img =
    game::get_instance().current_level_globals().get_image(font_name);

  std::string::size_type pos = font_name.find_last_of('-');
  claw::math::coordinate_2d<unsigned int> character_size;

  if (pos != std::string::npos)
    {
      std::string s = font_name.substr(pos+1);
      pos = s.find_first_of('x');

      if (pos != std::string::npos)
        {
          s[pos] = ' ';
          std::istringstream iss(s);

          if (iss >> character_size.x >> character_size.y)
            valid = true;
        }
    }

  if (valid)
    return font_type( new text::font( img, character_size ) );
  else
    throw CLAW_EXCEPTION( "Can't find font '" + font_name + "'" );
} // font_factory::create()
