/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file game.cpp
 * \brief Implementation of the bear::engine::game class.
 * \author Julien Jorge
 */
#include "engine/game.hpp"

#include <boost/filesystem/convenience.hpp>
#include <claw/exception.hpp>
#include <claw/logger.hpp>
#include <claw/string_algorithm.hpp>
#include <claw/system_info.hpp>
#include <sstream>
#include "input/system.hpp"
#include "engine/controller_layout.hpp"
#include "engine/input_layout.hpp"
#include "engine/string_base.hpp"
#include "engine/resource_pool.hpp"
#include "visual/grey_effect.hpp"

/*----------------------------------------------------------------------------*/
bear::engine::game* bear::engine::game::s_instance(NULL);
std::string bear::engine::game::s_init_game_function_prefix("init_");
std::string bear::engine::game::s_end_game_function_prefix("end_");

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the instance of the game.
 * \pre s_instance != NULL
 */
bear::engine::game& bear::engine::game::get_instance()
{
  CLAW_PRECOND( s_instance != NULL );

  return *s_instance;
} // game::get_instance()

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param f The file containing the game description.
 * \pre f.is_open()
 */
bear::engine::game::game( std::ifstream& f )
  : m_status(status_init), m_game_description(f), m_game_rules
  ( get_custom_game_file(m_game_description.base_layout_file_name()) ),
    m_screen(NULL), m_fullscreen(false), m_current_level(NULL),
    m_level_in_abeyance(NULL), m_input_layout(NULL), m_gui(NULL),
    m_time_step(15), m_sum_of_render_time(0), m_sum_of_progress_time(0),
    m_count_of_render(0), m_count_of_progress(0), m_date_of_last_statistics(0)
{
  CLAW_PRECOND( s_instance == NULL );

  init_environment();

  try
    {
      load_libraries();

      init_resource_pool();
      //apply_rules();

      m_screen = new visual::screen( m_game_description.camera_size(),
                                     m_game_description.game_name(),
                                     m_fullscreen );

      std::stringstream language_file;
      resource_pool::get_instance().get_file
        ( m_game_description.language_file(), language_file );
      string_base::get_instance().load( language_file );
    }
  catch(...)
    {
      clear();
      close_environment();
      throw;
    }

  s_instance = this;
} // game::game()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
bear::engine::game::~game()
{
  std::list<unsigned int>::const_iterator it_p, it_r;
  std::ofstream f("stats.dat");
  unsigned int i=0;

  f << "#it\tprogress\trender" << std::endl;

  for ( it_p=m_progress_time.begin(), it_r=m_render_time.begin();
        it_p!=m_progress_time.end(); ++it_p, ++it_r, ++i )
    f << i << '\t' << *it_p << '\t' << *it_r << std::endl;

  clear();
  close_environment();

  base_item::print_allocated();
} // game::~game()

/*----------------------------------------------------------------------------*/
/**
 * \brief Run the game.
 */
void bear::engine::game::run()
{
  init_game();

  load_level( m_game_description.start_level() );

  run_level();

  end_game();

  clear();
  close_environment();
} // game::run()

/*----------------------------------------------------------------------------*/
/**
 * \brief Toggle the pause status.
 * \param b Tell if we must activate the pause mode.
 */
void bear::engine::game::set_pause( bool b )
{
  CLAW_PRECOND( m_current_level );
  CLAW_PRECOND( (m_status == status_pause) || (m_status == status_run) );

  if (b)
    m_status = status_pause;
  else
    m_status = status_run;
} // game::set_pause()

/*----------------------------------------------------------------------------*/
/**
 * \brief Turn on/off the full screen mode.
 * \param b Tell if we must activate the fullscreen mode.
 */
void bear::engine::game::set_fullscreen( bool full )
{
  if ( m_fullscreen != full )
    {
      m_fullscreen = full;
      m_screen->fullscreen(m_fullscreen);
    }
} // game::set_fullscreen()

/*----------------------------------------------------------------------------*/
/**
 * \brief Return the full screen mode.
 */
bool bear::engine::game::get_fullscreen() const
{
  return m_fullscreen;
} // game::get_fullscreen()

/*----------------------------------------------------------------------------*/
/**
 * \brief Toggle the status of the fullscreen mode.
 */
void bear::engine::game::toggle_fullscreen()
{
  set_fullscreen( !m_fullscreen );
} // game::toggle_fullscreen()

/*----------------------------------------------------------------------------*/
/**
 * \brief Toggle the status of the fullscreen mode.
 */
void bear::engine::game::toggle_time_step()
{
  if ( m_time_step == 0 )
    m_time_step = 15;
  else
    m_time_step = 0;
} // game::toggle_time_step()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the current time step.
 */
unsigned int bear::engine::game::get_time_step() const
{
  return m_time_step;
} // game::get_time_step()

/*----------------------------------------------------------------------------*/
/**
 * \brief Take a shot of the screen.
 * \param img The image in which we save the screen.
 */
void bear::engine::game::screenshot( claw::graphic::image& img ) const
{
  m_screen->shot( img );
} // game::screenshot()

/*----------------------------------------------------------------------------*/
/**
 * \brief Take a shot of the whole level.
 * \param img The image in which we save the picture.
 * \param unsigned int z The ratio applied to the level to fit in the picture.
 */
void bear::engine::game::levelshot
( claw::graphic::image& img, unsigned int r ) const
{
  if ( r > 100 )
    r = 100;
  else if ( r == 0 )
    r = 1;

  img.set_size( (unsigned int)m_current_level->get_size().x * r / 100,
                (unsigned int)m_current_level->get_size().y * r / 100 );

  universe::rectangle_type rect
    ( 0, 0, get_screen_size().x * 100 / r, get_screen_size().y * 100 / r);
  claw::math::coordinate_2d<int> pos;
  claw::graphic::image shot( get_screen_size().x, get_screen_size().y );

  for ( rect.position.y = 0, pos.y = 0;
        rect.position.y < m_current_level->get_size().y;
        rect.position.y += rect.height, pos.y += shot.height() )
    for ( rect.position.x = 0, pos.x = 0;
          rect.position.x < m_current_level->get_size().x;
          rect.position.x += rect.width, pos.x += shot.width() )
      {
        m_screen->begin_render();
        m_current_level->shot( *m_screen, rect );
        m_screen->end_render();

        screenshot(shot);

        img.partial_copy( shot, pos );
      }
} // game::levelshot()

/*----------------------------------------------------------------------------*/
/**
 * \brief End the game.
 */
void bear::engine::game::end()
{
  m_status = status_quit;
} // game::end()

/*----------------------------------------------------------------------------*/
/**
 * \brief Give the name of a level to load as soon as possible.
 * \param level_name The name of the level to load.
 */
void bear::engine::game::set_waiting_level( const std::string& level_name )
{
  m_post_actions.push( new game_action_load_level(level_name) );
} // game::set_waiting_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the level to run as soon as possible.
 * \param the_level The level to run as soon as possible.
 */
void bear::engine::game::set_waiting_level( level* the_level )
{
  CLAW_PRECOND( the_level != NULL );

  m_post_actions.push( new game_action_set_current_level(the_level) );
} // game::set_waiting_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the size of the screen.
 */
const claw::math::coordinate_2d<unsigned int>&
bear::engine::game::get_screen_size() const
{
  return m_game_description.camera_size();
} // game::get_screen_size()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the ratio of the active area.
 */
double bear::engine::game::get_active_area_ratio() const
{
  return m_game_description.active_area_ratio();
} // game::get_active_area_ratio()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the path of a file in the user's game directory.
 * \param name The name of the file.
 */
std::string
bear::engine::game::get_custom_game_file( const std::string& name ) const
{
  std::string result = get_game_directory();

  if ( !result.empty() )
    {
      boost::filesystem::path path( result, boost::filesystem::native );
      path /= name;
      result = path.string();
    }
  else
    result = name;

  return result;
} // game::get_full_user_path()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the rules for a single player game.
 */
void bear::engine::game::one_player_game()
{
  m_game_rules.set_mode( game_rules::local );
  m_game_rules.set_number_of_players( 1 );
} // game::one_player_game()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the rules for a two players local game.
 */
void bear::engine::game::two_players_local_game()
{
  m_game_rules.set_mode( game_rules::local );
  m_game_rules.set_number_of_players( 2 );
} // game::two_players_local_game()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the rules of the game.
 */
bear::engine::game_rules& bear::engine::game::get_rules()
{
  return m_game_rules;
} // game::get_rules()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the rules of the game.
 */
const bear::engine::game_rules& bear::engine::game::get_const_rules() const
{
  return m_game_rules;
} // game::get_rules()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the path of the file of a given level.
 * \param level_name The name of the level for which we want the file.
 */
const std::string&
bear::engine::game::get_level_file( const std::string& level_name ) const
{
  CLAW_PRECOND( level_exists(level_name) );

  return m_game_description.level_files().find(level_name)->second;
} // game::get_level_file()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if there is a level with a given name.
 * \param level_name The name of the level to check.
 */
bool bear::engine::game::level_exists( const std::string& level_name ) const
{
  bool result = false;

  if ( m_game_description.level_files().find(level_name)
       != m_game_description.level_files().end() )
    {
      const std::string level_path =
        m_game_description.level_files().find(level_name)->second;

      if ( resource_pool::get_instance().exists( level_path ) )
        {
          std::stringstream f;
          resource_pool::get_instance().get_file( level_path, f );

          result = !!f;
        }
    }

  return result;
} // game::level_exists()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the current level.
 */
const bear::engine::level& bear::engine::game::get_current_level() const
{
  CLAW_PRECOND( m_current_level != NULL );
  return *m_current_level;
} // game::get_current_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the symbols of the libraries linked to the game.
 */
bear::engine::libraries_pool& bear::engine::game::get_symbols()
{
  return m_symbols;
} // game::get_symbols()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the name of the game.
 */
const std::string& bear::engine::game::get_name() const
{
  return m_game_description.game_name();
} // game::get_name()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the global resources of the current level.
 */
bear::engine::level_globals& bear::engine::game::current_level_globals()
{
  CLAW_PRECOND( m_current_level != NULL );

  return m_current_level->get_globals();
} // game::current_level_globals()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the layer to use to display the interface.
 * \param the_layer The layer to set.
 *
 * \a the_layer will is deleted if the game instance is destroyed or a new gui
 * layer is set (even if the new \a the_layer == NULL).
 */
void bear::engine::game::set_gui_layer( gui_layer* the_layer )
{
  if (m_gui != NULL)
    delete m_gui;

  m_gui = the_layer;

  if (m_gui != NULL)
    m_gui->start();
} // game::set_gui_layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Call the game specific function for initialisation.
 */
void bear::engine::game::init_game() const
{
  std::string game_proc
    ( s_init_game_function_prefix + get_game_name_as_filename() );

  claw::logger << claw::log_verbose << "Initialising game: '" << game_proc
               << "()'" << claw::lendl;

  if ( m_symbols.have_symbol( game_proc ) )
    {
      init_game_function_type func =
        m_symbols.get_symbol<init_game_function_type>( game_proc );
      func();
    }
} // game::init_game()

/*----------------------------------------------------------------------------*/
/**
 * \brief Call the game specific function for ending the game.
 */
void bear::engine::game::end_game() const
{
  std::string game_proc
    ( s_end_game_function_prefix + get_game_name_as_filename() );

  claw::logger << claw::log_verbose << "Ending game: '" << game_proc
               << "()'" << claw::lendl;

  if ( m_symbols.have_symbol( game_proc ) )
    {
      end_game_function_type func =
        m_symbols.get_symbol<end_game_function_type>( game_proc );
      func();
    }
} // game::end_game()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the name of the game in a string usable for a file name.
 */
std::string bear::engine::game::get_game_name_as_filename() const
{
  std::string result( m_game_description.game_name() );

  std::transform( result.begin(), result.end(), result.begin(), tolower );

  for (unsigned int i=0; i!=result.size(); ++i)
    if (result[i] == ' ')
      result[i] = '_';
    else if (result[i] == '\t')
      result[i] = '_';

  claw::text::squeeze( result, "_" );

  return result;
} // game::get_game_name_as_filename()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get, and create, the personal game directory of the user.
 */
std::string bear::engine::game::get_game_directory() const
{
  boost::filesystem::path dir
    (claw::system_info::get_user_directory(), boost::filesystem::native);

  std::string result;
  std::string subdir = '.' + get_game_name_as_filename();

  dir /= boost::filesystem::path(subdir, boost::filesystem::native);

  if ( create_game_directory(dir.string()) )
    result = dir.string();
  else
    claw::logger << claw::log_error << "Can't create game directory '"
                 << dir.string() << "'." << claw::lendl;

  return result;
} // game::get_game_directory()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the personal game directory of the user.
 * \param dir The path of the directory.
 */
bool bear::engine::game::create_game_directory( const std::string& dir ) const
{
  bool result = false;
  boost::filesystem::path path( dir, boost::filesystem::native );

  if ( boost::filesystem::exists( path ) )
    result = boost::filesystem::is_directory( path );
  else
    result = boost::filesystem::create_directory( path );

  return result;
} // game::create_game_directory()

/*----------------------------------------------------------------------------*/
/**
 * \brief Run the current_level.
 */
void bear::engine::game::run_level()
{
  m_status = status_run;

  while (m_status != status_quit)
    {
      m_last_progress.set();

      do
        {
          one_step_beyond();
        }
      while ( !do_post_actions() && (m_status != status_quit) );
    }
} // game::run_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Do one progress()/render() iteration.
 */
void bear::engine::game::one_step_beyond()
{
  time_ref::time_reference current_time;

  current_time.set();

  // one progress each 15 ms
  if ( current_time > m_last_progress + m_time_step )
    {
      universe::time_type dt( current_time - m_last_progress );
      m_last_progress = current_time;
      progress( dt / 1000 ); // seconds 
      render();

#ifndef NDEBUG
      print_statistics();
#endif // NDEBUG
    }
} // game::one_step_beyond()

/*----------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the progression of the game.
 * \param elapsed_time Elapsed time since the last call.
 */
void bear::engine::game::progress( universe::time_type elapsed_time )
{
  // statistics
  time_ref::time_reference t;
  t.set();
  const unsigned int start_time = t;

  // effective procedure
  update_inputs();

  if ( m_status != status_pause )
    m_current_level->progress( elapsed_time );

  if (m_gui != NULL)
    m_gui->progress( elapsed_time );

  // statistics
  t.set();
  m_progress_time.push_back( t - start_time );
  m_sum_of_progress_time += t - start_time;
  ++m_count_of_progress;
} // game::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render the current level.
 */
void bear::engine::game::render()
{
  // statistics
  time_ref::time_reference t;
  t.set();
  const unsigned int start_time = t;

  // effective procedure
  m_screen->begin_render();

  m_current_level->render( *m_screen );

  if (m_gui != NULL)
    m_gui->render( *m_screen );

  m_screen->end_render();

  // statistics
  t.set();
  m_render_time.push_back( t - start_time );
  m_sum_of_render_time += t - start_time;
  ++m_count_of_render;
} // game::render()

/*----------------------------------------------------------------------------*/
/**
 * \brief Process a key pressure information.
 * \param key The code of the pressed key.
 */
bool bear::engine::game::key_pressed( input::keyboard::key_code key )
{
  bool result = false;

  if (m_gui != NULL)
    result = m_gui->key_pressed( key );

  return result;
} // game::key_pressed()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell that a joystick button has been pressed.
 * \param button The code of the button.
 * \param joy_index The index of the joytick.
 */
bool bear::engine::game::button_pressed
( input::joystick::joy_code button, unsigned int joy_index )

{
  if (m_gui != NULL)
    return m_gui->button_pressed( button, joy_index );
  else
    return false;
} // game::button_pressed()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell that a mouse button has been pressed.
 * \param pos The current position of the cursor.
 * \param key The code of the pressed button.
 */
bool bear::engine::game::mouse_pressed
( input::mouse::mouse_code key,
  const claw::math::coordinate_2d<unsigned int>& pos )
{
  if (m_gui != NULL)
    return m_gui->mouse_pressed( key, pos );
  else
    return false;
} // game::mouse_pressed()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell that a mouse button is maintained.
 * \param key The code of the maintained button.
 * \param pos The current position of the cursor.
 */
bool bear::engine::game::mouse_maintained
( input::mouse::mouse_code key,
  const claw::math::coordinate_2d<unsigned int>& pos )
{
  if (m_gui != NULL)
    return m_gui->mouse_maintained( key, pos );
  else
    return false;
} // game::mouse_maintained()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell that a mouse button has been released.
 * \param key The code of the released button.
 * \param pos The current position of the cursor.
 */
bool bear::engine::game::mouse_released
( input::mouse::mouse_code key,
  const claw::math::coordinate_2d<unsigned int>& pos )
{
  if (m_gui != NULL)
    return m_gui->mouse_released( key, pos );
  else
    return false;
} // game::mouse_released()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell that the mouse has been moved.
 * \param pos The new position of the cursor.
 */
bool bear::engine::game::mouse_move
( const claw::math::coordinate_2d<unsigned int>& pos )

{
  if (m_gui != NULL)
    return m_gui->mouse_move( pos );
  else
    return false;
} // game::mouse_move()

/*----------------------------------------------------------------------------*/
/**
 * \brief Read inputs and apply them to the players.
 */
void bear::engine::game::update_inputs()
{
  CLAW_PRECOND( m_input_layout != NULL );

  input::system::get_instance().refresh();

  // check all local and distant inputs
  if ( m_status != status_pause )
    m_input_layout->update();

  // then we check if we come in pause mode or any particular things
  m_input_status.read();
  m_input_status.scan_inputs( *this );

} // game::update_inputs()

/*----------------------------------------------------------------------------*/
/**
 * \brief Initialize the environment (screen, inputs, sounds).
 */
void bear::engine::game::init_environment() const
{
  claw::logger << claw::log_verbose << "Initializing screen environment."
               << claw::lendl;

  visual::screen::initialize();

  claw::logger << claw::log_verbose << "Initializing input environment."
               << claw::lendl;

  input::system::initialize();

  claw::logger << claw::log_verbose << input::joystick::number_of_joysticks()
               << " joysticks found." << claw::lendl;

  claw::logger << claw::log_verbose << "Initializing sound environment."
               << claw::lendl;

  audio::sound_manager::initialize();
} // game::init_environment()

/*----------------------------------------------------------------------------*/
/**
 * \brief Close the environment (screen, inputs, sounds).
 */
void bear::engine::game::close_environment() const
{
  claw::logger << claw::log_verbose << "Closing screen environment."
               << claw::lendl;

  visual::screen::release();

  claw::logger << claw::log_verbose << "Closing input environment."
               << claw::lendl;

  input::system::release();

  claw::logger << claw::log_verbose << "Closing sound environment."
               << claw::lendl;

  audio::sound_manager::release();
} // game::close_environment()

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the settings of the rules.
 */
void bear::engine::game::apply_rules()
{

} // game::apply_rules()

/*----------------------------------------------------------------------------*/
/**
 * \brief Load the libraries given in the game description file.
 */
void bear::engine::game::load_libraries()
{
  game_description::string_list::const_iterator it;

  for ( it=m_game_description.libraries().begin();
        it!=m_game_description.libraries().end(); ++it )
    {
      claw::logger << claw::log_verbose << "Add library '" << *it << "'."
                   << claw::lendl;
      m_symbols.add_library(*it);
    }
} // game::load_libraries()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add paths in the resource_pool.
 */
void bear::engine::game::init_resource_pool() const
{
  std::list<std::string>::const_iterator it;

  for ( it = m_game_description.resources_path().begin();
        it != m_game_description.resources_path().end();
        ++it )
    {
      claw::logger << claw::log_verbose
                   << "Adding resource path '" << *it << "'."
                   << claw::lendl;
      resource_pool::get_instance().add_path(*it);
    }
} // game::init_resource_pool()

/*----------------------------------------------------------------------------*/
/**
 * \brief Do the pending actions.
 */
bool bear::engine::game::do_post_actions()
{
  bool result = false;

  while ( !m_post_actions.empty() )
    {
      if ( m_post_actions.front()->apply(*this) )
        result = true;

      delete m_post_actions.front();
      m_post_actions.pop();
    }

  return result;
} // game::do_post_actions()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the current level (and delete the level currently running).
 * \param the_level The new current level.
 */
void bear::engine::game::set_current_level( level* the_level )
{
  CLAW_PRECOND( the_level != NULL );

  if ( m_current_level != NULL )
    close_level();

  m_current_level = the_level;

  start_current_level();
} // game::set_current_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Load a level.
 * \param name The name of the level to load.
 */
void bear::engine::game::load_level( const std::string& name )
{
  claw::logger << claw::log_verbose << "------------ Loading level '"
               << name << "'... ------------" << claw::lendl;

  if ( !level_exists(name) )
    throw CLAW_EXCEPTION
      ( "Can't open level file '" + get_level_file(name) + "'." );
  else
    {
      std::stringstream f;
      resource_pool::get_instance().get_file( get_level_file(name), f );

      /// \todo test the file to see if it's text or binary
      compiled_file level_file( f, true );

      if ( m_current_level != NULL )
        close_level();

      m_current_level = new level( level_file, name );

      start_current_level();
    }
} // game::load_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Close the current level.
 * \pre m_current_level != NULL
 * \post m_current_level == NULL
 */
void bear::engine::game::close_level()
{
  CLAW_PRECOND( m_current_level != NULL );

  // The input layout keeps a pointer to the player, but the player is local to
  // the level. We must re-initialise it.
  if ( m_input_layout != NULL )
    {
      delete m_input_layout;
      m_input_layout = NULL;
    }

  if (m_gui != NULL)
    {
      delete m_gui;
      m_gui = NULL;
    }

  delete m_current_level;
  m_current_level = NULL;

  CLAW_POSTCOND( m_current_level == NULL );
} // game::close_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a level on the stack.
 * \param name The name of the level to load.
 * \todo Save the GUI layer too.
 */
void bear::engine::game::push_level( const std::string& name )
{
  CLAW_PRECOND( m_level_in_abeyance == NULL );

  m_level_in_abeyance = m_current_level;
  m_level_in_abeyance->stop();
  m_current_level = NULL;
  load_level( name );
} // game::push_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Restore the level on the top of the stack.
 */
void bear::engine::game::pop_level()
{
  CLAW_PRECOND( m_level_in_abeyance != NULL );
  CLAW_PRECOND( m_current_level != NULL );

  // The input layout keeps a pointer to the player, but the player is local to
  // the level. We must re-initialise it.
  if ( m_input_layout != NULL )
    {
      delete m_input_layout;
      m_input_layout = NULL;
    }

  close_level();

  m_current_level = m_level_in_abeyance;
  m_level_in_abeyance = NULL;

  // The input layout keeps a pointer to the player, but the player is local to
  // the level. We must re-initialise it.
  m_input_layout = new input_layout( m_game_rules );
} // game::pop_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Start the current level.
 */
void bear::engine::game::start_current_level()
{
  claw::logger << claw::log_verbose
               << "------------ Starting level. ------------" << claw::lendl;

  CLAW_PRECOND( m_current_level != NULL );
  CLAW_PRECOND( m_input_layout == NULL );

  // The input layout keeps a pointer to the player, but the player is local to
  // the level. We must re-initialise it.
  m_input_layout = new input_layout( m_game_rules );

  m_current_level->start();
} // game::start_current_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Remove all layers, controllers and the screen.
 */
void bear::engine::game::clear()
{
  if (m_gui != NULL)
    {
      delete m_gui;
      m_gui = NULL;
    }

  if (m_input_layout != NULL)
    {
      delete m_input_layout;
      m_input_layout = NULL;
    }

  if ( m_current_level != NULL )
    {
      if ( m_level_in_abeyance )
        pop_level();

      close_level();
    }

  if (m_screen != NULL)
    {
      delete m_screen;
      m_screen = NULL;
    }

  // deleting pending actions
  while ( !m_post_actions.empty() )
    {
      delete m_post_actions.front();
      m_post_actions.pop();
    }
} // game::clear()

/*----------------------------------------------------------------------------*/
/**
 * \brief Print some statistics
 */
void bear::engine::game::print_statistics()
{
  time_ref::time_reference current_time;

  current_time.set();

  if ( current_time - m_date_of_last_statistics > 5000 )
    {
      if ( m_count_of_render == 0 )
        claw::logger << claw::log_verbose << "No call to render()";
      else
        claw::logger << claw::log_verbose
                     << m_sum_of_render_time / m_count_of_render
                     << " ms. per call to render()";

      claw::logger << claw::lendl;

      if ( m_count_of_progress == 0 )
        claw::logger << claw::log_verbose << "No call to progress()";
      else
        claw::logger << claw::log_verbose
                     << m_sum_of_progress_time / m_count_of_progress
                     << " ms. per call to progress()";

      claw::logger << claw::lendl;

      m_date_of_last_statistics = current_time;
      m_sum_of_progress_time = 0;
      m_sum_of_render_time = 0;
      m_count_of_progress = 0;
      m_count_of_render = 0;
    }
} // game::print_statistics()
