/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file pointer_to_player.cpp
 * \brief Implementation of the bear::engine::pointer_to_player class.
 * \author Julien Jorge
 */
#include "engine/pointer_to_player.hpp"

#include "engine/game.hpp"

#include <claw/assert.hpp>

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param player_name The name of the player we are pointing to.
 */
bear::engine::pointer_to_player::pointer_to_player
( const std::string& player_name )
  : m_player_name(player_name)
{
  find_player_instance();
} // pointer_to_player::pointer_to_player()

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param item The player we are pointing to.
 */
bear::engine::pointer_to_player::pointer_to_player( player* item )
{
  if (item)
    {
      m_player = item;
      m_player_name = item->get_name();
    }
} // pointer_to_player::pointer_to_player()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the handle point to a valid entity.
 */
bool bear::engine::pointer_to_player::is_valid()
{
  return get() != NULL;
} // pointer_to_player::is_valid()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the pointed entity.
 * \return NULL if there's no player in memory with the wanted index.
 */
bear::engine::player* bear::engine::pointer_to_player::get()
{
  if ( !m_player.is_valid() )
    find_player_instance();

  return static_cast<player*>(m_player.get());
} // pointer_to_player::get()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the pointer point to a valid player.
 * \return is_valid()
 */
bear::engine::pointer_to_player::operator bool()
{
  return is_valid();
} // pointer_to_player::operator bool()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the pointer point to a valid player.
 * \return is_valid()
 */
bear::engine::pointer_to_player::operator bool() const
{
  return const_cast<pointer_to_player*>(this)->is_valid();
} // pointer_to_player::operator bool()

/*----------------------------------------------------------------------------*/
/**
 * \brief  Cast to pointer type.
 * \return get()
 */
bear::engine::pointer_to_player::operator bear::engine::player*()
{
  CLAW_PRECOND( is_valid() );
  return  get();
} // pointer_to_player::operator bear::engine::player*()

/*----------------------------------------------------------------------------*/
/**
 * \brief Pointer-to-member operator.
 * \return get()
 */
bear::engine::player* bear::engine::pointer_to_player::operator->()
{
  CLAW_PRECOND( is_valid() );
  return get();
} // pointer_to_player::operator->()

/*----------------------------------------------------------------------------*/
/**
 * \brief Pointer-to-member operator.
 * \return get()
 */
const bear::engine::player* bear::engine::pointer_to_player::operator->() const
{
  return const_cast<pointer_to_player*>(this)->operator->();
} // pointer_to_player::operator->()

/*----------------------------------------------------------------------------*/
/**
 * \brief Assignement operator.
 * \param item The player to assign.
 */
bear::engine::pointer_to_player& bear::engine::pointer_to_player::operator=
( player* item )
{
  if (!item)
    m_player_name.clear();
  else
    m_player_name = item->get_name();

  m_player = item;

  return *this;
} // pointer_to_player::operator=()

/*----------------------------------------------------------------------------*/
/**
 * \brief Find the instance of the player.
 */
void bear::engine::pointer_to_player::find_player_instance()
{
  player::get_instance_message msg;

  if ( !m_player_name.empty() )
    game::get_instance().current_level_globals().send_message
      ( m_player_name, msg );

  m_player = msg.get_instance();
} // pointer_to_player::find_player_instance()

