/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file game.hpp
 * \brief The class managing the levels and the development of the game.
 * \author Julien Jorge
 */
#ifndef __ENGINE_GAME_HPP__
#define __ENGINE_GAME_HPP__

#include <fstream>
#include <queue>

#include "communication/messageable.hpp"
#include "input/input_status.hpp"
#include "engine/class_export.hpp"
#include "engine/game_description.hpp"
#include "engine/game_rules.hpp"
#include "engine/level_globals.hpp"
#include "engine/libraries_pool.hpp"
#include "engine/game_action/game_action.hpp"
#include "engine/game_action/game_action_apply_rules.hpp"
#include "engine/game_action/game_action_load_level.hpp"
#include "engine/game_action/game_action_set_current_level.hpp"
#include "engine/layer/gui_layer.hpp"
#include "time/time_reference.hpp"
#include "visual/screen.hpp"

namespace bear
{
  namespace engine
  {
    class input_layout;
    class level;
    
    /**
     * \brief The class managing the levels and the development of the game.
     * \author Julien Jorge
     */
    class ENGINE_EXPORT game:
      public input::input_listener
    {
      friend class game_action;
      friend class game_action_load_level;
      friend class game_action_apply_rules;
      friend class game_action_set_current_level;

    private:
      /** \brief Type of the game specific initialisation procedure. */
      typedef void (*init_game_function_type)();

      /** \brief Type of the game specific ending procedure. */
      typedef void (*end_game_function_type)();

      /**
       * \brief Game status.
       */
      enum status
        {
          /** \brief The game is under initialization. */
          status_init,

          /** \brief The game is running. */
          status_run,

          /** \brief The game is paused. */
          status_pause,

          /** \brief We're quiting. */
          status_quit

        }; // enum status

    public:
      static game& get_instance();

      game( std::ifstream& f );
      ~game();

      void show_fps();

      void run();

      void set_pause( bool b );
      void set_fullscreen( bool full );
      bool get_fullscreen() const;
      void toggle_fullscreen();
      void toggle_time_step();
      unsigned int get_time_step() const;

      void screenshot( claw::graphic::image& img ) const;
      void levelshot( claw::graphic::image& img, unsigned int r ) const;

      void end();
      void set_waiting_level( const std::string& level_name );
      void set_waiting_level( level* the_level );

      const claw::math::coordinate_2d<unsigned int>& get_screen_size() const;
      double get_active_area_ratio() const;
      std::string get_custom_game_file( const std::string& name ) const;

      void one_player_game();
      void two_players_local_game();

      game_rules& get_rules();
      const game_rules& get_const_rules() const;

      const std::string& get_level_file( const std::string& level_name ) const;
      bool level_exists( const std::string& level_name ) const;

      const level& get_current_level() const;
      const std::string& get_name() const;

      level_globals& current_level_globals();

      libraries_pool& get_symbols();

      void set_gui_layer( gui_layer* the_layer );

    private:
      void init_game() const;
      void end_game() const;

      std::string get_game_name_as_filename() const;

      std::string get_game_directory() const;
      bool create_game_directory( const std::string& dir ) const;

      void run_level();
      void one_step_beyond();

      void progress( universe::time_type elapsed_time );
      void render();

      bool key_pressed( input::keyboard::key_code key );
      bool button_pressed( input::joystick::joy_code button,
                           unsigned int joy_index );
      bool mouse_pressed( input::mouse::mouse_code key,
                          const claw::math::coordinate_2d<unsigned int>& pos );
      bool mouse_released( input::mouse::mouse_code button,
                           const claw::math::coordinate_2d<unsigned int>& pos );
      bool mouse_maintained
      ( input::mouse::mouse_code button,
        const claw::math::coordinate_2d<unsigned int>& pos );
      bool mouse_move( const claw::math::coordinate_2d<unsigned int>& pos );

      void update_inputs();

      void init_environment() const;
      void close_environment() const;

      void apply_rules();
      void load_libraries();

      void init_resource_pool() const;

      bool do_post_actions();

      void set_current_level( level* the_level );
      void load_level( const std::string& name );
      void close_level();
      void push_level( const std::string& name );
      void pop_level();

      void start_current_level();

      void clear();

      void print_statistics();

    protected:
      /** \brief The instance of the game. */
      static game* s_instance;

    private:
      // must be declared before m_game_description
      /** \brief The libraries in which we take custom functions. */
      libraries_pool m_symbols;

      /** \brief The current status of the game. */
      status m_status;

      /** \brief Description of the game. */
      game_description m_game_description;

      /** \brief The parameters of the current game. */
      game_rules m_game_rules;

      /** \brief The screen. */
      visual::screen* m_screen;

      /** \brief Tell if we are fullscreen or not. */
      bool m_fullscreen;

      /** \brief The current level. */
      level* m_current_level;

      /** \brief A level in abeyance. */
      level* m_level_in_abeyance;

      /** \brief The input system and controllers. */
      input_layout* m_input_layout;

      /** \brief The status of the input controllers. */
      input::input_status m_input_status;

      /** \brief The layer containing all windows. */
      gui_layer* m_gui;

      /** \brief The name of the next level to load, if any. */
      std::string m_waiting_level;

      /** \brief Actions to do once an iteration is done. */
      std::queue<game_action*> m_post_actions;

      /** \brief Number of milliseconds between two iterations. */
      unsigned int m_time_step;

      /** \brief The date of the last call to progress. */
      time_ref::time_reference m_last_progress;

      /** \brief The prefix of the name of the game specific initialization
          procedure. */
      static std::string s_init_game_function_prefix;

      /** \brief The prefix of the name of the game specific ending
          procedure. */
      static std::string s_end_game_function_prefix;

      /** \brief For statistics. The sum of the time needed by the render
          procedure. */
      unsigned int m_sum_of_render_time;

      /** \brief For statistics. The sum of the time needed by the progress
          procedure. */
      unsigned int m_sum_of_progress_time;

      /** \brief For statistics. The count of calls to the render procedure. */
      unsigned int m_count_of_render;

      /** \brief For statistics. The count of calls to the progress
          procedure. */
      unsigned int m_count_of_progress;

      /** \brief For statistics. The date of the last printing of the
          statistics. */
      unsigned int m_date_of_last_statistics;

      /** \brief For statistics. The time needed for each call to progress. */
      std::list<unsigned int> m_progress_time;

      /** \brief For statistics. The time needed for each call to render. */
      std::list<unsigned int> m_render_time;

    }; // class game
  } // namespace engine
} // namespace bear

#endif // __ENGINE_GAME_HPP__
