/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file input_base_local_reader.cpp
 * \brief Implementation of the bear::engine::input_base_local_reader.
 * \author Julien Jorge
 */
#include "engine/input_reader/input_base_local_reader.hpp"
#include "input/system.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param player_index The index of the player to control.
 * \param layout The controller layout of the controlled player.
 */
bear::engine::input_base_local_reader::input_base_local_reader
( unsigned int player_index, const controller_layout& layout )
  : input_reader(player_index), m_layout(layout)
{

} // input_base_local_reader::input_base_local_reader()

/*----------------------------------------------------------------------------*/
/**
 * \brief Read inputs and apply them to the player.
 */
void bear::engine::input_base_local_reader::read_and_apply()
{
  if (m_player)
    {
      m_input_status.read();
      m_input_status.scan_inputs( *this );
    }
} // input_base_local_reader::read_and_apply()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell the player to start the action associated with a keyboard key.
 * \param key The code of the key.
 */
bool bear::engine::input_base_local_reader::key_pressed
( input::keyboard::key_code key )
{
  bool result = true;
  player::action action = m_layout(key);

  if ( action != player::action_null )
    send_start_action_to_player( action );
  else
    result = false;

  return result;
} // input_base_local_reader::key_pressed()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell the player to stop the action associated with a keyboard key.
 * \param key The code of the key.
 */
bool bear::engine::input_base_local_reader::key_released
( input::keyboard::key_code key )
{
  bool result = true;
  player::action action = m_layout(key);

  if ( action != player::action_null )
    send_stop_action_to_player( action );
  else
    result = false;

  return result;
} // input_base_local_reader::key_released()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell the player to continue the action associated with a keyboard key.
 * \param key The code of the key.
 */
bool bear::engine::input_base_local_reader::key_maintained
( input::keyboard::key_code key )
{
  bool result = true;
  player::action action = m_layout(key);

  if ( action != player::action_null )
    send_action_to_player( action );
  else
    result = false;

  return result;
} // input_base_local_reader::key_maintained()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell the player to start the action associated with a joystick button.
 * \param button The code of the button.
 * \param joy_index The index of the joystick.
 */
bool bear::engine::input_base_local_reader::button_pressed
( input::joystick::joy_code button, unsigned int joy_index )
{
  bool result = true;
  player::action action = m_layout(joy_index, button);

  if ( action != player::action_null )
    send_start_action_to_player( action );
  else
    result = false;

  return result;
} // input_base_local_reader::button_pressed()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell the player to stop the action associated with a joystick button.
 * \param button The code of the button.
 * \param joy_index The index of the joystick.
 */
bool bear::engine::input_base_local_reader::button_released
( input::joystick::joy_code button, unsigned int joy_index )
{
  bool result = true;
  player::action action = m_layout(joy_index, button);

  if ( action != player::action_null )
    send_stop_action_to_player( action );
  else
    result = false;

  return result;
} // input_base_local_reader::button_released()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell the player to continue the action associated with a joystick
 *        button.
 * \param button The code of the button.
 * \param joy_index The index of the joystick.
 */
bool bear::engine::input_base_local_reader::button_maintained
( input::joystick::joy_code button, unsigned int joy_index )
{
  bool result = true;
  player::action action = m_layout(joy_index, button);

  if ( action != player::action_null )
    send_action_to_player( action );
  else
    result = false;

  return result;
} // input_base_local_reader::button_maintained()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell the player to start the action associated with a mouse button.
 * \param button The code of the button.
 * \param pos The position of the cursor on the screen.
 */
bool bear::engine::input_base_local_reader::mouse_pressed
( input::mouse::mouse_code button,
  const claw::math::coordinate_2d<unsigned int>& pos )
{
  bool result = true;
  player::action action = m_layout(button);

  if ( action != player::action_null )
    send_start_action_to_player( action );
  else
    result = false;

  return result;
} // input_base_local_reader::mouse_pressed()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell the player to stop the action associated with a mouse button.
 * \param button The code of the button.
 * \param pos The position of the cursor on the screen.
 */
bool bear::engine::input_base_local_reader::mouse_released
( input::mouse::mouse_code button,
  const claw::math::coordinate_2d<unsigned int>& pos )
{
  bool result = true;
  player::action action = m_layout(button);

  if ( action != player::action_null )
    send_stop_action_to_player( action );
  else
    result = false;

  return result;
} // input_base_local_reader::mouse_released()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell the player to continue the action associated with a mouse button.
 * \param button The code of the button.
 * \param pos The position of the cursor on the screen.
 */
bool bear::engine::input_base_local_reader::mouse_maintained
( input::mouse::mouse_code button,
  const claw::math::coordinate_2d<unsigned int>& pos )
{
  bool result = true;
  player::action action = m_layout(button);

  if ( action != player::action_null )
    send_stop_action_to_player( action );
  else
    result = false;

  return result;
} // input_base_local_reader::mouse_maintained()
