/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file level_loader.hpp
 * \brief A class that loads a level file.
 * \author Julien Jorge
 */
#ifndef __ENGINE_LEVEL_LOADER_HPP__
#define __ENGINE_LEVEL_LOADER_HPP__

#include "engine/level.hpp"
#include "engine/base_item.hpp"

#include "engine/class_export.hpp"

namespace bear
{
  namespace engine
  {
    /**
     * \brief This class loads a level from a compiled level file.
     * \author Julien Jorge
     */
    class ENGINE_EXPORT level_loader
    {
    public:
      /** \brief The status of the loader (where we are in the file, the count
          of what we have to read, etc.) */
      struct status
      {
        /** \brief The position of the loader in the current section of the
            file. */
        unsigned int position;

        /** \brief The size of the current section. */
        unsigned int section_size;

        /** \brief The position of the loader in the file. */
        unsigned int global_position;

        /** \brief The size of the file (count of things to read). */
        unsigned int file_size;

        /** \brief The name of the section that we are loading. */
        std::string section_name;

      }; // class status

    public:
      level_loader( compiled_file& f );
      ~level_loader();

      const universe::size_box_type& get_level_size() const;
      const universe::size_box_type& get_camera_size() const;
      const std::string& get_level_music() const;

      void
      drop_level_content( std::vector<layer*>& layers, level_globals* &glob );

      void complete_run();
      bool one_step();

    private:
      bool one_step_item();
      bool one_step_level();

      void load_layer();

      void load_item_declaration();
      void load_item_definition();
      void load_item();

      void load_item_field_list();

      void load_item_field_int();
      void load_item_field_u_int();
      void load_item_field_real();
      void load_item_field_bool();
      void load_item_field_string();
      void load_item_field_sprite();
      void load_item_field_animation();
      void load_item_field_item();

      void load_item_field_int_list();
      void load_item_field_u_int_list();
      void load_item_field_real_list();
      void load_item_field_bool_list();
      void load_item_field_string_list();
      void load_item_field_sprite_list();
      void load_item_field_animation_list();
      void load_item_field_item_list();

      visual::sprite* load_sprite();
      visual::animation* load_animation();

      base_item* create_item_from_string( const std::string& name ) const;
      layer* create_layer_from_string
      ( const std::string& name, const universe::size_box_type& s ) const;

      template<typename T>
      std::string load_list( std::vector<T>& v );

    private:
      /** \brief The code of the next thing to read. */
      unsigned int m_next_code;

      /** \brief The music to play in the level. */
      std::string m_level_music;

      /** \brief The size of the level. */
      universe::size_box_type m_level_size;

      /** \brief The size of the camera. */
      universe::size_box_type m_camera_size;

      /** \brief The layers. */
      std::vector<layer*> m_layers;

      /** \brief Index+1 of the layer currently created. */
      unsigned int m_layer_index;

      /** \brief Resources of the level. */
      level_globals* m_level_globals;

      /** \brief The file that we are reading. */
      compiled_file& m_file;

      /** \brief The item we are currently loading, if any. */
      base_item* m_current_item;

      /** \brief Tell if the current item is fixed or not. */
      bool m_fixed;

      /** \brief Referenced items. */
      std::vector<base_item*> m_referenced;

      /** \brief Count of items in the current layer. */
      unsigned int m_items_count;

      /** \brief Index of the currently built item. */
      unsigned int m_item_index;

      /** \brief Index of the next item definition to read. */
      unsigned int m_referenced_index;

    }; // class level_loader
  } // namespace engine
} // namespace bear

#include "engine/code/level_loader.tpp"

#endif // __ENGINE_LEVEL_LOADER_HPP__
