/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file frame.cpp
 * \brief Implementation of the bear::gui::frame class.
 * \author Julien Jorge
 */
#include "gui/frame.hpp"

#include <claw/assert.hpp>

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param owner The owner of the frame.
 * \param corner Sprite to use to draw corners (should be the top-left corner).
 * \param h_border Sprite to use to draw horizontal borders.
 * \param v_border Sprite to use to draw vertical borders.
 * \param background Sprite to use to draw the background.
 *
 * \remark
 * - all sprites will be deleted in the destructor,
 * - sprites can be NULL.
 *
 * \pre !corner.is_mirrored() && !corner.is_flipped()
 */
bear::gui::frame::frame( visual_component* owner, visual::sprite* corner,
                   visual::sprite* h_border, visual::sprite* v_border,
                   visual::sprite* background )
  : visual_component(owner), m_corner(corner), m_horizontal_border(h_border),
    m_vertical_border(v_border), m_background(background),
    m_border_size( v_border ? v_border->width() : 0,
                   h_border ? h_border->height() : 0 )
{
  CLAW_PRECOND( corner ? !corner->is_mirrored() : true );
  CLAW_PRECOND( corner ? !corner->is_flipped() : true );
} // frame::frame()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
bear::gui::frame::~frame()
{
  if (m_background)
    delete m_background;

  if (m_horizontal_border)
    delete m_horizontal_border;

  if (m_vertical_border)
    delete m_vertical_border;

  if (m_corner)
    delete m_corner;
} // frame::~frame()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the transparency of the background.
 * \param aplha The new value.
 */
void bear::gui::frame::set_background_alpha( double alpha )
{
  if (m_background)
    m_background->set_alpha_blend(alpha);
} // frame::set_background_alpha()

/*----------------------------------------------------------------------------*/
/**
 * \brief Method called after the component has been resized.
 */
void bear::gui::frame::on_resized()
{
  const unsigned int min_width  = 2 * (corner_width()  - m_border_size.x);
  const unsigned int min_height = 2 * (corner_height() - m_border_size.y);

  if ( width() < min_width )
    {
      if ( height() < min_height )
        set_size( min_width, min_height );
      else
        set_size( min_width, height() );
    }
  else if ( height() < min_height )
    set_size( width(), min_height );
  else
    {
      if (m_background)
        m_background->set_size( width(), height() );

      if (m_horizontal_border)
        m_horizontal_border->set_width( width() - min_width );

      if (m_vertical_border)
        m_vertical_border->set_height( height() - min_height );
    }
} // frame::resized()

/*----------------------------------------------------------------------------*/
/**
 * \brief Draw the frame on a screen.
 * \param screen The screen on which to draw.
 * \param pos The position of this frame in the screen.
 */
void
bear::gui::frame::display( visual::screen& screen,
                     const claw::math::coordinate_2d<unsigned int>& pos ) const
{
  if (m_background)
    display_background( screen, pos );

  if (m_horizontal_border)
    display_horizontal_borders( screen, pos );

  if (m_vertical_border)
    display_vertical_borders( screen, pos );

  if (m_corner)
    display_corners( screen, pos );
} // frame::display()

/*----------------------------------------------------------------------------*/
/**
 * \brief Draw the background of the frame on a screen.
 * \param screen The screen on which to draw.
 * \param pos The position of this frame.
 */
void bear::gui::frame::display_background
( visual::screen& screen,
  const claw::math::coordinate_2d<unsigned int>& pos ) const
{
  screen.render(pos, *m_background);
} // frame::display_background()

/*----------------------------------------------------------------------------*/
/**
 * \brief Draw the horizontal borders of the frame on a screen.
 * \param screen The screen on which to draw.
 * \param pos The position of this frame.
 */
void bear::gui::frame::display_horizontal_borders
( visual::screen& screen,
  const claw::math::coordinate_2d<unsigned int>& pos ) const
{
  claw::math::coordinate_2d<int> sprite_pos(pos - m_border_size);
  sprite_pos.x += corner_width();
  screen.render(sprite_pos, *m_horizontal_border);

  visual::sprite bottom_border(*m_horizontal_border);
  bottom_border.flip(true);

  sprite_pos.y += height() + m_border_size.y;
  screen.render(sprite_pos, bottom_border);
} // frame::display_horizontal_borders()

/*----------------------------------------------------------------------------*/
/**
 * \brief Draw the vertical borders of the frame on a screen.
 * \param screen The screen on which to draw.
 * \param pos The position of this frame.
 */
void bear::gui::frame::display_vertical_borders
( visual::screen& screen,
  const claw::math::coordinate_2d<unsigned int>& pos ) const
{
  claw::math::coordinate_2d<int> sprite_pos(pos - m_border_size);
  sprite_pos.y += corner_height();
  screen.render(sprite_pos, *m_vertical_border);

  visual::sprite right_border(*m_vertical_border);
  right_border.mirror(true);

  sprite_pos.x += width() + m_border_size.x;
  screen.render(sprite_pos, right_border);
} // frame::display_vertical_borders()

/*----------------------------------------------------------------------------*/
/**
 * \brief Draw the corners of the frame on a screen.
 * \param screen The screen on which to draw.
 * \param pos The position of this frame.
 */
void bear::gui::frame::display_corners
( visual::screen& screen,
  const claw::math::coordinate_2d<unsigned int>& pos ) const
{
  claw::math::coordinate_2d<int> sprite_pos(pos - m_border_size);
  visual::sprite corner(*m_corner);

  screen.render(sprite_pos, corner);

  sprite_pos.x += width() + 2 * m_border_size.x - corner.width();
  corner.mirror(true);
  screen.render(sprite_pos, corner);

  sprite_pos.y += height() + 2 * m_border_size.y - corner.height();
  corner.flip(true);
  screen.render(sprite_pos, corner);

  sprite_pos.x = pos.x - m_border_size.x;
  corner.mirror(false);
  screen.render(sprite_pos, corner);
} // frame::display_corners()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the width of the corners.
 */
unsigned int bear::gui::frame::corner_width() const
{
  if (m_corner)
    return m_corner->width();
  else
    return m_border_size.x;
} // frame::corner_width()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the height of the corners.
 */
unsigned int bear::gui::frame::corner_height() const
{
  if (m_corner)
    return m_corner->height();
  else
    return m_border_size.y;
} // frame::corner_height()
