/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file menu.cpp
 * \brief Implementation of the bear::gui::menu class.
 * \author Julien Jorge
 */
#include "gui/menu.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param owner The component owning this menu.
 * \param cursor The sprite to use for indicating which item is selected.
 * \param font The font to use for items.
 * \param horizontal_margins The size of the margins on the left and on the
 *        right.
 * \param vertical_margins The size of the margins at the top and at the bottom.
 */
bear::gui::menu::menu( visual_component* owner, visual::sprite* cursor,
                 font_type font, unsigned int horizontal_margins,
                 unsigned int vertical_margins,
                 unsigned int line_space)
  : visual_component(owner), m_selected(0), m_font(font),
    m_margins(horizontal_margins, vertical_margins),
    m_line_space(line_space - line_space%2)
{
  CLAW_PRECOND( cursor != NULL );
  CLAW_PRECOND( font != NULL );

  set_size( cursor->width(), cursor->height() );

  m_cursor = new bear::gui::picture( this, cursor );
} // menu::menu()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a component at the end of the menu.
 */
bear::gui::static_text& bear::gui::menu::add()
{
  m_items.push_back( new static_text(this, m_font) );

  return *m_items.back();
} // menu::add()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the item currently selected.
 */
bear::gui::static_text& bear::gui::menu::selected()
{
  CLAW_PRECOND( !m_items.empty() );
  return *m_items[m_selected];
} // menu::selected()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get a component of the menu.
 * \pre i < m_items.size()
 */
bear::gui::static_text& bear::gui::menu::item( unsigned int i )
{
  CLAW_PRECOND( i < m_items.size() );

  return *m_items[i];
} // menu::item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the cursor's position.
 */
unsigned int bear::gui::menu::cursor_position() const
{
  CLAW_PRECOND( !m_items.empty() );
  return m_selected;
} // menu::cursor_position()

/*----------------------------------------------------------------------------*/
/**
 * \brief Adjust the size of the menu to fit the sizes of its children.
 */
void bear::gui::menu::auto_size()
{
  set_items_position();

  claw::math::coordinate_2d<unsigned int> s(0, 0);

  for (unsigned int i=0; i!=m_items.size(); ++i)
    if ( m_items[i]->width() > s.x )
      s.x = m_items[i]->width();

  s.x += m_cursor->width() + 3 * m_margins.x;

  // be sure that, when aligned with the first or the last item, the cursor is
  // not out of the box.
  if ( !m_items.empty() )
    {
      s.y = m_items.back()->bottom() + m_margins.y + 1;

      if ( m_cursor->height() > (m_items.back()->height() + 2 * m_margins.y) )
        s.y += (m_cursor->height()
                - m_items.back()->height() + 2 * m_margins.y) / 2;
    }
  else
    s.y = std::max(2 * m_margins.y, m_cursor->height());

  set_size( s );
  align_cursor();
} // menu::auto_size()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the position of the items.
 */
void bear::gui::menu::set_items_position()
{
  claw::math::coordinate_2d<unsigned int> pos
    ( 2*m_margins.x + m_cursor->width(), m_margins.y );

  if ( !m_items.empty() )
    if ( m_cursor->height() > (m_items[0]->height() + 2 * m_margins.y) )
      pos.y += (m_cursor->height()
                - m_items[0]->height() + 2 * m_margins.y) / 2;

  for (unsigned int i=0; i!=m_items.size(); ++i)
    {
      m_items[i]->set_position( pos );
      pos.y += m_items[i]->height() + m_line_space;
    }
} // menu::set_items_position()

/*----------------------------------------------------------------------------*/
/**
 * \brief Move the cursor up.
 */
void bear::gui::menu::move_up()
{
  CLAW_PRECOND( !m_items.empty() );

  if ( m_selected == 0 )
    m_selected = m_items.size() - 1;
  else
    --m_selected;

  align_cursor();
} // menu::move_up()

/*----------------------------------------------------------------------------*/
/**
 * \brief Move the cursor down.
 */
void bear::gui::menu::move_down()
{
  CLAW_PRECOND( !m_items.empty() );

  m_selected = (m_selected + 1) % m_items.size();
  align_cursor();
} // menu::move_down()

/*----------------------------------------------------------------------------*/
/**
 * \brief Align the cursor with the selected item.
 */
void bear::gui::menu::align_cursor()
{
  CLAW_PRECOND( m_selected < m_items.size() );

  claw::math::coordinate_2d<unsigned int> pos( m_margins.x,
                                               selected().get_position().y );

  if ( m_cursor->height() >= selected().height() )
    pos.y -= (m_cursor->height() - selected().height()) / 2;
  else
    pos.y += (selected().height() - m_cursor->height()) / 2;

  m_cursor->set_position(pos);
} // menu::align_cursor()

/*----------------------------------------------------------------------------*/
/**
 * \brief Process a keyboard key.
 * \param key The pressed key.
 * \return true if the key has been processed.
 */
bool bear::gui::menu::on_key_press( input::keyboard::key_code key )
{
  bool result = true;

  if ( m_items.empty() )
    result = false;
  else
    switch( key )
      {
      case input::keyboard::kc_tab :
      case input::keyboard::kc_down :
        {
          move_down();
          break;
        }
      case input::keyboard::kc_up :
        {
          move_up();
          break;
        }
      default: result = false;
      }

  return result;
} // menu::on_key_press()

/*----------------------------------------------------------------------------*/
/**
 * \brief Process a joystick button.
 * \param button The pressed button.
 * \param joystick The index of the joystick on which is this button.
 * \return true if the button has been processed.
 */
bool bear::gui::menu::on_button_press
( input::joystick::joy_code button, unsigned int joy_index )
{
  bool result = true;

  if ( m_items.empty() )
    result = false;
  else
    switch( button )
      {
      case input::joystick::jc_axis_down :
        {
          move_down();
          break;
        }
      case input::joystick::jc_axis_up :
        {
          move_up();
          break;
        }
      default: result = false;
      }

  return result;
} // menu::on_button_press()

/*----------------------------------------------------------------------------*/
/**
 * \brief Process a movement of the mouse.
 * \param pos The new position of the mouse.
 * \return true if the position has been processed.
 */
bool bear::gui::menu::on_mouse_move
( const claw::math::coordinate_2d<unsigned int>& pos )
{
  if ( !m_items.empty() )
    {
      const unsigned int index
        = ( pos.y - m_margins.y + m_line_space / 2 ) / 
        ( m_items[0]->height() + m_line_space );
      
      if ( index < m_items.size() )
        {
          m_selected = index;
          align_cursor();
        }
    }

  return true;
} // menu::on_mouse_move()
