/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file visual_component.cpp
 * \brief Implementation of the bear::gui::visual_component class.
 * \author Julien Jorge
 */
#include "gui/visual_component.hpp"

#include <algorithm>
#include <claw/assert.hpp>
#include <claw/functional.hpp>

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param owner The component containing this component.
 */
bear::gui::visual_component::visual_component( visual_component* owner )
  : m_box(0, 0, 0, 0), m_owner(owner), m_focused_component(-1),
    m_visible(true), m_input_priority(false)
{
  if (m_owner)
    m_owner->add_component(this);
} // visual_component::visual_component()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
bear::gui::visual_component::~visual_component()
{
  std::for_each( m_components.begin(), m_components.end(),
                 claw::delete_function<visual_component*>() );
} // visual_component::~visual_component()

/*----------------------------------------------------------------------------*/
/**
 * \brief Draw the component and its sub components on a screen.
 * \param screen The screen on which to draw the component.
 */
void bear::gui::visual_component::render( visual::screen& screen ) const
{
  if (m_owner)
    render( screen, m_owner->get_position() );
  else
    render( screen, claw::math::coordinate_2d<unsigned int>(0, 0) );
} // visual_component::render()

/*----------------------------------------------------------------------------*/
/**
 * \brief Inform the component that a key had been pressed.
 * \param key The value of the pressed key.
 */
bool bear::gui::visual_component::key_pressed( input::keyboard::key_code key )
{
  bool result;

  if (m_input_priority)
    {
      result = on_key_press(key);

      if ( !result )
        if ( m_focused_component >= 0)
          result = m_components[m_focused_component]->key_pressed(key);
    }
  else if ( m_focused_component >= 0)
    {
      result = m_components[m_focused_component]->key_pressed(key);

      if ( !result )
        result = on_key_press(key);
    }
  else
    result = on_key_press(key);

  return result;
} // visual_component::key_pressed()

/*----------------------------------------------------------------------------*/
/**
 * \brief Inform the component that a joystick button had been pressed.
 * \param button The value of the pressed button.
 * \param joy_index The index of the joystick.
 */
bool bear::gui::visual_component::button_pressed
( input::joystick::joy_code button, unsigned int joy_index )
{
  bool result;

  if (m_input_priority)
    {
      result = on_button_press(button, joy_index);

      if ( !result )
        if ( m_focused_component >= 0)
          result = m_components[m_focused_component]->button_pressed
            (button, joy_index);
    }
  else if ( m_focused_component >= 0)
    {
      result = m_components[m_focused_component]->button_pressed
        (button, joy_index);

      if ( !result )
        result = on_button_press(button, joy_index);
    }
  else
    result = on_button_press(button, joy_index);

  return result;
} // visual_component::button_pressed()

/*----------------------------------------------------------------------------*/
/**
 * \brief Inform the component that a mouse button had been pressed.
 * \param key The value of the pressed button.
 * \param pos The current position of the cursor.
 */
bool bear::gui::visual_component::mouse_pressed
( input::mouse::mouse_code key,
  const claw::math::coordinate_2d<unsigned int>& pos )
{
  bool result;

  if (m_input_priority)
    {
      result = on_mouse_press(key, pos);

      if ( !result )
        result = broadcast_mouse_press(key, pos);
    }
  else
    {
      result = broadcast_mouse_press(key, pos);

      if ( !result )
        result = on_mouse_press(key, pos);
    }

  return result;
} // visual_component::mouse_pressed()

/*----------------------------------------------------------------------------*/
/**
 * \brief Inform the component that the mouse has been moved.
 * \param pos The new position of the cursor.
 */
bool bear::gui::visual_component::mouse_move
( const claw::math::coordinate_2d<unsigned int>& pos )
{
  bool result;

  if (m_input_priority)
    {
      result = on_mouse_move(pos);

      if ( !result )
        result = broadcast_mouse_move( pos );
    }
  else
    {
      result = broadcast_mouse_move( pos );

      if ( !result )
        result = on_mouse_move(pos);
    }

  return result;
} // visual_component::mouse_move()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the size of the component.
 * \param size The new size.
 */
void bear::gui::visual_component::set_size
( const claw::math::coordinate_2d<unsigned int>& size )
{
  set_size( size.x, size.y );
} // visual_component::set_size()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the size of the component.
 * \param w The new width.
 * \param h The new height.
 */
void bear::gui::visual_component::set_size( unsigned int w, unsigned int h )
{
  const unsigned int old_w = m_box.width;
  const unsigned int old_h = m_box.height;

  m_box.width = w;
  m_box.height = h;

  stay_in_owner();

  if ( (old_w != m_box.width) || (old_h != m_box.height) )
    on_resized();
} // visual_component::set_size()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the width of the component.
 * \param w The new width.
 */
void bear::gui::visual_component::set_width( unsigned int w )
{
  set_size( w, height() );
} // visual_component::set_width()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the height of the component.
 * \param h The new height.
 */
void bear::gui::visual_component::set_height( unsigned int h )
{
  set_size( width(), h );
} // visual_component::set_height()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the position of the component in its owner.
 * \param pos The new position.
 */
void bear::gui::visual_component::set_position
( const claw::math::coordinate_2d<unsigned int>& pos )
{
  set_position( pos.x, pos.y );
} // visual_component::set_position()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the position of the component.
 * \param x The new x-coordinate.
 * \param y The new y-coordinate.
 */
void bear::gui::visual_component::set_position( unsigned int x, unsigned int y )
{
  const unsigned int w = m_box.width;
  const unsigned int h = m_box.height;

  m_box.position.x = x;
  m_box.position.y = y;
  stay_in_owner();

  if ( (w != m_box.width) || (h != m_box.height) )
    on_resized();
} // visual_component::set_size()

/*----------------------------------------------------------------------------*/
/**
 * \brief Show or hide the component.
 * \param b The new visibility.
 */
void bear::gui::visual_component::set_visible( bool b )
{
  m_visible = b;
} // visual_component::set_visible()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the position of this component when tabbing.
 * \param v The new position.
 */
void bear::gui::visual_component::set_tab_order( unsigned int v )
{
  if (m_owner)
    m_owner->change_tab_position( this, v );
} // visual_component::set_tab_order()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set if the current component must process key_ press event before its
 *        children.
 * \param this_first True to set this compoment as priority.
 */
void bear::gui::visual_component::set_input_priority( bool this_first )
{
  m_input_priority = this_first;
} // visual_component::set_input_priority()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the width of the component.
 */
unsigned int bear::gui::visual_component::width() const
{
  return m_box.width;
} // visual_component::width()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the height of the component.
 */
unsigned int bear::gui::visual_component::height() const
{
  return m_box.height;
} // visual_component::height()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the x-coordinate of the component's right edge.
 */
unsigned int bear::gui::visual_component::right() const
{
  return m_box.right();
} // visual_component::right()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the y-coordinate of the component's bottom edge.
 */
unsigned int bear::gui::visual_component::bottom() const
{
  return m_box.bottom();
} // visual_component::bottom()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the size of the component.
 */
claw::math::coordinate_2d<unsigned int>
bear::gui::visual_component::get_size() const
{
  return claw::math::coordinate_2d<unsigned int>( m_box.width, m_box.height );
} // visual_component::get_size()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the position of the component in its owner.
 */
const claw::math::coordinate_2d<unsigned int>&
bear::gui::visual_component::get_position() const
{
  return m_box.position;
} // visual_component::get_position()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the rectangle bounding this component.
 */
const claw::math::rectangle<unsigned int>&
bear::gui::visual_component::get_rectangle() const
{
  return m_box;
} // visual_component::get_rectangle()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the component is visible.
 */
bool bear::gui::visual_component::get_visible() const
{
  return m_visible;
} // visual_component::get_visible()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a component in this component.
 * \param that The component to add.
 */
void bear::gui::visual_component::add_component( visual_component* that )
{
  CLAW_PRECOND( std::find(m_components.begin(), m_components.end(), that)
                == m_components.end() );

  m_components.push_back(that);

  if (m_focused_component < 0)
    m_focused_component = 0;
} // visual_component::add_component()

/*----------------------------------------------------------------------------*/
/**
 * \brief Change the position of a component when tabbing.
 * \param that The component to move.
 * \param pos The new position.
 * \remark The item which was at the position \a pos will be placed at the old
 *         position of \a that.
 */
void bear::gui::visual_component::change_tab_position
( const visual_component* that, unsigned int pos )
{
  CLAW_PRECOND( std::find(m_components.begin(), m_components.end(), that)
                != m_components.end() );

  if ( pos > m_components.size() - 1 )
    pos = m_components.size() - 1;

  std::swap( m_components[pos], *std::find(m_components.begin(),
                                           m_components.end(), that) );
} // visual_component::change_tab_position()

/*----------------------------------------------------------------------------*/
/**
 * \brief Method called after the component has been resized.
 */
void bear::gui::visual_component::on_resized()
{
  // nothing to do
} // visual_component::on_resized()

/*----------------------------------------------------------------------------*/
/**
 * \brief Draw the component on a screen.
 * \param screen The screen on which to draw the component.
 * \param pos The position of the component in the screen.
 */
void bear::gui::visual_component::display
( visual::screen& screen,
  const claw::math::coordinate_2d<unsigned int>& pos ) const
{
  // nothing to do
} // visual_component::display()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell the focused component that a key has been pressed.
 * \param key The pressed key.
 * \return true if the key has been processed.
 */
bool bear::gui::visual_component::on_key_press( input::keyboard::key_code key )
{
  bool result = false;

  if (key == input::keyboard::kc_tab )
    {
      result = true;

      if ( !m_components.empty() )
        m_focused_component = (m_focused_component + 1) % m_components.size();
    }

  return result;
} // visual_component::on_key_press()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell the component that a joystick button has been pressed.
 * \param button The value of the pressed button.
 * \param joy_index The index of the joystick.
 * \return true if the button has been processed.
 */
bool bear::gui::visual_component::on_button_press
( input::joystick::joy_code button, unsigned int joy_index )
{
  return false;
} // visual_component::on_button_press()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell the focused component that a mouse button has been pressed.
 * \param key The pressed key.
 * \param pos The current position of the cursor.
 * \return true if the key has been processed.
 */
bool bear::gui::visual_component::on_mouse_press
( input::mouse::mouse_code key,
  const claw::math::coordinate_2d<unsigned int>& pos )
{
  return false;
} // visual_component::on_mouse_press()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell the component that the mouse has been moved.
 * \param pos The new position of the cursor.
 * \return true if the button has been processed.
 */
bool bear::gui::visual_component::on_mouse_move
( const claw::math::coordinate_2d<unsigned int>& pos )
{
  return false;
} // visual_component::on_mouse_move()

/*----------------------------------------------------------------------------*/
/**
 * \brief Inform the sub components that the mouse has been moved.
 * \param pos The new position of the cursor.
 */
bool bear::gui::visual_component::broadcast_mouse_move
( const claw::math::coordinate_2d<unsigned int>& pos )
{
  bool result(false);
  component_list::iterator it;

  for (it=m_components.begin(); !result && (it!=m_components.end()); ++it)
    if ( (*it)->m_box.includes(pos) )
      result = (*it)->mouse_move( pos - (*it)->get_position() );

  return result;
} // visual_component::broadcast_mouse_move()

/*----------------------------------------------------------------------------*/
/**
 * \brief Inform the sub components that the mouse has been moved.
 * \param button The pressed key.
 * \param pos The current position of the cursor.
 * \return true if the key has been processed.
 */
bool bear::gui::visual_component::broadcast_mouse_press
( input::mouse::mouse_code button,
  const claw::math::coordinate_2d<unsigned int>& pos )
{
  bool result(false);
  component_list::iterator it;

  for (it=m_components.begin(); !result && (it!=m_components.end()); ++it)
    if ( (*it)->m_box.includes(pos) )
      result = (*it)->mouse_pressed( button, pos - (*it)->get_position() );

  return result;
} // visual_component::broadcast_mouse_press()

/*----------------------------------------------------------------------------*/
/**
 * \brief Draw the component and its sub components on a screen.
 * \param screen The screen on which to draw the component.
 * \param pos The position of the owner on the screen.
 */
void bear::gui::visual_component::render
( visual::screen& screen,
  const claw::math::coordinate_2d<unsigned int>& pos ) const
{
  if (m_visible)
    {
      display( screen, pos + m_box.position );

      for ( component_list::const_iterator it=m_components.begin();
            it!=m_components.end();
            ++it )
        (*it)->render( screen, pos + m_box.position );
    }
} // visual_component::render()

/*----------------------------------------------------------------------------*/
/**
 * \brief Adjust the width and the height of the component to stay in the owner.
 */
void bear::gui::visual_component::stay_in_owner()
{
  if (m_owner)
    {
      if ( m_box.position.x >= m_owner->width() )
        m_box.position.x = m_owner->width() - 1;

      if ( m_box.position.y >= m_owner->height() )
        m_box.position.y = m_owner->height() - 1;

      if ( m_box.right() >= m_owner->width() )
        m_box.width = m_owner->width() - m_box.position.x;

      if ( m_box.bottom() >= m_owner->height() )
        m_box.height = m_owner->height() - m_box.position.y;
    }
} // visual_component::stay_in_owner()
