/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file static_text.hpp
 * \brief Simple component to print a text on the screen.
 * \author Julien Jorge
 */
#ifndef __GUI_STATIC_TEXT_HPP__
#define __GUI_STATIC_TEXT_HPP__

#include "text/font.hpp"
#include "gui/visual_component.hpp"

#include <claw/smart_ptr.hpp>
#include <claw/coordinate_2d.hpp>

#include "gui/class_export.hpp"

namespace bear
{
  namespace gui
  {
    /**
     * \brief Simple component to print a text on the screen.
     * \author Julien Jorge
     */
    class GUI_EXPORT static_text:
      public visual_component
    {
    public:
      /** \brief The type of the font used to display the text. */
      typedef claw::memory::smart_ptr<text::font> font_type;

    public:
      static_text( visual_component* owner );
      static_text( visual_component* owner, font_type f );

      void set_font( font_type f );
      void set_auto_size( bool b );
      void set_auto_expand( bool b );
      void set_text( const std::string& text );

      const std::string& get_text() const;
      font_type get_font() const;

      claw::math::coordinate_2d<unsigned int>
      get_size_with_max_width( unsigned int w ) const;

      std::size_t
      get_longest_text( const std::string& text, std::size_t i ) const;

    private:
      claw::math::coordinate_2d<unsigned int>
      get_auto_size_with_max_width( unsigned int w ) const;

      std::size_t display_word_dummy
      ( const std::string& text,
        claw::math::coordinate_2d<unsigned int>& cursor,
        std::size_t& i, const std::size_t line_length ) const;

      std::size_t display_word_dummy
      ( const std::string& text,
        claw::math::coordinate_2d<unsigned int>& cursor,
        std::size_t& i, const std::size_t n,
        const std::size_t line_length ) const;


      void display( visual::screen& screen,
                    const claw::math::coordinate_2d<unsigned int>& pos ) const;

      void display_word( visual::screen& screen,
                         const claw::math::coordinate_2d<unsigned int>& pos,
                         claw::math::coordinate_2d<unsigned int>& cursor,
                         std::size_t& i ) const;

      void display_word( visual::screen& screen,
                         const claw::math::coordinate_2d<unsigned int>& pos,
                         claw::math::coordinate_2d<unsigned int>& cursor,
                         std::size_t& i, const std::size_t n,
                         const std::size_t line_length ) const;

      void adjust_size_to_text();
      void adjust_height_to_text();

    private:
      /** \brief The text to draw. */
      std::string m_text;

      /** \brief The font used to draw the text. */
      font_type m_font;

      /** \brief Tell if we must adjust the size of the component to the size of
          the text. */
      bool m_auto_size;

      /** \brief Tell if we must adjust the height of the component to the size
          of the text. */
      bool m_auto_expand;

    }; // class static_text
  } // namespace gui
} // namespace bear

#endif // __GUI_STATIC_TEXT_HPP__
