/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file joystick.cpp
 * \brief Implementation of the bear::input::joystick class.
 * \author Julien Jorge
 */
#include "input/joystick.hpp"
#include <SDL/SDL.h>
#include <claw/assert.hpp>
#include <claw/exception.hpp>

/*----------------------------------------------------------------------------*/
std::vector<std::string> bear::input::joystick::s_button_strings;

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the number of plugged joysticks.
 */
unsigned int bear::input::joystick::number_of_joysticks()
{
  return SDL_NumJoysticks();
} // joystick::number_of_joysticks()

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
bear::input::joystick::joystick( unsigned int joy_id )
  : m_id(joy_id)
{
  CLAW_PRECOND( joy_id < number_of_joysticks() );

  m_joystick = SDL_JoystickOpen(joy_id);

  if (!m_joystick)
    throw CLAW_EXCEPTION( SDL_GetError() );

  if ( s_button_strings.empty() )
    default_joy_code_strings();
} // joystick::joystick()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
bear::input::joystick::~joystick()
{
  SDL_JoystickClose(m_joystick);
} // joystick::~joystick()

/*----------------------------------------------------------------------------*/
/**
 * \brief Convert a joy_code to a human-readable string.
 * \param k The button to convert.
 */
const std::string& bear::input::joystick::get_name_of( joy_code k )
{
  CLAW_PRECOND( k < s_button_strings.size() );

  return s_button_strings[k];
} // joystick::get_name_of()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the name of a button.
 * \param k The button to name.
 * \param s The string describing this button.
 */
void bear::input::joystick::set_name_of( joy_code k, const std::string& s )
{
  CLAW_PRECOND( k < s_button_strings.size() );

  s_button_strings[k] = s;
} // joystick::set_name_of()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get an iterator on the first pressed button.
 */
bear::input::joystick::const_iterator bear::input::joystick::begin() const
{
  return m_pressed_buttons.begin();
} // joystick::begin()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get an iterator after the last pressed button.
 */
bear::input::joystick::const_iterator bear::input::joystick::end() const
{
  return m_pressed_buttons.end();
} // joystick::end()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if no buttons are pressed.
 */
bool bear::input::joystick::empty() const
{
  return m_pressed_buttons.empty();
} // joystick::empty()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the identifier of the joystick.
 */
unsigned int bear::input::joystick::get_id() const
{
  return m_id;
} // joystick::get_id()

/*----------------------------------------------------------------------------*/
/**
 * \brief Re-read the status of all buttons.
 * \pre The caller is an instance of bear::input::system.
 */
void bear::input::joystick::refresh()
{
  m_pressed_buttons.clear();

  joy_code axis = get_pressed_axis();

  if (axis != jc_invalid)
    m_pressed_buttons.push_back( axis );

  unsigned int num_buttons = c_number_of_buttons;

  if ( num_buttons > (unsigned int)SDL_JoystickNumButtons(m_joystick) )
    num_buttons = SDL_JoystickNumButtons(m_joystick);

  for (unsigned int button=0; button!=num_buttons; ++button)
    if ( SDL_JoystickGetButton(m_joystick, button) )
      m_pressed_buttons.push_back( sdl_button_to_local(button) );

} // joystick::refresh()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the currently pressed axis.
 */
bear::input::joystick::joy_code bear::input::joystick::get_pressed_axis() const
{
  const bool up    = SDL_JoystickGetAxis( m_joystick, 1 ) < -1000;
  const bool down  = SDL_JoystickGetAxis( m_joystick, 1 ) > 1000;
  const bool left  = SDL_JoystickGetAxis( m_joystick, 0 ) < -1000;
  const bool right = SDL_JoystickGetAxis( m_joystick, 0 ) > 1000;

  joy_code result = jc_invalid;

  if (up)
    {
      if (left)
        result = jc_axis_up_left;
      else if (right)
        result = jc_axis_up_right;
      else
        result = jc_axis_up;
    }
  else if (down)
    {
      if (left)
        result = jc_axis_down_left;
      else if (right)
        result = jc_axis_down_right;
      else
        result = jc_axis_down;
    }
  else if (left)
    result = jc_axis_left;
  else if (right)
    result = jc_axis_right;

  return result;
} // joystick::get_pressed_axis()

/*----------------------------------------------------------------------------*/
/**
 * \brief Convert a SDL_Joystick button index value to the corresponding
 *        joy_code.
 */
bear::input::joystick::joy_code
bear::input::joystick::sdl_button_to_local( unsigned int sdl_val ) const
{
  unsigned int b = jc_button_1 + sdl_val;

  if (b > c_number_of_buttons)
    return jc_invalid;
  else
    return b;
} // joystick::sdl_button_to_local()

/*----------------------------------------------------------------------------*/
/**
 * \brief Fill the m_button_string table with default strings.
 */
void bear::input::joystick::default_joy_code_strings()
{
  s_button_strings.resize(c_joy_codes_count);
  std::fill( s_button_strings.begin(), s_button_strings.end(), "Undefined" );

  s_button_strings[jc_axis_up]         = "up";
  s_button_strings[jc_axis_down]       = "down";
  s_button_strings[jc_axis_left]       = "left";
  s_button_strings[jc_axis_right]      = "right";
  s_button_strings[jc_axis_up_left]    = "up left";
  s_button_strings[jc_axis_up_right]   = "up right";
  s_button_strings[jc_axis_down_left]  = "down left";
  s_button_strings[jc_axis_down_right] = "down_right";

  s_button_strings[jc_button_1]  = "button 1";
  s_button_strings[jc_button_2]  = "button 2";
  s_button_strings[jc_button_3]  = "button 3";
  s_button_strings[jc_button_4]  = "button 4";
  s_button_strings[jc_button_5]  = "button 5";
  s_button_strings[jc_button_6]  = "button 6";
  s_button_strings[jc_button_7]  = "button 7";
  s_button_strings[jc_button_8]  = "button 8";
  s_button_strings[jc_button_9]  = "button 9";
  s_button_strings[jc_button_10] = "button 10";
  s_button_strings[jc_button_11] = "button 11";
  s_button_strings[jc_button_12] = "button 12";
  s_button_strings[jc_button_13] = "button 13";
  s_button_strings[jc_button_14] = "button 14";
  s_button_strings[jc_button_15] = "button 15";
  s_button_strings[jc_button_16] = "button 16";
} // joystick::default_joy_code_strings()
