/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file keyboard.cpp
 * \brief Implementation of the bear::input::keyboard class.
 * \author Julien Jorge
 */
#include "input/keyboard.hpp"
#include <SDL/SDL.h>
#include <algorithm>
#include <claw/assert.hpp>

/*----------------------------------------------------------------------------*/
std::vector<std::string> bear::input::keyboard::s_key_strings;

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
bear::input::keyboard::keyboard()
{
  if ( s_key_strings.empty() )
    default_key_strings();
} // keyboard::keyboard()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if a key_code represent a printable symbol.
 * \param k The code to test.
 */
bool bear::input::keyboard::is_symbol( key_code k )
{
  return ( (k >= kc_space) && (k <= kc_tilde) )
    || ( (k >= kc_keypad_0) && (k <= kc_keypad_plus) );
} // keyboard::is_symbol()

/*----------------------------------------------------------------------------*/
/**
 * \brief Convert a key_code to the conresponding symbol.
 * \param k The code to convert.
 * \pre is_symbol(k) == true
 */
char bear::input::keyboard::code_to_char( key_code k )
{
  CLAW_PRECOND( is_symbol(k) );

  if ( (k >= kc_space) && (k <= kc_tilde) )
    return (char)k;
  else
    switch(k)
      {
      case kc_keypad_0 : return '0'; break;
      case kc_keypad_1 : return '1'; break;
      case kc_keypad_2 : return '2'; break;
      case kc_keypad_3 : return '3'; break;
      case kc_keypad_4 : return '4'; break;
      case kc_keypad_5 : return '5'; break;
      case kc_keypad_6 : return '6'; break;
      case kc_keypad_7 : return '7'; break;
      case kc_keypad_8 : return '8'; break;
      case kc_keypad_9 : return '9'; break;
      case kc_keypad_period   : return '.'; break;
      case kc_keypad_divide   : return '/'; break;
      case kc_keypad_multiply : return '*'; break;
      case kc_keypad_minus    : return '-'; break;
      case kc_keypad_plus     : return '+'; break;
      default:
        {
          CLAW_ASSERT( false, "You shouldn't see this error");
          return '\0';
        }
      }
} // keyboard::code_to_char()

/*----------------------------------------------------------------------------*/
/**
 * \brief Convert a key_code to a human-readable string.
 * \param k The key to convert.
 */
const std::string& bear::input::keyboard::get_name_of( key_code k )
{
  CLAW_PRECOND( k < s_key_strings.size() );

  return s_key_strings[k];
} // keyboard::get_name_of()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the name of a key.
 * \param k The key to name.
 * \param s The string describing this key.
 */
void bear::input::keyboard::set_name_of( key_code k, const std::string& s )
{
  CLAW_PRECOND( k < s_key_strings.size() );

  s_key_strings[k] = s;
} // keyboard::set_name_of()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get an iterator on the first pressed key.
 */
bear::input::keyboard::const_iterator bear::input::keyboard::begin() const
{
  return m_pressed_keys.begin();
} // bear::input::keyboard::begin()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get an iterator after the last pressed key.
 */
bear::input::keyboard::const_iterator bear::input::keyboard::end() const
{
  return m_pressed_keys.end();
} // bear::input::keyboard::end()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if no keys are pressed.
 */
bool bear::input::keyboard::empty() const
{
  return m_pressed_keys.empty();
} // bear::input::keyboard::empty()

/*----------------------------------------------------------------------------*/
/**
 * \brief Re-read the status of all keys.
 * \pre The caller is an instance of bear::input::system.
 */
void bear::input::keyboard::refresh()
{
  int num_keys;
  Uint8* keys;

  keys = SDL_GetKeyState( &num_keys );
  m_pressed_keys.clear();

  for (unsigned int i=0; i!=(unsigned int)num_keys; ++i)
    if ( keys[i] )
      {
        SDLMod mod = SDL_GetModState();
        key_code k = sdl_key_to_local(i, mod & KMOD_SHIFT, mod & KMOD_ALT);

        if ( (k != kc_not_a_key) &&
             (k != kc_num_lock) &&
             (k != kc_caps_lock) &&
             (k != kc_scroll_lock) )
          m_pressed_keys.push_back( k );
      }
} // keyboard::refresh()

/*----------------------------------------------------------------------------*/
/**
 * \brief Convert a SDLK_* value to the corresponding key_code.
 * \param sdl_val The SDL value to convert.
 * \param shift Tell if a shift button is considered pressed.
 * \param alt Tell if an alt button is considered pressed.
 */
bear::input::keyboard::key_code bear::input::keyboard::sdl_key_to_local
( unsigned int sdl_val, bool shift, bool alt ) const
{
  switch(sdl_val)
    {
    case SDLK_BACKSPACE : return kc_backspace; break;
    case SDLK_TAB       : return kc_tab;       break;
    case SDLK_CLEAR     : return kc_new_page;  break;
    case SDLK_RETURN    : return kc_new_line;  break;
    case SDLK_PAUSE     : return kc_pause;     break;
    case SDLK_ESCAPE    : return kc_escape;    break;

    case SDLK_SPACE      : return kc_space;             break;
    case SDLK_EXCLAIM    : return kc_exclaim;           break;
    case SDLK_QUOTEDBL   : return kc_double_quotes;     break;
    case SDLK_HASH       : return kc_hash;              break;
    case SDLK_DOLLAR     : return kc_dollar;            break;
    case SDLK_AMPERSAND  : return kc_ampersand;         break;
    case SDLK_QUOTE      : return kc_quote;             break;
    case SDLK_LEFTPAREN  : return kc_left_parenthesis;  break;
    case SDLK_RIGHTPAREN : return kc_right_parenthesis; break;
    case SDLK_ASTERISK   : return kc_asterisk;          break;
    case SDLK_PLUS       : return kc_plus;              break;
    case SDLK_COMMA      : return kc_comma;             break;
    case SDLK_MINUS      : return kc_minus;             break;
    case SDLK_PERIOD     : return kc_period;            break;
    case SDLK_SLASH      : return kc_slash;             break;

    case SDLK_0 : return kc_0; break;
    case SDLK_1 : return kc_1; break;
    case SDLK_2 : return kc_2; break;
    case SDLK_3 : return kc_3; break;
    case SDLK_4 : return kc_4; break;
    case SDLK_5 : return kc_5; break;
    case SDLK_6 : return kc_6; break;
    case SDLK_7 : return kc_7; break;
    case SDLK_8 : return kc_8; break;
    case SDLK_9 : return kc_9; break;

    case SDLK_COLON        : return kc_colon;         break;
    case SDLK_SEMICOLON    : return kc_semicolon;     break;
    case SDLK_LESS         : return kc_less;          break;
    case SDLK_EQUALS       : return kc_equal;         break;
    case SDLK_GREATER      : return kc_greater;       break;
    case SDLK_QUESTION     : return kc_question;      break;
    case SDLK_AT           : return kc_at;            break;
    case SDLK_LEFTBRACKET  : return kc_left_bracket;  break;
    case SDLK_BACKSLASH    : return kc_backslash;     break;
    case SDLK_RIGHTBRACKET : return kc_right_bracket; break;
    case SDLK_CARET        : return kc_caret;         break;
    case SDLK_UNDERSCORE   : return kc_underscore;    break;
    case SDLK_BACKQUOTE    : return kc_backquote;     break;

    case SDLK_a : if (shift) return kc_A; else return kc_a; break;
    case SDLK_b : if (shift) return kc_B; else return kc_b; break;
    case SDLK_c : if (shift) return kc_C; else return kc_c; break;
    case SDLK_d : if (shift) return kc_D; else return kc_d; break;
    case SDLK_e : if (shift) return kc_E; else return kc_e; break;
    case SDLK_f : if (shift) return kc_F; else return kc_f; break;
    case SDLK_g : if (shift) return kc_G; else return kc_g; break;
    case SDLK_h : if (shift) return kc_H; else return kc_h; break;
    case SDLK_i : if (shift) return kc_I; else return kc_i; break;
    case SDLK_j : if (shift) return kc_J; else return kc_j; break;
    case SDLK_k : if (shift) return kc_K; else return kc_k; break;
    case SDLK_l : if (shift) return kc_L; else return kc_l; break;
    case SDLK_m : if (shift) return kc_M; else return kc_m; break;
    case SDLK_n : if (shift) return kc_N; else return kc_n; break;
    case SDLK_o : if (shift) return kc_O; else return kc_o; break;
    case SDLK_p : if (shift) return kc_P; else return kc_p; break;
    case SDLK_q : if (shift) return kc_Q; else return kc_q; break;
    case SDLK_r : if (shift) return kc_R; else return kc_r; break;
    case SDLK_s : if (shift) return kc_S; else return kc_s; break;
    case SDLK_t : if (shift) return kc_T; else return kc_t; break;
    case SDLK_u : if (shift) return kc_U; else return kc_u; break;
    case SDLK_v : if (shift) return kc_V; else return kc_v; break;
    case SDLK_w : if (shift) return kc_W; else return kc_w; break;
    case SDLK_x : if (shift) return kc_X; else return kc_x; break;
    case SDLK_y : if (shift) return kc_Y; else return kc_y; break;
    case SDLK_z : if (shift) return kc_Z; else return kc_z; break;

    case SDLK_DELETE : return kc_delete; break;

    case SDLK_KP0 : return kc_keypad_0; break;
    case SDLK_KP1 : return kc_keypad_1; break;
    case SDLK_KP2 : return kc_keypad_2; break;
    case SDLK_KP3 : return kc_keypad_3; break;
    case SDLK_KP4 : return kc_keypad_4; break;
    case SDLK_KP5 : return kc_keypad_5; break;
    case SDLK_KP6 : return kc_keypad_6; break;
    case SDLK_KP7 : return kc_keypad_7; break;
    case SDLK_KP8 : return kc_keypad_8; break;
    case SDLK_KP9 : return kc_keypad_9; break;
    case SDLK_KP_PERIOD   : return kc_keypad_period;   break;
    case SDLK_KP_DIVIDE   : return kc_keypad_divide;   break;
    case SDLK_KP_MULTIPLY : return kc_keypad_multiply; break;
    case SDLK_KP_MINUS    : return kc_keypad_minus;    break;
    case SDLK_KP_PLUS     : return kc_keypad_plus;     break;
    case SDLK_KP_ENTER    : return kc_keypad_enter;    break;
    case SDLK_KP_EQUALS   : return kc_keypad_equals;   break;

    case SDLK_UP    : return kc_up;    break;
    case SDLK_DOWN  : return kc_down;  break;
    case SDLK_RIGHT : return kc_right; break;
    case SDLK_LEFT  : return kc_left;  break;

    case SDLK_INSERT   : return kc_insert;    break;
    case SDLK_HOME     : return kc_home;      break;
    case SDLK_END      : return kc_end;       break;
    case SDLK_PAGEUP   : return kc_page_up;   break;
    case SDLK_PAGEDOWN : return kc_page_down; break;

    case SDLK_F1  : return kc_F1;  break;
    case SDLK_F2  : return kc_F2;  break;
    case SDLK_F3  : return kc_F3;  break;
    case SDLK_F4  : return kc_F4;  break;
    case SDLK_F5  : return kc_F5;  break;
    case SDLK_F6  : return kc_F6;  break;
    case SDLK_F7  : return kc_F7;  break;
    case SDLK_F8  : return kc_F8;  break;
    case SDLK_F9  : return kc_F9;  break;
    case SDLK_F10 : return kc_F10; break;
    case SDLK_F11 : return kc_F11; break;
    case SDLK_F12 : return kc_F12; break;
    case SDLK_F13 : return kc_F13; break;
    case SDLK_F14 : return kc_F14; break;
    case SDLK_F15 : return kc_F15; break;

    case SDLK_NUMLOCK   : return kc_num_lock;    break;
    case SDLK_CAPSLOCK  : return kc_caps_lock;   break;
    case SDLK_SCROLLOCK : return kc_scroll_lock; break;

    case SDLK_RSHIFT : return kc_right_shift;   break;
    case SDLK_LSHIFT : return kc_left_shift;    break;
    case SDLK_RCTRL  : return kc_right_control; break;
    case SDLK_LCTRL  : return kc_left_control;  break;
    case SDLK_RALT   : return kc_right_alt;     break;
    case SDLK_LALT   : return kc_left_alt;      break;
    case SDLK_LSUPER : return kc_left_super;    break;
    case SDLK_RSUPER : return kc_right_super;   break;

    case SDLK_PRINT  : return kc_print_screen; break;
    case SDLK_SYSREQ : return kc_system;       break;
    case SDLK_BREAK  : return kc_break;        break;
    case SDLK_MENU   : return kc_menu;         break;
    default: return kc_not_a_key;
    }

} // keyboard::sdl_key_to_local()

/*----------------------------------------------------------------------------*/
/**
 * \brief Fill the m_key_string table with default strings.
 */
void bear::input::keyboard::default_key_strings()
{
  s_key_strings.resize(c_key_codes_count);
  std::fill( s_key_strings.begin(), s_key_strings.end(), "Undefined" );

  s_key_strings[kc_backspace] = "backspace";
  s_key_strings[kc_tab] = "tab";
  s_key_strings[kc_new_line] = "new line";

  s_key_strings[kc_escape] = "escape";
  s_key_strings[kc_space] = "space";
  s_key_strings[kc_exclaim] = "!";
  s_key_strings[kc_double_quotes] = "\"";
  s_key_strings[kc_hash] = "#";
  s_key_strings[kc_dollar] = "$";
  s_key_strings[kc_percent] = "%";
  s_key_strings[kc_ampersand] = "&";
  s_key_strings[kc_quote] = "'";
  s_key_strings[kc_left_parenthesis] = "(";
  s_key_strings[kc_right_parenthesis] = ")";
  s_key_strings[kc_asterisk] = "*";
  s_key_strings[kc_plus] = "+";
  s_key_strings[kc_comma] = ",";
  s_key_strings[kc_minus] = "-";
  s_key_strings[kc_period] = ".";
  s_key_strings[kc_slash] = "/";

  s_key_strings[kc_0] = "0";
  s_key_strings[kc_1] = "1";
  s_key_strings[kc_2] = "2";
  s_key_strings[kc_3] = "3";
  s_key_strings[kc_4] = "4";
  s_key_strings[kc_5] = "5";
  s_key_strings[kc_6] = "6";
  s_key_strings[kc_7] = "7";
  s_key_strings[kc_8] = "8";
  s_key_strings[kc_9] = "9";

  s_key_strings[kc_colon] = ":";
  s_key_strings[kc_semicolon] = ";";
  s_key_strings[kc_less] = "<";
  s_key_strings[kc_equal] = "=";
  s_key_strings[kc_greater] = ">";
  s_key_strings[kc_question] = "?";
  s_key_strings[kc_at] = "@";

  s_key_strings[kc_A] = "A";
  s_key_strings[kc_B] = "B";
  s_key_strings[kc_C] = "C";
  s_key_strings[kc_D] = "D";
  s_key_strings[kc_E] = "E";
  s_key_strings[kc_F] = "F";
  s_key_strings[kc_G] = "G";
  s_key_strings[kc_H] = "H";
  s_key_strings[kc_I] = "I";
  s_key_strings[kc_J] = "J";
  s_key_strings[kc_K] = "K";
  s_key_strings[kc_L] = "L";
  s_key_strings[kc_M] = "M";
  s_key_strings[kc_N] = "N";
  s_key_strings[kc_O] = "O";
  s_key_strings[kc_P] = "P";
  s_key_strings[kc_Q] = "Q";
  s_key_strings[kc_R] = "R";
  s_key_strings[kc_S] = "S";
  s_key_strings[kc_T] = "T";
  s_key_strings[kc_U] = "U";
  s_key_strings[kc_V] = "V";
  s_key_strings[kc_W] = "W";
  s_key_strings[kc_X] = "X";
  s_key_strings[kc_Y] = "Y";
  s_key_strings[kc_Z] = "Z";

  s_key_strings[kc_left_bracket] = "[";
  s_key_strings[kc_backslash] = "\\";
  s_key_strings[kc_right_bracket] = "]";
  s_key_strings[kc_caret] = "^";
  s_key_strings[kc_underscore] = "_";
  s_key_strings[kc_backquote] = "`";

  s_key_strings[kc_a] = "a";
  s_key_strings[kc_b] = "b";
  s_key_strings[kc_c] = "c";
  s_key_strings[kc_d] = "d";
  s_key_strings[kc_e] = "e";
  s_key_strings[kc_f] = "f";
  s_key_strings[kc_g] = "g";
  s_key_strings[kc_h] = "h";
  s_key_strings[kc_i] = "i";
  s_key_strings[kc_j] = "j";
  s_key_strings[kc_k] = "k";
  s_key_strings[kc_l] = "l";
  s_key_strings[kc_m] = "m";
  s_key_strings[kc_n] = "n";
  s_key_strings[kc_o] = "o";
  s_key_strings[kc_p] = "p";
  s_key_strings[kc_q] = "q";
  s_key_strings[kc_r] = "r";
  s_key_strings[kc_s] = "s";
  s_key_strings[kc_t] = "t";
  s_key_strings[kc_u] = "u";
  s_key_strings[kc_v] = "v";
  s_key_strings[kc_w] = "w";
  s_key_strings[kc_x] = "x";
  s_key_strings[kc_y] = "y";
  s_key_strings[kc_z] = "z";

  s_key_strings[kc_left_brace] = "{";
  s_key_strings[kc_vertical_line] = "|";
  s_key_strings[kc_right_brace] = "}";
  s_key_strings[kc_tilde] = "~";

  s_key_strings[kc_keypad_0] = "keypad 0";
  s_key_strings[kc_keypad_1] = "keypad 1";
  s_key_strings[kc_keypad_2] = "keypad 2";
  s_key_strings[kc_keypad_3] = "keypad 3";
  s_key_strings[kc_keypad_4] = "keypad 4";
  s_key_strings[kc_keypad_5] = "keypad 5";
  s_key_strings[kc_keypad_6] = "keypad 6";
  s_key_strings[kc_keypad_7] = "keypad 7";
  s_key_strings[kc_keypad_8] = "keypad 8";
  s_key_strings[kc_keypad_9] = "keypad 9";

  s_key_strings[kc_keypad_period] = "keypad .";
  s_key_strings[kc_keypad_divide] = "keypad /";
  s_key_strings[kc_keypad_multiply] = "keypad *";
  s_key_strings[kc_keypad_minus] = "keypad -";
  s_key_strings[kc_keypad_plus] = "keypad +";
  s_key_strings[kc_keypad_enter] = "keypad enter";
  s_key_strings[kc_keypad_equals] = "keypad =";

  s_key_strings[kc_up] = "up";
  s_key_strings[kc_down] = "down";
  s_key_strings[kc_right] = "right";
  s_key_strings[kc_left] = "left";

  s_key_strings[kc_insert] = "insert";
  s_key_strings[kc_home] = "home";
  s_key_strings[kc_end] = "end";
  s_key_strings[kc_page_up] = "page up";
  s_key_strings[kc_page_down] = "page down";

  s_key_strings[kc_F1] = "F1";
  s_key_strings[kc_F2] = "F2";
  s_key_strings[kc_F3] = "F3";
  s_key_strings[kc_F4] = "F4";
  s_key_strings[kc_F5] = "F5";
  s_key_strings[kc_F6] = "F6";
  s_key_strings[kc_F7] = "F7";
  s_key_strings[kc_F8] = "F8";
  s_key_strings[kc_F9] = "F9";
  s_key_strings[kc_F10] = "F10";
  s_key_strings[kc_F11] = "F11";
  s_key_strings[kc_F12] = "F12";
  s_key_strings[kc_F13] = "F13";
  s_key_strings[kc_F14] = "F14";
  s_key_strings[kc_F15] = "F15";

  s_key_strings[kc_num_lock] = "numlock";
  s_key_strings[kc_caps_lock] = "capslock";
  s_key_strings[kc_scroll_lock] = "scrolllock";
  s_key_strings[kc_right_shift] = "right shift";
  s_key_strings[kc_left_shift] = "left shift";
  s_key_strings[kc_right_control] = "right control";
  s_key_strings[kc_left_control] = "left control";
  s_key_strings[kc_right_alt] = "right alt";
  s_key_strings[kc_left_alt] = "left alt";
  s_key_strings[kc_right_super] = "right super";
  s_key_strings[kc_left_super] = "left super";
  s_key_strings[kc_print_screen] = "print screen";
  s_key_strings[kc_system] = "system";
  s_key_strings[kc_break] = "break";
  s_key_strings[kc_menu] = "menu";
} // keyboard::default_key_strings()
