/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file collision_detection.cpp
 * \brief Implementation of the bear::universe::collision_detection class.
 * \author Julien Jorge
 */
#include "universe/collision_detection.hpp"

#include <algorithm>
#include <claw/functional.hpp>

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param f The item to store.
 * \param s The initial state of the item.
 */
bear::universe::collision_detection::item_set_item::item_set_item
( item_set_item::first_type f, item_set_item::second_type s )
  : super(f, s)
{

} // collision_detection::item_set_item::item_set_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Compare two item_set_item by decreasing lexicographic order of the
 *        position of their first field.
 * \param that The item to compare two.
 */
bool bear::universe::collision_detection::item_set_item::operator<
  ( const item_set_item& that ) const
{
  return first < that.first;
} // collision_detection::item_set_item::operator<()




/*----------------------------------------------------------------------------*/
/**
 * \brief Set the items for the next collision check.
 * \param static_items The static items : they don't move.
 * \param mobile_items The mobile items : they move.
 */
void bear::universe::collision_detection::add_items(const item_list& items)
{
  item_list::const_iterator it;

  for (it=items.begin(); it!=items.end(); ++it)
    if ( (*it)->is_fixed() ) 
      if ( (*it)->can_move_items() ) 
        m_fixed_move_items.insert( item_set_item(*it, **it) );
      else
        m_fixed_not_move_items.insert( item_set_item(*it, **it) );
    else
      if ( (*it)->can_move_items() ) 
        m_mobile_move_items.insert( item_set_item(*it, **it) );
      else
        m_mobile_not_move_items.insert( item_set_item(*it, **it) );
} // collision_detection::add_static_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Check for collisions.
 */
void bear::universe::collision_detection::detect_collisions()
{
  item_set::const_iterator it;
  bool loop;
  
  do
    {
      loop = false;
      for(it=m_mobile_move_items.begin(); 
          it!=m_mobile_move_items.end(); ++it)
        {
          bool fixed_corrected = detect_collision_all( *it,
                                                       m_fixed_move_items );
          bool mobile_corrected = detect_collision_remaining( it, 
                                                      m_mobile_move_items );

          loop = loop || fixed_corrected || mobile_corrected;
        }
    }
  while (loop);

  for(it=m_mobile_move_items.begin(); 
      it!=m_mobile_move_items.end(); ++it)
    detect_collision_all( *it, m_fixed_not_move_items );
    
  do
    {
      loop = false;
      for(it=m_mobile_not_move_items.begin(); 
          it!=m_mobile_not_move_items.end(); ++it)
        {
          bool fixed_corrected = detect_collision_all( *it,
                                                        m_fixed_move_items );
          bool mobile_corrected = detect_collision_all( *it, 
                                                      m_mobile_move_items );

          loop = loop || fixed_corrected || mobile_corrected;
        }
    }
  while (loop);

  for(it=m_mobile_not_move_items.begin(); 
      it!=m_mobile_not_move_items.end(); ++it)
    {
      detect_collision_all( *it, m_fixed_not_move_items );
      detect_collision_remaining( it, m_mobile_not_move_items );
    }

  final_collision_detection();
  
  clear_lists();  
} // collision_detection::progress_positions()

/*----------------------------------------------------------------------------*/
/**
 * \brief Remove all items from the last progression.
 */
void bear::universe::collision_detection::clear_lists()
{
  m_fixed_move_items.clear();
  m_fixed_not_move_items.clear();
  m_mobile_move_items.clear();
  m_mobile_not_move_items.clear();
  m_corrected.clear();
} // collision_detection::clear_lists()


/*----------------------------------------------------------------------------*/
/**
 * \brief Check for collisions of an item with all items of the list.
 * \param item Iterator on the current tested item 
 *             in the m_mobile_solid_item list.
 * \param items List of items. 
 * \param test_phantom Indictaes if we realise the phantom test.
 * \remark Tests are made only with the items of the list after "item".
 */
bool bear::universe::collision_detection::detect_collision_remaining
( const item_set::const_iterator& item, item_set& items)
{
  item_set::const_iterator it;
  bool result = false;

  it=item;
  
  for (++it; it!=items.end(); ++it)
    if ( m_corrected.find( item_pair(item->first, it->first) )
         == m_corrected.end() )
      {
        // we need a temporary boolean to be sure that detect_collision is
        // called
        bool r = detect_collision( *item, *it );
        result = result || r;
      }
  
  return result;
} // collision_detection::detect_collision_remaining()


/*----------------------------------------------------------------------------*/
/**
 * \brief Check for collisions of an item with each item of a list.
 * \param item The item currently tested
 * \param items List of items. 
 * \param test_phantom Indictaes if we realise the phantom test.
*/
bool bear::universe::collision_detection::detect_collision_all
( const item_set_item& item, item_set& items )
{
  bool result = false;

  item_set::const_iterator it;
  
  for (it=items.begin(); 
       it!=items.end(); ++it)
    if ( m_corrected.find( item_pair(item.first, it->first) )
         == m_corrected.end())
      {
        // we need a temporary boolean to be sure that detect_collision is
        // called
        bool r = detect_collision( item, *it );
        result = result || r;
      }

  return result;
} // collision_detection::detect_collision_all()

/*----------------------------------------------------------------------------*/
/**
 * \brief Teste if it stay collision.
 */
void bear::universe::collision_detection::final_collision_detection()
{
  claw::avl<item_pair>::const_iterator it;

  for ( it = m_corrected.begin(); it != m_corrected.end(); ++it) 
    {
      claw::math::rectangle<coordinate_type> bounding_box_1
        ( it->first->get_bounding_box() );
      claw::math::rectangle<coordinate_type> bounding_box_2
        ( it->second->get_bounding_box() );

      if ( bounding_box_1.intersects(bounding_box_2) )
        {
          if ( (! it->first->is_fixed()) && it->second->can_move_items() )
            it->first->set_contact_after_collision(true);

          if ( (! it->second->is_fixed()) && it->first->can_move_items() )
            it->second->set_contact_after_collision(true);
        }
    }
} // collision_detection::final_collision_detection()

/*----------------------------------------------------------------------------*/
/**
 * \brief Check if a collision occurs between two items and call the
 *        apply_collision() method if any.
 * \param item_1 The first item in the collision.
 * \param item_2 The other item.
 */
bool bear::universe::collision_detection::detect_collision
( const item_set_item& item_1, const item_set_item& item_2 )
{
  claw::math::rectangle<coordinate_type> bounding_box_1
    ( item_1.first->get_bounding_box() );
  claw::math::rectangle<coordinate_type> bounding_box_2
    ( item_2.first->get_bounding_box() );
  bool result = false;

  if ( bounding_box_1.intersects(bounding_box_2) )
    {
      apply_collision( item_1, item_2 );
      apply_collision( item_2, item_1 );

      result = true;
      m_corrected.insert( item_pair(item_1.first, item_2.first) );
    }

  return result;
} // collision_detection::detect_collision_static()

/*----------------------------------------------------------------------------*/
/**
 * \brief Call the bear::universe::physical_item::collision() method on an item.
 * \param item_1 The item on which we call
 *               bear::universe::physical_item::collision().
 * \param item_2 The other item.
 */
void bear::universe::collision_detection::apply_collision
( const item_set_item& item_1, const item_set_item& item_2 ) const
{
  item_1.first->collision( *item_2.first, item_1.second, item_2.second );
} // collision_detection::collision()
