/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
#include "universe/forced_train.hpp"

#include <iostream>

/*---------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param moving_item The controlled item.
 * \param targets List of the targets of the path.
 */
bear::universe::forced_train::forced_train
( physical_item& moving_item, const target_list& targets )
  : forced_movement( moving_item ), m_targets( targets ),
    m_current_target(0), m_index(0),m_playing(false),
    m_loops(0), m_loop_back(true), m_forward(true), m_play_count(0)
{
  m_origin_position = m_targets[0].first->get_center_of_mass();

  if (m_targets.size() > 1 )
    m_current_target = 1;
} // forced_train::forced_train()

/*---------------------------------------------------------------------------*/
/**
 * \brief Stop the movement.
 */
void bear::universe::forced_train::stop()
{
  m_playing = false;
} // forced_train::stop()

/*---------------------------------------------------------------------------*/
/**
 * \brief Do an iteration of the movement.
 */
void bear::universe::forced_train::next_position()
{
  if ( m_playing && ( (m_play_count != m_loops) || !m_loops ) )
    {
      position_type pos;

      ++m_index;
      
      if ( m_index == m_targets[m_current_target].second )
        {
          m_index = 0;
          if (m_forward)
            next_forward( pos );
          else
            next_backward( pos );
	      }
      else
        next( pos );
			
      m_moving_item.set_center_of_mass(pos);
    }
} // forced_train::progress()

/*---------------------------------------------------------------------------*/
/**
 * \brief Start the movement.
 */
void bear::universe::forced_train::start()
{
  m_playing = true;
} // forced_train::start()

/*---------------------------------------------------------------------------*/
/**
 * \brief Set loop back mode.
 */
void bear::universe::forced_train::set_loop_back( bool loop_back )
{
  m_loop_back = loop_back;
} // forced_train::start()

/*---------------------------------------------------------------------------*/
/**
 * \brief Set the number of times the path should be take.
 */
void bear::universe::forced_train::set_loops( unsigned int loops )
{
  m_loops = loops;
} // forced_train::start()

/*---------------------------------------------------------------------------*/
/**
 * \brief Go to the next position, forward playing.
 * \param pos (out) The resulting position.
 */
void bear::universe::forced_train::next_forward( position_type& pos )
{
  m_origin_position = m_targets[m_current_target].first->get_center_of_mass();
  pos = m_origin_position;
  ++m_current_target;

  if ( m_current_target == m_targets.size() ) // we're after the end
    {
      if ( m_loop_back ) // should we play reverse ?
        {
          m_forward = false;

          if (m_current_target >= 2)
            m_current_target -= 2;
          else
            m_current_target = 0;
          next(pos);
        }
      else // come back directly to the first frame
        {
          m_current_target = 0;
          ++m_play_count;

          if ( (m_play_count != m_loops) || !m_loops )
            next(pos);
        }
    }
} // forced_train::next_forward()

/*---------------------------------------------------------------------------*/
/**
 * \brief Go to the next position, backward playing.
 * \param pos (out) The resulting position.
 */
void bear::universe::forced_train::next_backward( position_type& pos )
{
  m_origin_position = m_targets[m_current_target].first->get_center_of_mass();
  pos = m_origin_position;
  if ( m_current_target == 0 ) // we're at the begining
    {
      ++m_play_count;

      if ( m_targets.size() > 1 )
        {
          m_forward = true;

          if ( (m_play_count != m_loops) || !m_loops )
            {
              ++m_current_target;
              next(pos);
            }
        }
    }
  else
    --m_current_target;
} // forced_train::next_backward()

/*---------------------------------------------------------------------------*/
/**
 * \brief Go to the next position.
 * \param pos (out) The resulting position.
 */
void bear::universe::forced_train::next( position_type& pos )
{
  position_type center;
  position_type position;

  center = m_targets[m_current_target].first->get_center_of_mass();
  position.x = m_origin_position.x +
    ( ( center.x - m_origin_position.x ) * m_index )
    / m_targets[m_current_target].second;

  position.y = m_origin_position.y +
    ( ( center.y - m_origin_position.y ) * m_index )
    / m_targets[m_current_target].second;

  pos = position ;
} // forced_train::next()
