/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file physical_item_state.cpp
 * \brief Implementation of the bear::universe::physical_item_state class.
 * \author Julien Jorge
 */
#include "universe/physical_item_state.hpp"
#include "universe/physical_item.hpp"

#include "additional_stream_formatting.hpp"

#include <sstream>

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
bear::universe::physical_item_state::physical_item_state()
  : m_mass(0), m_speed(0, 0), m_acceleration(0, 0),
    m_gravity(0, 2), m_friction(0.92), m_environment_friction(1), 
    m_position(0, 0), 
    m_size(0, 0), m_angle(0), m_fixed(false), m_can_move_items(true), 
    m_contact_after_collision(false), m_left_contact(false), 
    m_right_contact(false), m_top_contact(false), m_bottom_contact(false), 
    m_is_phantom(false), m_speed_epsilon(0.01,0.01)
{
} // physical_item_state::physical_item_state()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
bear::universe::physical_item_state::~physical_item_state()
{
  // nothing to do
} // physical_item_state::~physical_item_state()

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply one more force to the item.
 * \param force The force to apply.
 */
void bear::universe::physical_item_state::add_force( const force_type& force )
{
  if (!m_fixed)
    m_acceleration += force / get_mass();
} // physical_item_state::add_force()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the size of the item.
 */
const bear::universe::size_box_type&
bear::universe::physical_item_state::get_size() const
{
  return m_size;
} // physical_item_state::get_size()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the width of the item.
 */
bear::universe::size_type bear::universe::physical_item_state::get_width() const
{
  return m_size.x;
} // physical_item_state::get_width()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the height of the item.
 */
bear::universe::size_type
bear::universe::physical_item_state::get_height() const
{
  return m_size.y;
} // physical_item_state::get_height()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the highest X-coordinate covered by the bounding box of this item.
 */
bear::universe::coordinate_type
bear::universe::physical_item_state::get_right() const
{
  return m_position.x + m_size.x;
} // physical_item_state::get_right()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the highest Y-coordinate covered by the bounding box of this item.
 */
bear::universe::coordinate_type
bear::universe::physical_item_state::get_bottom() const
{
  return m_position.y + m_size.y;
} // physical_item_state::get_bottom()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the bouding box of this item.
 */
bear::universe::rectangle_type
bear::universe::physical_item_state::get_bounding_box() const
{
  return rectangle_type(m_position, m_size);
} // physical_item_state::get_bounding_box()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the current acceleration of the item.
 */
const bear::universe::force_type&
bear::universe::physical_item_state::get_acceleration() const
{
  return m_acceleration;
} // physical_item_state::get_acceleration()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the acceleration of the item.
 */
void bear::universe::physical_item_state::set_acceleration(const force_type& a)
{
  if (!m_fixed)
    m_acceleration = a;
} // physical_item_state::set_acceleration()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the mass of the item.
 */
double bear::universe::physical_item_state::get_mass() const
{
  return m_mass;
} // physical_item_state::get_mass()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the current speed of the item.
 */
const bear::universe::speed_type&
bear::universe::physical_item_state::get_speed() const
{
  return m_speed;
} // physical_item_state::get_speed()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the current speed of the item.
 */
void bear::universe::physical_item_state::set_speed( const speed_type& speed )
{
  if (!m_fixed)
    {
      m_speed = speed;
      if ( (m_speed.x < m_speed_epsilon.x) && 
           (m_speed.x > -m_speed_epsilon.x) )
        m_speed.x = 0;
      if ( (m_speed.y < m_speed_epsilon.y) && 
           (m_speed.y > -m_speed_epsilon.y) )
        m_speed.y = 0;
    }
} // physical_item_state::set_speed()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the current speed of the item.
 */
const bear::universe::speed_type& 
bear::universe::physical_item_state::get_speed_epsilon() const
{
  return m_speed_epsilon;
} // physical_item_state::get_speed_epsilon()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the current speed of the item.
 */
void bear::universe::physical_item_state::set_speed_epsilon
( const speed_type& speed )
{
  m_speed_epsilon = speed;
} // physical_item_state::set_speed_epsilon()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the current position of the item.
 */
const bear::universe::position_type&
bear::universe::physical_item_state::get_position() const
{
  return m_position;
} // physical_item_state::get_position()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set (force) the position of the item.
 * \param pos The new position of the item.
 */
void
bear::universe::physical_item_state::set_position( const position_type& pos )
{
  if (!m_fixed)
    m_position = pos;
} // physical_item_state::set_position()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set (force) the position of the item.
 * \param x The x-coordinate of the item.
 * \param y The y-coordinate of the item.
 */
void bear::universe::physical_item_state::set_position
( coordinate_type x, coordinate_type y )
{
  if (!m_fixed)
    m_position.set(x, y);
} // physical_item_state::set_position()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the gravity applied to this item.
 */
const bear::universe::force_type&
bear::universe::physical_item_state::get_gravity() const
{
  return m_gravity;
} // physical_item_state::get_gravity()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the gravity applied to this item.
 * \param g The value of the gravity applied to this item.
 */
void bear::universe::physical_item_state::set_gravity( const force_type& g )
{
  m_gravity = g;
} // physical_item_state::set_gravity();

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the friction applied to this item.
 */
double bear::universe::physical_item_state::get_friction() const
{
  return m_friction;
} // physical_item_state::get_friction();

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the friction applied to this item.
 * \param f The value of the friction applied to this item.
 */
void bear::universe::physical_item_state::set_friction( double f )
{
  m_friction = f;
} // physical_item_state::set_friction()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the friction applied to this item by its environment.
 */
double bear::universe::physical_item_state::get_environment_friction() const
{
  return m_environment_friction;
} // physical_item_state::get_environment_friction();

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the friction applied to this item by its environment.
 * \param f The value of the friction applied to this item.
 */
void bear::universe::physical_item_state::set_environment_friction( double f )
{
  m_environment_friction = f;
} // physical_item_state::set_environment_friction()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the angle of this item.
 */
double bear::universe::physical_item_state::get_angle() const
{
  return m_angle;
} // physical_item_state::get_angle();

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the angle of this item.
 * \param a The value of the angle.
 */
void bear::universe::physical_item_state::set_angle( double a )
{
  m_angle = a;
} // physical_item_state::set_angle()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set (force) the position of the middle of the top edge of the item.
 * \param pos The new position.
 */
void
bear::universe::physical_item_state::set_top_middle( const position_type& pos )
{
  if (!m_fixed)
    m_position.set( pos.x - m_size.x / 2, pos.y );
} // physical_item_state::set_top_middle()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set (force) the position of the top right corner of the item.
 * \param pos The new position.
 */
void
bear::universe::physical_item_state::set_top_right( const position_type& pos )
{
  if (!m_fixed)
    m_position.set( pos.x - m_size.x, pos.y );
} // physical_item_state::set_top_right()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set (force) the position of the bottom left corner of the item.
 * \param pos The new position.
 */
void
bear::universe::physical_item_state::set_bottom_left( const position_type& pos )
{
  if (!m_fixed)
    m_position.set( pos.x, pos.y - m_size.y);
} // physical_item_state::set_bottom_left()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set (force) the position of the middle of the bottom edge of the item.
 * \param pos The new position.
 */
void bear::universe::physical_item_state::set_bottom_middle
( const position_type& pos )
{
  if (!m_fixed)
    m_position.set( pos.x - m_size.x / 2, pos.y - m_size.y);
} // physical_item_state::set_bottom_middle()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set (force) the position of the bottom right corner of the item.
 * \param pos The new position.
 */
void bear::universe::physical_item_state::set_bottom_right
( const position_type& pos )
{
  if (!m_fixed)
    m_position.set( pos.x - m_size.x,  pos.y - m_size.y );
} // physical_item_state::set_bottom_right()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set (force) the position of the middle of the left edge of the item.
 * \param pos The new position.
 */
void
bear::universe::physical_item_state::set_left_middle( const position_type& pos )
{
  if (!m_fixed)
    m_position.set( pos.x, pos.y - m_size.y / 2 );
} // physical_item_state::set_left_middle()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set (force) the position of the middle of the right edge of the item.
 * \param pos The new position.
 */
void bear::universe::physical_item_state::set_right_middle
( const position_type& pos )
{
  if (!m_fixed)
    m_position.set( pos.x - m_size.x, pos.y - m_size.y / 2 );
} // physical_item_state::set_right_middle()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the position of the middle of the top edge of the item.
 */
bear::universe::position_type
 bear::universe::physical_item_state::get_top_middle() const
{
  return position_type( get_center_of_mass().x, m_position.y );
} // physical_item_state::get_top_middle()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the position of the top right corner of the item.
 */
bear::universe::position_type
bear::universe::physical_item_state::get_top_right() const
{
  return position_type( get_right(), m_position.y );
} // physical_item_state::get_top_right()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the position of the bottom left corner of the item.
 */
bear::universe::position_type
bear::universe::physical_item_state::get_bottom_left() const
{
  return position_type( m_position.x, get_bottom() );
} // physical_item_state::get_bottom_left()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the position of the middle of the bottom edge of the item.
 */
bear::universe::position_type 
bear::universe::physical_item_state::get_bottom_middle() const
{
  return position_type( get_center_of_mass().x, get_bottom() );
} // physical_item_state::get_bottom_middle()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the position of the bottom right corner of the item.
 */
bear::universe::position_type
bear::universe::physical_item_state::get_bottom_right() const
{
  return position_type( get_right(), get_bottom() );
} // physical_item_state::get_bottom_right()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the position of the middle of the left edge of the item.
 */
bear::universe::position_type
bear::universe::physical_item_state::get_left_middle() const
{
  return position_type( m_position.x, get_center_of_mass().y );
} // physical_item_state::get_left_middle()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the position of the middle of the right edge of the item.
 */
bear::universe::position_type
bear::universe::physical_item_state::get_right_middle() const
{
  return position_type( get_right(), get_center_of_mass().y );
} // physical_item_state::get_right_middle()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the position, in the world, of the center of mass.
 */
bear::universe::position_type
bear::universe::physical_item_state::get_center_of_mass() const
{
  return m_position + m_size / 2;
} // physical_item_state::get_center_of_mass()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set (force) the position of the center of the item.
 */
void bear::universe::physical_item_state::set_center_of_mass
( const position_type& pos )
{
  if (!m_fixed)
    m_position = pos - m_size / 2;
} // physical_item_state::set_center_of_mass()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the item is fixed.
 */
bool bear::universe::physical_item_state::is_fixed() const
{
  return m_fixed;
} // physical_item_state::is_fixed()

/*----------------------------------------------------------------------------*/
/**
 * \brief Fix the item. Its speed and acceleration are set to zero. Its
 *        position, speed and acceleration won't be able to cange anymore.
 */
void bear::universe::physical_item_state::fix()
{
  m_acceleration = m_speed = speed_type(0, 0);
  m_fixed = true;
} // physical_item_state::fix()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the item can move items.
 */
bool bear::universe::physical_item_state::can_move_items() const
{
  return m_can_move_items;
} // physical_item_state::can_move_items()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the field can_move_items.
 */
void bear::universe::physical_item_state::set_can_move_items(bool value)
{
  m_can_move_items = value;
} // physical_item_state::set_can_move_items()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the item has a contact after collision detection.
 */
bool bear::universe::physical_item_state::has_contact_after_collision() const
{
  return m_contact_after_collision;
} // physical_item_state::has_contact_after_collision()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the field has_contact_after_collision.
 */
void
bear::universe::physical_item_state::set_contact_after_collision( bool contact )
{
  m_contact_after_collision = contact;
} // physical_item_state::set_contact_after_collision()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the item has a contact on its left.
 */
bool bear::universe::physical_item_state::has_left_contact() const
{
  return m_left_contact;
} // physical_item_state::has_left_contact()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the item has a contact on its right.
 */
bool bear::universe::physical_item_state::has_right_contact() const
{
  return m_right_contact;
} // physical_item_state::has_right_contact()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the item has a contact on its top.
 */
bool bear::universe::physical_item_state::has_top_contact() const
{
  return m_top_contact;
} // physical_item_state::has_top_contact()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the item has a contact on its bottom.
 */
bool bear::universe::physical_item_state::has_bottom_contact() const
{
  return m_bottom_contact;
} // physical_item_state::has_bottom_contact()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the item has a contact.
 */
bool bear::universe::physical_item_state::has_contact() const
{
  return m_bottom_contact || m_top_contact ||
    m_right_contact || m_left_contact;
} // physical_item_state::has_contact()


/*----------------------------------------------------------------------------*/
/**
 * \brief Tell that the item has a contact on its left.
 */
void
bear::universe::physical_item_state::set_left_contact(physical_item_state& item)
{
  set_contact(item);
  m_left_contact = true;
} // physical_item_state::set_left_contact()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell that the item has a contact on its right.
 */
void bear::universe::physical_item_state::set_right_contact
(physical_item_state& item)
{
  set_contact(item);
  m_right_contact = true;
} // physical_item_state::set_right_contact()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell that the item has a contact on its top.
 */
void
bear::universe::physical_item_state::set_top_contact(physical_item_state& item)
{
  set_contact(item);
  m_top_contact = true;
} // physical_item_state::set_top_contact()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell that the item has a contact on its bottom.
 */
void bear::universe::physical_item_state::set_bottom_contact
(physical_item_state& item)
{
  set_contact(item);
  m_bottom_contact = true;
} // physical_item_state::set_bottom_contact()


/*----------------------------------------------------------------------------*/
/**
 * \brief Tell that the item has a contact.
 */
void bear::universe::physical_item_state::set_contact(physical_item_state& item)
{
} // physical_item_state::set_contact()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell that the item has not any contact.
 */
void bear::universe::physical_item_state::clear_contacts()
{
  m_left_contact = m_right_contact = m_top_contact = m_bottom_contact = false;
  m_contact_after_collision = false;
} // physical_item_state::clear_contacts()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the phantom statut.
 * \param phantom The new statut.
 */
void bear::universe::physical_item_state::set_phantom( bool phantom )
{
  m_is_phantom = phantom;
} // physical_item_state::set_phantom()

/*----------------------------------------------------------------------------*/
/**
 * \brief Return the phantom statut.
 */
bool bear::universe::physical_item_state::is_phantom() const
{
  return m_is_phantom;
} // physical_item_state::is_phantom()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the size of the object.
 * \param size The new size.
 */
void bear::universe::physical_item_state::set_size( const size_box_type& size )
{
  m_size = size;
} // physical_item_state::set_size()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the size of the object.
 * \param width The width of the object.
 * \param height The height of the object.
 */
void bear::universe::physical_item_state::set_size
( size_type width, size_type height )
{
  set_size( size_box_type(width, height) );
} // physical_item_state::set_size()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the width of the object.
 * \param width The width of the object.
 */
void bear::universe::physical_item_state::set_width( size_type width )
{
  m_size.x = width;
} // physical_item_state::set_width()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the height of the object.
 * \param height The height of the object.
 */
void bear::universe::physical_item_state::set_height( size_type height )
{
  m_size.y = height;
} // physical_item_state::set_height()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the mass of the object.
 * \param m The new mass.
 */
void bear::universe::physical_item_state::set_mass( double m )
{
  m_mass = m;
} // physical_item_state::set_mass()

/*----------------------------------------------------------------------------*/
/**
 * \brief Give a string representation of the item.
 * \param str (out) The result of the convertion.
 */
void bear::universe::physical_item_state::to_string( std::string& str ) const
{
  text::convertible::to_string(str);
  std::ostringstream oss;

  oss << "\nmass: " << m_mass;
  oss << "\npos: " << m_position;
  oss << "\nsize: " << m_size;
  oss << "\nspeed: " << m_speed;
  oss << "\naccel: " << m_acceleration;
  oss << "\ngravity: " << m_gravity;
  oss << "\nfriction: " << m_friction;
  oss << "   environment_friction: " << m_environment_friction;
  oss << "\nangle: " << m_angle;
  oss << "\nfixed: " << m_fixed;
  oss << "\ncan move items: " << m_can_move_items;
  oss << "\ncontact after collision: " << m_contact_after_collision;
  oss << "\ncontact: { ";

  if ( m_left_contact )
    oss << "left ";
  if ( m_right_contact )
    oss << "right ";
  if ( m_top_contact )
    oss << "top ";
  if ( m_bottom_contact )
    oss << "bottom ";

  oss << "}";

  str += oss.str();
} // physical_item_state::to_string()
