/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file align_take_speed.tpp
 * \brief Implementation of the bear::universe::align_take_speed class.
 * \author Julien Jorge
 */

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param ratio The coefficient to apply to the other item's speed before taking
 *        it for us.
 */
template<class Alignment>
bear::universe::align_take_speed<Alignment>::align_take_speed( double ratio )
  : m_ratio(ratio)
{

} // align_take_speed::align_take_speed()

/*----------------------------------------------------------------------------*/
/**
 * \brief Align the other item and take a percentage of its speed.
 * \param info Informations on the collision.
 * \param self The first item in the collision.
 * \param that The item to align.
 */
template<class Alignment>
void bear::universe::align_take_speed<Alignment>::execute
( const collision_info& info, physical_item& self, physical_item& that ) const
{
  super::execute( info, self, that );

  if( !that.is_phantom() )
    {
      double coeff;

      if (info.get_collision_side() == zone::middle_left_zone)
	coeff = 1;
      else if (info.get_collision_side() == zone::middle_right_zone)
	coeff = -1;

      if ( (info.get_collision_side() == zone::middle_left_zone)
	   || (info.get_collision_side() == zone::middle_right_zone) )
	{
	  speed_type self_speed(self.get_speed());
	  speed_type that_speed(that.get_speed());
	
	  double diff = fabs(info.other_previous_state().get_speed().x)
	    - fabs(info.reference_previous_state().get_speed().y) * m_ratio;
	
	  if ( diff > 0 )
	    {
	      that_speed.x = coeff * std::min(diff, fabs(that_speed.x));
	      self_speed.x += coeff * diff;
	    }
	  else
	    that_speed.x = 0;
	
	  self.set_speed( self_speed );
	  that.set_speed( that_speed );
	}

      if (info.get_collision_side() == zone::top_zone)
	{
	  that.set_bottom_contact(self);
	  self.set_top_contact(that);
	
	  speed_type speed(that.get_speed().x, 0);
	  speed_type acceleration(that.get_acceleration().x, 0);
	
	  that.set_speed( speed );
	  that.set_acceleration( acceleration );
	
	  that.set_position
	    ( that.get_position()
	      + (self.get_position()
		 - info.reference_previous_state().get_position()) );
	}
    }
} // align_take_speed::execute()
