/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file elastic_link.cpp
 * \brief Implementation of the bear::universe::elastic_link class.
 * \author Julien Jorge
 */
#include "universe/link/elastic_link.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param first_item The first linked item.
 * \param second_item The second linked item.
 * \param strength The strength of the elastic.
 * \remark The elastic's length is set to the distance between the center of
 *         mass of each item.
 */
bear::universe::elastic_link::elastic_link
( physical_item& first_item, physical_item& second_item, double strength )
  : base_link(first_item, second_item), m_strength(strength),
    m_length( m_first_item->get_center_of_mass().distance
              ( m_second_item->get_center_of_mass() ) )
{

} // elastic_link::elastic_link()

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param first_item The first linked item.
 * \param second_item The second linked item.
 * \param strength The strength of the elastic.
 * \param length The length of the elastic.
 */
bear::universe::elastic_link::elastic_link
( physical_item& first_item, physical_item& second_item,
  double strength, bear::universe::coordinate_type length )
  : base_link(first_item, second_item), m_strength(strength), m_length(length)
{

} // elastic_link::elastic_link()

/*----------------------------------------------------------------------------*/
/**
 * \brief Update the forces applied to each item.
 */
void bear::universe::elastic_link::adjust()
{
  force_type force( m_second_item->get_center_of_mass(),
		    m_first_item->get_center_of_mass() );

  double d = force.length();

  if (d > m_length)
    {
      force_type a( force );
      a.normalize();

      a *= m_strength * (d - (double)m_length) / d;

      m_first_item->add_force(-a);
      m_second_item->add_force(a);
    }
} // elastic_link::adjust()
