/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file physical_item_state.hpp
 * \brief Important fields of the physical items.
 * \author Julien Jorge
 */
#ifndef __UNIVERSE_PHYSICAL_ITEM_STATE_HPP__
#define __UNIVERSE_PHYSICAL_ITEM_STATE_HPP__

#include "universe/types.hpp"
#include "text/conversion.hpp"

#include "universe/class_export.hpp"

namespace bear
{
  namespace universe
  {
    /**
     * \brief Important fields of the physical items.
     * \author Julien Jorge
     */
    class UNIVERSE_EXPORT physical_item_state:
      public text::convertible
    {
    public:
      physical_item_state();
      virtual ~physical_item_state();

      void add_force( const force_type& force );

      const size_box_type& get_size() const;
      size_type get_width() const;
      size_type get_height() const;

      coordinate_type get_right() const;
      coordinate_type get_bottom() const;

      universe::rectangle_type get_bounding_box() const;

      const force_type& get_acceleration() const;
      void set_acceleration( const force_type& a );
      double get_mass() const;
      void set_mass( double m );

      const speed_type& get_speed() const;
      void set_speed( const speed_type& speed );

      const speed_type& get_speed_epsilon() const;
      void set_speed_epsilon( const speed_type& speed );

      const position_type& get_position() const;
      void set_position( const position_type& pos );
      void set_position( coordinate_type x, coordinate_type y );

      const force_type& get_gravity() const;
      void set_gravity( const force_type& g );

      double get_friction() const;
      void set_friction( double f );

      double get_environment_friction() const;
      void set_environment_friction( double f );

      double get_angle() const;
      void set_angle( double a );

      void set_top_middle( const position_type& pos );
      void set_top_right( const position_type& pos );
      void set_bottom_left( const position_type& pos );
      void set_bottom_middle( const position_type& pos );
      void set_bottom_right( const position_type& pos );
      void set_left_middle( const position_type& pos );
      void set_right_middle( const position_type& pos );

      position_type get_top_middle() const;
      position_type get_top_right() const;
      position_type get_bottom_left() const;
      position_type get_bottom_middle() const;
      position_type get_bottom_right() const;
      position_type get_left_middle() const;
      position_type get_right_middle() const;

      position_type get_center_of_mass() const;
      void set_center_of_mass( const position_type& pos );

      bool is_fixed() const;
      void fix();

      bool can_move_items() const;
      void set_can_move_items(bool value);

      bool has_contact_after_collision() const;
      void set_contact_after_collision(bool contact);

      bool has_left_contact() const;
      bool has_right_contact() const;
      bool has_top_contact() const;
      bool has_bottom_contact() const;
      bool has_contact() const;

      virtual void set_left_contact(physical_item_state& item);
      virtual void set_right_contact(physical_item_state& item);
      virtual void set_top_contact(physical_item_state& item);
      virtual void set_bottom_contact(physical_item_state& item);
      virtual void set_contact(physical_item_state& item);

      void clear_contacts();

      void set_phantom( bool phantom );
      bool is_phantom() const;

    protected:
      void set_size( const size_box_type& size );
      void set_size( size_type width, size_type height );
      void set_width( size_type width );
      void set_height( size_type height );

      void to_string( std::string& str ) const;

    private:
      /** \brief Item's mass. */
      double m_mass;

      /** \brief Current item's speed. */
      speed_type m_speed;

      /** \brief Current item's acceleration (forces coming from the item). */
      force_type m_acceleration;

      /** \brief Gravity applied to the item. */
      force_type m_gravity;

      /** \brief Item friction as a percentage of the kept movement. */
      double m_friction;

      /** \brief Milieu friction as a percentage of the kept movement. */
      double m_environment_friction;

      /** \brief Current item's position. */
      position_type m_position;

      /** \brief Item's size. */
      size_box_type m_size;

      /** \brief Item's orientation. */
      double m_angle;

      /**
       * \brief Tell if the item is fixed or not.
       *
       * If an item is fixed, its position, speed and acceleration won't be able
       * to change any more.
       */
      bool m_fixed;

      /**
       * \brief Tell if the item can move an other item.
       */
      bool m_can_move_items;

      /**
       * \brief Tell if the item has a contact after collision detection.
       */
      bool m_contact_after_collision;

      /** \brief Tell if the item is in contact with an other item on its
          left. */
      bool m_left_contact;

      /** \brief Tell if the item is in contact with an other item on its
          right. */
      bool m_right_contact;

      /** \brief Tell if the item is in contact with an other item on its
          top. */
      bool m_top_contact;

      /** \brief Tell if the item is in contact with an other item on its
          bottom (ie. the current item is laid down an item. */
      bool m_bottom_contact;

      /** \brief Indicate if the item is a phantom.
       * i.e. the item goes throw the walls. */
      bool m_is_phantom;

      /** \brief The minimal speed. */
      speed_type m_speed_epsilon;
    
    }; // class physical_item_state
  } // namespace universe
} // namespace bear

#endif // __UNIVERSE_PHYSICAL_ITEM_STATE_HPP__
