/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file image.cpp
 * \brief Implementation of the bear::visual::image class.
 * \author Julien Jorge
 */
#include "visual/image.hpp"

#include <claw/exception.hpp>
#include <claw/assert.hpp>

/*----------------------------------------------------------------------------*/
std::vector<unsigned int> bear::visual::image::s_texture_references;

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor with a claw::graphic::image object.
 * \param data The image to copy.
 */
bear::visual::image::image(const claw::graphic::image& data)
  : m_texture_id(0)
{
  create_texture( data.width(), data.height() );
  copy_scanlines( data );
} // image::image() [claw::graphic::image]

/*----------------------------------------------------------------------------*/
/**
 * \brief Copy constructor.
 * \param that The image to copy from.
 */
bear::visual::image::image(const image& that)
  : m_texture_id(that.m_texture_id), m_size(that.m_size)
{
  register_texture();
} // image::image() [copy constructor]

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
bear::visual::image::~image()
{
  release_texture();
} // image::~image()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get OpenGL texture identifier.
 */
GLuint bear::visual::image::texture_id() const
{
  return m_texture_id;
} // image::texture_id()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get image's width.
 */
unsigned int bear::visual::image::width() const
{
  return m_size.x;
} // image::width()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get image's height.
 */
unsigned int bear::visual::image::height() const
{
  return m_size.y;
} // image::height()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get image's size.
 */
const claw::math::coordinate_2d<unsigned int>& bear::visual::image::size() const
{
  return m_size;
} // image::size()

/*----------------------------------------------------------------------------*/
/**
 * \brief Assigment operator.
 */
bear::visual::image&
bear::visual::image::operator=( const bear::visual::image& that )
{
  if (m_texture_id != that.m_texture_id)
    {
      release_texture();

      m_texture_id = that.m_texture_id;
      m_size = that.m_size;

      register_texture();
    }

  return *this;
} // image::size()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create an empty OpenGL texture.
 * \param width Width of the texture.
 * \param height Height of the texture.
 */
void
 bear::visual::image::create_texture( unsigned int width, unsigned int height )
{
#ifndef NDEBUG
  unsigned int precond;
  unsigned int nb_1;

  for (precond = width, nb_1=0; precond && (nb_1 <= 1); precond >>= 1)
    nb_1 += precond & 1;

  if ( nb_1 > 1 )
    throw CLAW_EXCEPTION("width must be a power of 2");

  for (precond = height, nb_1=0; precond && (nb_1 <=1); precond >>= 1)
    nb_1 += precond & 1;

  if ( nb_1 > 1 )
    throw CLAW_EXCEPTION("height must be a power of 2");
#endif

  m_size.x = width;
  m_size.y = height;

  glGenTextures(1, &m_texture_id);
  glBindTexture(GL_TEXTURE_2D, m_texture_id);
  glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, width, height, 0, GL_RGBA,
               GL_UNSIGNED_BYTE, NULL );

  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);

  if ( glGetError() != GL_NO_ERROR)
    throw CLAW_EXCEPTION( "OpenGL error" );

  register_texture();
} // image::create_texture()

/*----------------------------------------------------------------------------*/
/**
 * \brief Copy the content of an image to the texture.
 * \param data The image to copy.
 */
void bear::visual::image::copy_scanlines( const claw::graphic::image& data )
{
  claw::graphic::pixel32* line = new claw::graphic::pixel32[data.width()];

  for (unsigned int y=0; y!=data.height(); ++y)
    {
      std::copy( data[y].begin(), data[y].end(), line );
      glTexSubImage2D(GL_TEXTURE_2D, 0, 0, y, data.width(), 1, GL_RGBA,
		      GL_UNSIGNED_BYTE, line );
    }

  delete[] line;
} // image::copy_scanlines()

/*----------------------------------------------------------------------------*/
/**
 * \brief Register an OpenGL texture.
 * \param if The identifier of the texture.
 */
void bear::visual::image::register_texture() const
{
  if ( s_texture_references.size() <= m_texture_id )
    {
      unsigned int i = s_texture_references.size();
      s_texture_references.resize( m_texture_id + 1 );

      for (; i!=s_texture_references.size(); ++i)
        s_texture_references[i] = 0;
    }

  ++s_texture_references[m_texture_id];
} // image::register_texture()

/*----------------------------------------------------------------------------*/
/**
 * \brief Release an OpenGL texture.
 * \param if The identifier of the texture.
 */
void bear::visual::image::release_texture()
{
  CLAW_PRECOND( m_texture_id < s_texture_references.size() );
  CLAW_PRECOND( s_texture_references[m_texture_id] );

  --s_texture_references[m_texture_id];

  if ( !s_texture_references[m_texture_id] )
    glDeleteTextures(1, &m_texture_id);
} // image::release_texture()

