/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file screen.cpp
 * \brief Implementation of the bear::visual::screen class.
 * \author Julien Jorge
 */
#include "visual/screen.hpp"
#include <claw/exception.hpp>
#include <claw/bitmap.hpp>
#include <claw/logger.hpp>

#include <SDL/SDL.h>
#include <GL/gl.h>
#include <GL/glext.h>

#include <limits>

/*----------------------------------------------------------------------------*/
/**
 * \brief Global initializations common to all screens. Must be called at the
 *        begining of your program.
 */
void bear::visual::screen::initialize()
{
  if ( !SDL_WasInit(SDL_INIT_VIDEO) )
    if ( SDL_InitSubSystem(SDL_INIT_VIDEO) != 0 )
      throw CLAW_EXCEPTION( SDL_GetError() );

  if ( SDL_GL_SetAttribute( SDL_GL_DOUBLEBUFFER, 1 ) != 0 )
    {
      SDL_QuitSubSystem(SDL_INIT_VIDEO);
      throw CLAW_EXCEPTION( SDL_GetError() );
    }

  glClearColor(0.0, 0.0, 0.0, 0.0);
  glClearDepth(1.0);
} // screen::initialize()

/*----------------------------------------------------------------------------*/
/**
 * \brief Global uninitializations common to all screens. Must be called at the
 *        end of your program.
 */
void bear::visual::screen::release()
{
  SDL_QuitSubSystem(SDL_INIT_VIDEO);
} // screen::quit()

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param size Size of the screen.
 * \param title The title of the window created.
 * \param full Tell if the window is full screen or not.
 */
bear::visual::screen::screen
( const claw::math::coordinate_2d<unsigned int>& size,
  const std::string& title, bool full )
  : m_mode(SCREEN_IDLE), m_size(size)
{
  fullscreen(full);

  SDL_WM_SetCaption( title.c_str(), NULL );
  glEnable(GL_TEXTURE_2D);

  resize_view(m_size.x, m_size.y);
} // screen::screen() [constructor]

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a new dimension for the resulting projection (doesn't change
 *        the screen's size.
 * \param width Target's width.
 * \param height Target's height.
 */
void bear::visual::screen::resize_view(unsigned int width, unsigned int height)
{
  glViewport(0, 0, width, height );
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  glOrtho(0, m_size.x, m_size.y, 0, 0, 1);
  glMatrixMode(GL_MODELVIEW);

  failure_check( __FUNCTION__ );
} // screen::resize_view()

/*----------------------------------------------------------------------------*/
/**
 * \brief Turn fullscreen mode on/off.
 * \param b Tell if we want a fullscreen mode.
 */
void bear::visual::screen::fullscreen( bool b )
{
  Uint32 flags = SDL_OPENGL | SDL_RESIZABLE;

  if (b)
    flags |= SDL_FULLSCREEN;

  SDL_Surface* s = SDL_SetVideoMode( m_size.x, m_size.y, 32, flags );

  if (!s)
    throw CLAW_EXCEPTION( SDL_GetError() );
} // screen::fullscreen()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the size of the screen.
 */
const claw::math::coordinate_2d<unsigned int>&
bear::visual::screen::get_size() const
{
  return m_size;
} // screen::get_size()

/*----------------------------------------------------------------------------*/
/**
 * \brief Initialize the rendering process.
 */
void bear::visual::screen::begin_render()
{
  CLAW_PRECOND(m_mode == SCREEN_IDLE);

  m_mode = SCREEN_RENDER;
  glClear( GL_COLOR_BUFFER_BIT );
} // screen::begin_render()

/*----------------------------------------------------------------------------*/
/**
 * \brief Draw a sprite on the screen.
 * \param pos On screen position of the sprite.
 * \param s The sprite to draw.
 * \param a Rotate the sprite with this angle (in radians and in trigonometric
 *        direction). The center of the rotation is the center of the box
 *       (pos, s.get_size).
 */
void bear::visual::screen::render
( const claw::math::coordinate_2d<int>& pos, const sprite& s, double a )
{
  CLAW_PRECOND(m_mode == SCREEN_RENDER);

  double alpha = s.get_alpha_blend();

  glColor4f(1.0f, 1.0f, 1.0f, alpha);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glEnable(GL_BLEND);

  glBindTexture( GL_TEXTURE_2D, s.get_image().texture_id() );

  render_sprite( pos, s, a );

  glDisable(GL_BLEND);

  failure_check( __FUNCTION__ );
} // screen::render()

/*----------------------------------------------------------------------------*/
/**
 * \brief Stop the rendering process.
 */
void bear::visual::screen::end_render()
{
  CLAW_PRECOND(m_mode == SCREEN_RENDER);

  glFlush();
  SDL_GL_SwapBuffers();

  m_mode = SCREEN_IDLE;
  failure_check( __FUNCTION__ );
} // screen::end_render()

/*----------------------------------------------------------------------------*/
/**
 * \brief Initialize effects processing. Must be closed by end_effects().
 * \pre Screen is in rendering mode.
 * \post Screen is in effects mode.
 */
void bear::visual::screen::begin_effects()
{
  CLAW_PRECOND( m_mode == SCREEN_RENDER );

  m_mode = SCREEN_EFFECTS;

  //glFlush();

  failure_check( __FUNCTION__ );
} // screen::begin_effects()

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply an effect on the screen. You can apply several effects between
 * one pair of begin_effect() / end_effect().
 * \param effect The effect to apply.
 * \pre Screen is in effects mode.
 */
void bear::visual::screen::apply_effect(screen_effect& effect)
{
  CLAW_PRECOND( m_mode == SCREEN_EFFECTS );

  effect.apply();
} // screen::apply_effect()

/*----------------------------------------------------------------------------*/
/**
 * \brief Close effects processing. Must be initialized by begin_effects().
 * \pre Screen is in effects mode.
 * \post Screen is in rendering mode.
 */
void bear::visual::screen::end_effects()
{
  CLAW_PRECOND( m_mode == SCREEN_EFFECTS );

  m_mode = SCREEN_RENDER;
} // screen::end_effects()

/*----------------------------------------------------------------------------*/
/**
 * \brief Do a screen shot.
 * \param bitmap_name The name of the bitmap file where we'll dump image.
 */
void bear::visual::screen::shot( const std::string& bitmap_name ) const
{
  std::ofstream f(bitmap_name.c_str());

  if (!f)
    claw::logger << claw::log_warning
                 << "bear::visual::screen::shot: Can't open file '" 
                 << bitmap_name << "'" << claw::lendl;
  else
    {
      claw::graphic::bitmap bmp( m_size.x, m_size.y );
      shot(bmp);
      bmp.save(f);
      f.close();
    }
} // screen::shot()

/*----------------------------------------------------------------------------*/
/**
 * \brief Do a screen shot.
 * \param img The image in which we save the content of the screen.
 */
void bear::visual::screen::shot( claw::graphic::image& img ) const
{
  img.set_size( m_size.x, m_size.y );
  claw::graphic::pixel32* line = new claw::graphic::pixel32[m_size.x];

  for (unsigned int y=0; y!=m_size.y; ++y)
    {
      glReadPixels( 0, y, m_size.x, 1, GL_RGBA, GL_UNSIGNED_BYTE, line );

      for ( claw::graphic::pixel32* it=line; it!=line + m_size.x; ++it )
        it->components.alpha = 255;

      std::copy( line, line + m_size.x, img[m_size.y - y - 1].begin() );
    }

  delete[] line;

  failure_check( __FUNCTION__ );
} // screen::shot()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render a sprite with transformation (flip or mirror).
 * \param pos On screen position of the sprite.
 * \param s The sprite to draw.
 * \param a Rotate the sprite with this angle (in radians and in trigonometric
 *        direction). The center of the rotation is the center of the box
 *       (pos, s.get_size).
 */
void bear::visual::screen::render_sprite
( const claw::math::coordinate_2d<int>& pos, const sprite& s, double a )
{
  claw::math::box_2d<unsigned int> clip_vertices;
  claw::math::rectangle<unsigned int> clip_rectangle(s.clip_rectangle());

  if ( s.is_flipped() )
    {
      clip_vertices.first_point.y  = clip_rectangle.bottom();
      clip_vertices.second_point.y = clip_rectangle.position.y;
    }
  else
    {
      clip_vertices.first_point.y  = clip_rectangle.position.y;
      clip_vertices.second_point.y = clip_rectangle.bottom();
    }

  if ( s.is_mirrored() )
    {
      clip_vertices.first_point.x  = clip_rectangle.right();
      clip_vertices.second_point.x = clip_rectangle.position.x;
    }
  else
    {
      clip_vertices.first_point.x  = clip_rectangle.position.x;
      clip_vertices.second_point.x = clip_rectangle.right();
    }

  //claw::math::rectangle<int> render_box( pos.x, pos.y, s.width(), s.height()
  //);
  typedef claw::math::coordinate_2d<GLdouble> coord_double;
  coord_double render_coord[4];

  const coord_double center =
    (coord_double)pos + (coord_double)s.get_size() / 2;

  coord_double top_right( pos );
  coord_double bottom_left( pos );
  top_right.x += s.width();
  bottom_left.y += s.height();

  render_coord[0] = rotate( pos, a, center );
  render_coord[1] = rotate( top_right, a, center );
  render_coord[2] = rotate( pos + s.get_size(), a, center );
  render_coord[3] = rotate( bottom_left, a, center );

  render_image( render_coord, clip_vertices, s.get_image().size() );
} // screen::render_sprite()

/*----------------------------------------------------------------------------*/
/**
 * \brief Draw current texture.
 * \param render_box On screen position and size of the texture.
 * \param clip Part of the texture to draw.
 * \param tex_size The full size of the used texture.
 */
void bear::visual::screen::render_image
( const claw::math::coordinate_2d<GLdouble> render_coord[],
  const claw::math::box_2d<unsigned int>& clip,
  const claw::math::coordinate_2d<unsigned int>& tex_size )
{
  glBegin(GL_QUADS);
  {
    // Top-left corner
    glTexCoord2d( (GLdouble)clip.first_point.x / (GLdouble)tex_size.x,
                  (GLdouble)clip.first_point.y / (GLdouble)tex_size.y );
    glVertex2d(render_coord[0].x, render_coord[0].y);

    // Top-right corner
    glTexCoord2d( (GLdouble)clip.second_point.x / (GLdouble)tex_size.x,
                  (GLdouble)clip.first_point.y  / (GLdouble)tex_size.y );
    glVertex2d(render_coord[1].x, render_coord[1].y);

    // Bottom-right corner
    glTexCoord2d( (GLdouble)clip.second_point.x / (GLdouble)tex_size.x,
                  (GLdouble)clip.second_point.y / (GLdouble)tex_size.y );
    glVertex2d(render_coord[2].x, render_coord[2].y);

    // Bottom-left corner
    glTexCoord2d( (GLdouble)clip.first_point.x  / (GLdouble)tex_size.x,
                  (GLdouble)clip.second_point.y / (GLdouble)tex_size.y );
    glVertex2d(render_coord[3].x, render_coord[3].y);
  }
  glEnd();

  failure_check( __FUNCTION__ );
} // screen::render_image()

/*----------------------------------------------------------------------------*/
claw::math::coordinate_2d<GLdouble> bear::visual::screen::rotate
( const claw::math::coordinate_2d<GLdouble>& pos, GLdouble a,
  const claw::math::coordinate_2d<GLdouble>& center ) const
{
  claw::math::coordinate_2d<GLdouble> result(center);

  result.x += (pos.x - center.x) * cos(a) - (pos.y - center.y) * sin(-a);
  result.y += (pos.x - center.x) * sin(-a) + (pos.y - center.y) * cos(a);

  return result;
} // screen::rotate()

/*----------------------------------------------------------------------------*/
/**
 * \brief Check if no error has occured.
 * \param where Information on where this function is called.
 */
void bear::visual::screen::failure_check( const std::string& where ) const
{
  GLenum err = glGetError();

  if ( err != GL_NO_ERROR)
    {
      std::string err_string(where + ": ");

      switch (err)
        {
        case GL_NO_ERROR:
          err_string += "no error (?).";
          break;
        case GL_INVALID_ENUM:
          err_string +=
            "unacceptable value is specified for an enumerated argument.";
          break;
        case GL_INVALID_VALUE:
          err_string += "numeric argument is out of range.";
          break;
        case GL_INVALID_OPERATION:
          err_string += "operation is not allowed in the current state.";
          break;
        case GL_STACK_OVERFLOW: err_string += "stack overflow.";
          break;
        case GL_STACK_UNDERFLOW: err_string += "stack underflow.";
          break;
        case GL_OUT_OF_MEMORY:
          err_string += "not enough memory to execute the command.";
          break;
        case GL_TABLE_TOO_LARGE:
          err_string +=
            "table exceeds the implementation's maximum supported table size";
          break;
        default:
          err_string += "unknow error code.";
        }

      throw claw::exception( err_string );
    }
} // screen::failure_check()
