/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file base_block.hpp
 * \brief A base class for blocks.
 * \author Sebastien Angibaud
 */
#ifndef __BEAR_BASE_BLOCK_HPP__
#define __BEAR_BASE_BLOCK_HPP__


#include "universe/collision_event/collision_event.hpp"
#include <map>

namespace bear
{
  /**
   * \brief A class representing a base_block.
   *
   * \b template \b parameters :
   * - \a Base : the base class for this item. Must inherit from
   *    engine::base_item,
   * - EventGenerator : A class with a method
   *    <tt> template<zone Z> collision_event* get( const type& ) const; </tt>
   *    returning an instance of the collision_event corresponding to a zone.
   *    The \a type argument is of any type in the hierarchy above this
   *    base_block.
   *
   * The custom fields of this class are:
   * - block_type : string, the type of the block, must be one of the following:
   *  - \b left_top_wall: a block that joins a ground and a left wall.
   *  - \b right_top_wall: a block that joins a ground and a left wall.
   *  - \b left_center_wall: a piece of a serie of blocks used for a left wall
   *    (ie. a wall that aligns to the left).
   *  - \b center_center_wall: a block alone, in the middle of nothing.
   *  - \b right_center_wall: a piece of a serie of blocks used for a right wall
   *    (ie. a wall that aligns to the right).
   *  - \b left_bottom_wall: a block that joins a ceiling and a left wall.
   *  - \b right_bottom_wall: a block that joins a ceiling and a right wall.
   *  - \b center_ceiling: a block positioned between some others blocks of
   *    ceiling.
   *  - \b left_ground: left most block of a ground.
   *  - \b center_ground: a piece of ground positioned between others blocks of
   *    ground.
   *  - \b right_ground: right most block of a ground.
   *  - \b shifted_up_ground: a ground that always align to the top when the
   *    other item is on the top (left, middle and right), the left or the
   *    right.
   *  - \b left_rod: left most part of a serie of blocks used both for ground
   *    and ceiling.
   *  - \b right_rod: right most part of a serie of blocks used both for ground
   *    and ceiling.
   *  - \b top_rod: top part of a serie of blocks used for a both left and
   *    right wall.
   *  - \b bottom_rod: bottom part of a serie of blocks used for a both left and
   *    right wall.
   *  - \b vertical_rod: a piece of a serie of blocks used for a both left and
   *    right wall.
   *  - \b horizontal_rod: a piece of a serie of blocks used both for ground and
   *    ceiling.
   *
   * \author Sebastien Angibaud
   */
  template<class Base, class EventGenerator>
  class base_block: public Base
  {
    typedef Base super;

  public:
    /**
     * \brief The type of the block. Used to determine the direction of the
     *        alignment in the collision events.
     *
     * \remark See the global documentation of the class to know the meaning of
     *         each value.
     */
    enum block_type
      {
        left_top_wall,
        right_top_wall,
        left_center_wall,
        center_center_wall,
        right_center_wall,
        left_bottom_wall,
        right_bottom_wall,

        center_ceiling,

        left_ground,
        center_ground,
        right_ground,
        shifted_up_ground,

        left_rod,
        right_rod,
        top_rod,
        bottom_rod,
        vertical_rod,
        horizontal_rod,

        undefined_block
      }; // enum block_type

  public:
    base_block();

    virtual
    bool set_string_field( const std::string& name, const std::string& value );

    bool is_valid() const;

  protected:
    void set_collision_events_left_top_wall();
    void set_collision_events_right_top_wall();
    void set_collision_events_left_center_wall();
    void set_collision_events_center_center_wall();
    void set_collision_events_right_center_wall();
    void set_collision_events_left_bottom_wall();
    void set_collision_events_right_bottom_wall();

    void set_collision_events_center_ceiling();

    void set_collision_events_left_ground();
    void set_collision_events_center_ground();
    void set_collision_events_right_ground();
    void set_collision_events_shifted_up_ground();

    void set_collision_events_left_rod();
    void set_collision_events_right_rod();
    void set_collision_events_top_rod();
    void set_collision_events_bottom_rod();
    void set_collision_events_vertical_rod();
    void set_collision_events_horizontal_rod();

    virtual bool block_type_is_valid() const;

  private:
    bool set_type( const std::string& value );
    void set_collision_events_table();

    template<universe::zone::position Zone, universe::zone::position EventZone>
    void set_collision_event();

  protected:
    /** \brief The type of our block. */
    block_type m_block_type;

  }; // class base_block

} // namespace bear

#include "generic_items/code/base_block.tpp"

#endif // __BEAR_BASE_BLOCK_HPP__
