/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file base_train.cpp
 * \brief Implementation of the bear::base_train class.
 * \author Sebastien Angibaud
 */
#include "generic_items/base_train.hpp"
#include "universe/physical_item.hpp"

#include "engine/export.hpp"

BASE_ITEM_EXPORT( base_train, bear )

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
bear::base_train::base_train()
{
  m_loop_back = false;
  m_loops = 0;
  set_global(true);
} // base_train::base_train()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
bear::base_train::~base_train()
{

} // base_train::~base_train()

/*----------------------------------------------------------------------------*/
/**
 * \brief Do post creation actions.
 */
void bear::base_train::start()
{
  generate_forced_movement();

  m_last_position = get_position();

  super::start();
} // bear::base_train::start()

/*----------------------------------------------------------------------------*/
/**
 * \brief Do one step in the progression of the item.
 * \param elapsed_time Elasped time since the last progress.
 */
void bear::base_train::progress( universe::time_type elapsed_time )
{
  super::progress( elapsed_time );

  item_list::iterator it;
  std::list<item_list::iterator> dead;

  for (it=m_list_items.begin(); it!=m_list_items.end(); ++it)
    if ( (base_item*)*it == NULL )
      dead.push_front(it);

  for( ; !dead.empty(); dead.pop_front() )
    m_list_items.erase( dead.front() );
} // bear::base_train::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type <unsigned integer>.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool bear::base_train::set_u_integer_field
( const std::string& name, unsigned int value )
{
  bool ok = true;

  if ( name == "loops" )
    m_loops = value;
  else
    ok = super::set_u_integer_field(name,value);

  return ok;
} // bear::base_train::set_u_integer_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type vector of base_item.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool bear::base_train::set_item_list_field
( const std::string& name, const std::vector<base_item*>& value)
{
  bool ok = true;

  if ( name == "path" ) 
    m_targets = std::vector<base_item*>(value);
  else
    ok = super::set_item_list_field(name,value);

  return ok;
} // base_train::set_item_list_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type \c bool.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool bear::base_train::set_bool_field( const std::string& name, bool value )
{
  bool ok = true;

  if ( name == "loop_back" )
    m_loop_back = value;
  else
    ok = super::set_bool_field(name,value);

  return ok;
} // base_train::set_bool_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the item is correctly initialized.
 */
bool bear::base_train::is_valid() const
{
  bool ok = false;

  if ( m_targets.empty() )
    claw::logger << claw::log_error << "base_train needs at least one target."
                 << claw::lendl;
  else 
    ok = true;

  return ok && super::is_valid();
} // bear::base_train::is_valid()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell that the item has a contact on its top.
 */
void bear::base_train::set_top_contact(universe::physical_item_state& item)
{
  super::set_top_contact(item);
  m_list_items.push_front(static_cast<base_item*>(&item));
} // base_train::set_top_contact()

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the movement of the item.
 * \param elapsed_time Elasped time since the last call.
 */
void bear::base_train::move( universe::time_type elapsed_time )
{
  super::move(elapsed_time);

  item_list::iterator it;
  universe::position_type position = get_position();

  for(it = m_list_items.begin(); it != m_list_items.end(); ++it)
    {
      // move items
      universe::position_type pos = (*it)->get_position();
      pos += position - m_last_position;
      (*it)->set_position(pos);
    }

  m_last_position = position;
  m_list_items.clear();
} // base_train::move()

/*----------------------------------------------------------------------------*/
/**
 * \brief Return the list of dependent items.
 */
void bear::base_train::get_dependent_items
( std::set<universe::physical_item_state*>& items_set )
{
  super::get_dependent_items(items_set);
  items_set.insert( m_list_items.begin(), m_list_items.end());
} // base_train::get_dependent_items()

/*----------------------------------------------------------------------------*/
/**
 * \brief Give a string representation of the item.
 * \param str (out) The result of the convertion.
 */
void bear::base_train::to_string( std::string& str ) const
{
  std::ostringstream oss;
  oss << "nb_items: " << m_list_items.size() << "\n";

  super::to_string(str);

  str = str + oss.str();
} // base_train::to_string()

/*----------------------------------------------------------------------------*/
/**
 * \brief Generate the forced movement.
 */
void bear::base_train::generate_forced_movement()
{
  std::vector<base_item*>::iterator it;
  universe::forced_train::target_list target_list;
  unsigned int i = 0;

  target_list.resize(m_targets.size());

  for ( it = m_targets.begin(); it != m_targets.end(); ++it, ++i)
    {
      universe::forced_train::target target;

      target.first = *it;
      target.second = 100; // defaut TO CHANGE
      target_list[i] = target;
    }

  universe::forced_train* movement =
    new universe::forced_train (*this, target_list);

  movement->set_loop_back( m_loop_back );
  movement->set_loops( m_loops );

  set_forced_movement( *movement );
  get_forced_movement().start();
} // bear::base_train::generate_forced_movement()
