/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file invisible_straight_slope.cpp
 * \brief Implementation of the bear::invisible_straight_slope class.
 * \author Julien Jorge
 */
#include "generic_items/invisible_straight_slope.hpp"

#include "engine/export.hpp"

BASE_ITEM_EXPORT( invisible_straight_slope, bear )

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
bear::invisible_straight_slope::invisible_straight_slope()
  : m_wall(false), m_solid_top_or_left(true),
    m_solid_bottom_or_right(false), m_margin(10)
{

} // invisible_straight_slope::invisible_straight_slope()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type \c boolean.
 * \param name The name of the field to set.
 * \param value The new value of the field.
 */
bool bear::invisible_straight_slope::set_bool_field
( const std::string& name, bool value )
{
  bool result = true;

  if ( name == "wall" )
    m_wall = value;
  else if ( name == "solid_top" )
    m_solid_top_or_left = value;
  else if ( name == "solid_left" )
    m_solid_top_or_left = value;
  else if ( name == "solid_bottom" )
    m_solid_bottom_or_right = value;
  else if ( name == "solid_right" )
    m_solid_bottom_or_right = value;
  else
    result = super::set_bool_field(name, value);

  return result;
} // invisible_straight_slope::set_bool_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type \c <real>.
 * \param name The name of the field to set.
 * \param value The new value of the field.
 */
bool bear::invisible_straight_slope::set_real_field
( const std::string& name, double value )
{
  bool result = true;

  if ( name == "margin" )
    m_margin = value;
  else
    result = super::set_real_field(name, value);

  return result;
} // invisible_straight_slope::set_real_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a collision.
 * \param that The other item in the collision.
 * \param old_self The previous state of the current item.
 * \param old_that The previous state of the other item.
 */
void bear::invisible_straight_slope::collision
( universe::physical_item& that, const state_type& old_self,
  const state_type& old_that )
{
  if ( !that.is_phantom() )
    {
      universe::collision_info info( old_self, old_that, *this, that );

      if ( m_wall )
        wall_alignment(info, that);
      else
        ground_alignment(info, that);
    }
} // invisible_straight_slope::collision()

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the alignment for a ground.
 * \param info The informations on the collision.
 * \param that The other item in the collision.
 */
void bear::invisible_straight_slope::ground_alignment
( const universe::collision_info& info, universe::physical_item& that )
{
  const universe::coordinate_type pos
    ( info.get_position_on_contact().x + that.get_width() / 2 );

  if ( (pos >= get_position().x) && (pos <= get_right()) )
    {
      universe::collision_event* align(NULL);
      const universe::zone::position z = info.get_collision_side();

      if ( z == universe::zone::top_zone )
        {
          if ( m_solid_top_or_left )
            align = new ce_align_stop_top();
        }
      else if ( z == universe::zone::bottom_zone )
        {
          if ( m_solid_bottom_or_right )
            align = new ce_align_stop_bottom();
        }
      else if ( (z == universe::zone::middle_left_zone)
                || (z == universe::zone::middle_zone)
                || (z == universe::zone::middle_right_zone) )
        {
          if ( that.get_bottom() <= get_position().y + m_margin )
            {
              if ( m_solid_top_or_left )
                align =  new ce_align_stop_top();
            }
          else if ( that.get_position().y >= get_bottom() - m_margin )
            if ( m_solid_bottom_or_right )
              align = new ce_align_stop_bottom();
        }

      if (align != NULL)
        {
          align->execute(info, *this, that);
          delete align;
        }
    }
} // invisible_straight_slope::ground_alignment()

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the alignment for a wall.
 * \param info The informations on the collision.
 * \param that The other item in the collision.
 */
void bear::invisible_straight_slope::wall_alignment
( const universe::collision_info& info, universe::physical_item& that )
{
  const universe::coordinate_type pos
    ( info.get_position_on_contact().x + that.get_width() / 2 );

  if ( (pos < get_position().x) || (pos > get_right()) )
    {
      universe::collision_event* align(NULL);
      const universe::zone::position z = info.get_collision_side();

      if ( z == universe::zone::middle_left_zone )
        {
          if ( m_solid_top_or_left )
            align = new ce_align_stop_left;
        }
      else if ( z == universe::zone::middle_right_zone )
        {
          if ( m_solid_bottom_or_right )
            align = new ce_align_stop_right;
        }
      else
        {
          if ( that.get_right() <= get_position().x + m_margin )
            {
              if ( m_solid_top_or_left )
                align = new ce_align_stop_left;
            }
          else if ( that.get_position().x >= get_right() - m_margin )
            if ( m_solid_bottom_or_right )
              align = new ce_align_stop_right;
        }

      if (align != NULL)
        {
          align->execute(info, *this, that);
          delete align;
        }
    }
} // invisible_straight_slope::wall_alignment()
