/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file pendulum.cpp
 * \brief Implementation of the bear::pendulum class.
 * \author Julien Jorge
 */
#include "generic_items/pendulum.hpp"

#include <cmath>
#include "engine/export.hpp"

BASE_ITEM_EXPORT( pendulum, bear )

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
bear::pendulum::pendulum()
  : m_link(NULL), m_step(0), m_speed(1), m_vertical_ratio(2)
{

} // pendulum::pendulum()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
bear::pendulum::~pendulum()
{
  if (m_link)
    delete m_link;
} // pendulum::~pendulum()

/*----------------------------------------------------------------------------*/
/**
 * \brief Initialize the item.
 */
void bear::pendulum::start()
{
  m_length = (unsigned int)(get_height() / m_link->height());
  m_step = 0;
} // pendulum::start()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a sprite field of the item.
 * \param name The name of the field.
 * \param value The sprite to set.
 */
bool bear::pendulum::set_sprite_field
( const std::string& name, visual::sprite* value )
{
  bool result = true;

  if ( name == "link" )
    m_link = value;
  else
    result = super::set_sprite_field( name, value );

  return result;
} // pendulum::set_sprite_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set an integer field of the item.
 * \param name The name of the field.
 * \param value The integer to set.
 */
bool bear::pendulum::set_u_integer_field
( const std::string& name, unsigned int value )
{
  bool result = true;

  if ( name == "width" )
    set_width( value );
  else if ( name == "height" )
    set_height( value );
  else if ( name == "speed" )
    m_speed = value;
  else if ( name == "vertical_ratio" )
    m_vertical_ratio = value;
  else
    result = super::set_u_integer_field( name, value );

  return result;
} // pendulum::set_u_integer_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set an integer field of the item.
 * \param name The name of the field.
 * \param value The integer to set.
 */
bool bear::pendulum::set_real_field( const std::string& name, double value )
{
  bool result = true;

  if ( name == "width" )
    set_width( value );
  else if ( name == "height" )
    set_height( value );
  else
    result = super::set_real_field( name, value );

  return result;
} // pendulum::set_real_field()


/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if all fields are set.
 */
bool bear::pendulum::is_valid() const
{
  return get_width() && get_height() && m_link && m_speed && super::is_valid();
} // pendulum::is_valid()

/*----------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the progression of the item.
 */
void bear::pendulum::progress( universe::time_type elapsed_time )
{
  m_step = (m_step + m_speed) % 360;
} // pendulum::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the sprite representing the item.
 * \param visuals (out) The sprites of the item, and their positions.
 */
void
bear::pendulum::get_visual( std::list<engine::scene_visual>& visuals ) const
{
  double angle = (double)(m_step) * 3.14159 / 180.0;
  const universe::position_type link_delta( m_link->get_size() / 2 );
  universe::position_type init( get_position() );

  init.x += get_width() / 2;

  for (unsigned int i=0; i!=m_length; ++i)
    {
      universe::position_type pos( init );
      pos.y += i * m_link->height();

      double amplitude_x = (double)(i*i * get_width() / 2)
        / (double)(m_length * m_length);
      double amplitude_y = (double)(i*i * get_height() / m_vertical_ratio)
        / (double)(m_length * m_length);

      pos.x += cos(angle) * amplitude_x;
      pos.y -= ((fabs(cos(2.0 * angle) + 1) / 2)* amplitude_y) ;

      pos -= link_delta;

      visuals.push_front( engine::scene_visual( get_position(), *m_link ) );
    }
} // pendulum::get_visual()
