/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file players_finish_level.cpp
 * \brief Implementation of the ptb::players_finish_level class.
 * \author Sbastien Angibaud
 */
#include "generic_items/players_finish_level.hpp"
#include "engine/player.hpp"
#include "engine/export.hpp"
#include "engine/game.hpp"

BASE_ITEM_EXPORT( players_finish_level, bear )

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
bear::players_finish_level::players_finish_level()
{
  set_phantom(true);
  set_can_move_items(false);
} // players_finish_level::players_finish_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Initialise the item.
 */
void bear::players_finish_level::start()
{
  m_first_player = engine::pointer_to_player( engine::player::player_name(1) );
  m_second_player = engine::pointer_to_player( engine::player::player_name(2) );
} // players_finish_level::start()

/*----------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the progression of the item.
 */
void bear::players_finish_level::progress( universe::time_type elapsed_time )
{
  bool ok1 = true;
  bool ok2 = true;

  if ( m_first_player )
    {
      claw::math::rectangle<universe::coordinate_type> bounding_box_1
        ( m_first_player->get_bounding_box() );

      if ( ! bounding_box_1.intersects(get_bounding_box()) )
        ok1 = false;
    }

  if ( m_second_player )
    {
      claw::math::rectangle<universe::coordinate_type> bounding_box_2
        ( m_second_player->get_bounding_box() );

      if ( ! bounding_box_2.intersects(get_bounding_box()) )
        ok2 = false;
    }

  if ( ok1 && ok2  )
    engine::game::get_instance().set_waiting_level( m_level_name );
} // players_finish_level::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type <std::string>.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool bear::players_finish_level::set_string_field
( const std::string& name, const std::string& value )
{
  bool result;

  if ( name == "level" )
    {
      m_level_name = value;
      result = true;
    }
  else
    result = super::set_string_field( name, value );

  return result;
} // players_finish_level::set_string_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the item is correctly initialized.
 */
bool bear::players_finish_level::is_valid() const
{
  return !m_level_name.empty() && super::is_valid();
} // players_finish_level::is_valid()
