/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file straight_slope.hpp
 * \brief A piece of ground/wall intended to be used next some slopes.
 * \author Julien Jorge
 */
#ifndef __BEAR_INVISIBLE_STRAIGHT_SLOPE_HPP__
#define __BEAR_INVISIBLE_STRAIGHT_SLOPE_HPP__

#include "engine/base_item.hpp"
#include "universe/collision_event/collision_event_align_stop.hpp"

namespace bear
{
  /**
   * \brief A piece of invisible ground/wall intended to be used 
   *        next some slopes.
   *
   * This items works like a slope. That is, the other item is aligned only if
   * the X-coordinate of his center of mass is in the range covered by the
   * current item.
   *
   * The parameters accepted by this item are:
   *  - \a wall: \c boolean Tell that this block is intented to be used as a
   *    wall. Default is \c false (this is  ground or a ceiling).
   *  - \a solid_top: \c boolean Tell that the top edge is solid (ie. align
   *    other items). If (wall = true), then it set the left edge as solid.
   *  - \a solid_left: \c boolean Tell that the left edge is solid (ie. align
   *    other items). If (wall = false), then it set the top edge as solid.
   *  - \a solid_bottom: \c boolean Tell that the bottom edge is solid (ie.
   *    align other items). If (wall = true), then it set the right edge as
   *    solid.
   *  - \a solid_right: \c boolean Tell that the right edge is solid (ie. align
   *    other items). If (wall = false), then it set the bottom edge as solid.
   *  - \a margin: \c real The tolerance between the position of the other
   *    item and the current item for alignments. Positive value means that the
   *    margin is inside the block.
   *  - any field supported by the parent class.
   *
   * \author Julien Jorge
   */
  class invisible_straight_slope :
    public engine::base_item
  {
  public:
    /** \brief The type of the parent class. */
    typedef engine::base_item super;

  private:
    typedef universe::collision_event_align_stop<universe::align_top>
    ce_align_stop_top;

    typedef universe::collision_event_align_stop<universe::align_left>
    ce_align_stop_left;

    typedef universe::collision_event_align_stop<universe::align_right>
    ce_align_stop_right;

    typedef universe::collision_event_align_stop<universe::align_bottom>
    ce_align_stop_bottom;

  public:
    invisible_straight_slope();

    bool set_bool_field( const std::string& name, bool value );
    bool set_real_field( const std::string& name, double value );

  private:
    void collision( universe::physical_item& that, const state_type& old_self,
                    const state_type& old_that );

    void ground_alignment
    ( const universe::collision_info& info, universe::physical_item& that );
    void wall_alignment
    ( const universe::collision_info& info, universe::physical_item& that );

  private:
    /** \brief Tell if the block is a wall (otherwise a ground). */
    bool m_wall;

    /** \brief Tell if the ground (resp. the wall) is solid on its top (resp. on
        its left). */
    bool m_solid_top_or_left;

    /** \brief Tell if the ground (resp. the wall) is solid on its bottom (resp.
        on its right). */
    bool m_solid_bottom_or_right;

    /** \brief The tolerance of the distance between the position of the other
        item and the edge of the collision, to apply the alignment. */
    universe::coordinate_type m_margin;

  }; // class invisible_straight_slope
} // namespace bear

#endif // __BEAR_INVISIBLE_STRAIGHT_SLOPE_HPP__
