/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file running_bear.cpp
 * \brief Implementation of the bear::running_bear class.
 * \author Julien Jorge
 */
#include "running_bear.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param argc Number of program arguments.
 * \param argv Program arguments.
 */
bear::running_bear::running_bear( int& argc, char** &argv )
  : claw::application(argc, argv), m_game(NULL), m_quit(false),
    m_game_file("default_game_description")
{
  check_arguments( argc, argv );

  if (!m_quit)
    create_game();
} // running_bear::running_bear()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
bear::running_bear::~running_bear()
{
  //  delete m_game;
} // running_bear::~running_bear()

/*----------------------------------------------------------------------------*/
/**
 * \brief Run the application.
 */
int bear::running_bear::run()
{
  try
    {
      if ( !m_quit )
        {
          m_game->run();
          delete m_game;
        }

      return EXIT_SUCCESS;
    }
  catch( std::exception& e )
    {
      std::cerr << "Exception: " << e.what() << std::endl;
      delete m_game;
      return EXIT_FAILURE;
    }
} // running_bear::run()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the game.
 */
void bear::running_bear::create_game()
{
  try
    {
      std::ifstream f( m_game_file.c_str() );
      m_game = new engine::game( f );
      f.close();
    }
  catch( std::exception& e )
    {
      std::cerr << "Exception: " << e.what() << std::endl;

      if ( m_game != NULL )
        {
          delete m_game;
          m_game = NULL;
        }

      m_quit = true;
    }
} // running_bear::create_game()

/*----------------------------------------------------------------------------*/
/**
 * \brief Print program usage.
 */
void bear::running_bear::help() const
{
  m_arguments.help();
} // running_bear::help()

/*----------------------------------------------------------------------------*/
/**
 * \brief Check program arguments.
 * \param argc Number of program arguments.
 * \param argv Program arguments.
 */
void bear::running_bear::check_arguments( int& argc, char** &argv )
{
  m_arguments.add("-g", "--game", "The game description file.", false, "file");
  m_arguments.add("-h", "--help", "Print this message and exit.", true, "");

  m_arguments.parse( argc, argv );

  if ( m_arguments.get_bool("-h") )
    {
      help();
      m_quit = true;
    }

  if ( m_arguments.has_value("-g") )
    m_game_file = m_arguments.get_string("-g");

  std::ifstream f( m_game_file.c_str() );
  if (!f)
    {
      std::cerr << "Can't open game file : '" + m_game_file + "'."
                << std::endl;
      help();
      m_quit = true;
    }
  else
    f.close();
} // running_bear::application()
