/*
  Plee The Bear

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file additional_stream_formatting.hpp
 * \brief Some methods to print complex structures on output streams.
 * \author Julien Jorge
 */
#ifndef __COMMON_ADDITIONAL_STREAM_FORMATTING_HPP__
#define __COMMON_ADDITIONAL_STREAM_FORMATTING_HPP__

#include <list>
#include <set>
#include <map>
#include <vector>
#include <ostream>
#include <string>

#include <claw/avl.hpp>
#include <claw/math.hpp>


/*----------------------------------------------------------------------------*/
/**
 * \brief Print a range of items in an output stream.
 * \param os The stream in which we write.
 * \param first Iterator on the first item.
 * \param last Iterator just past the last item.
 * \param sep The string to write between to items.
 */
template<typename Iterator>
std::ostream& output_range
  ( std::ostream& os, Iterator first, Iterator last, const std::string& sep )
{
  os << *first;

  for (++first; first!=last; ++first)
    os << sep << *first;

  return os;
} // output_range()

/*----------------------------------------------------------------------------*/
/**
 * \brief operator<<() overload for std::pair.
 * \param os The stream in which we write.
 * \param that The instance to write.
 */
template<typename T, typename U>
std::ostream& operator<<( std::ostream& os, const std::pair<T, U>& that )
{
  return os << "<" << that.first << ", " << that.second << ">";
} // operator<<() [ std::pair ]

/*----------------------------------------------------------------------------*/
/**
 * \brief operator<<() overload for std::list.
 * \param os The stream in which we write.
 * \param that The instance to write.
 */
template<typename T, typename Alloc>
std::ostream& operator<<( std::ostream& os, const std::list<T, Alloc>& that )
{
  os << "(";

  if ( !that.empty() )
    os << " " << output_range( os, that.begin(), that.end(), ", " ) << " ";

  return os << ")";
} // operator<<() [ std::list ]

/*----------------------------------------------------------------------------*/
/**
 * \brief operator<<() overload for std::vector.
 * \param os The stream in which we write.
 * \param that The instance to write.
 */
template<typename T, typename Alloc>
std::ostream& operator<<( std::ostream& os, const std::vector<T, Alloc>& that )
{
  os << "[";

  if ( !that.empty() )
    os << " " << output_range( os, that.begin(), that.end(), ", " ) << " ";

  return os << "]";
} // operator<<() [ std::vector ]

/*----------------------------------------------------------------------------*/
/**
 * \brief operator<<() overload for std::set.
 * \param os The stream in which we write.
 * \param that The instance to write.
 */
template<typename Key, typename Compare, typename Alloc>
std::ostream&
operator<<( std::ostream& os, const std::set<Key, Compare, Alloc>& that )
{
  os << "{";

  if ( !that.empty() )
    os << " " << output_range( os, that.begin(), that.end(), ", " ) << " ";

  return os << "}";
} // operator<<() [ std::set ]

/*----------------------------------------------------------------------------*/
/**
 * \brief operator<<() overload for std::map.
 * \param os The stream in which we write.
 * \param that The instance to write.
 */
template<typename Key, typename Data, typename Compare, typename Alloc>
std::ostream&
operator<<( std::ostream& os, const std::map<Key, Data, Compare, Alloc>& that )
{
  os << "[";

  if ( !that.empty() )
    os << " " << output_range( os, that.begin(), that.end(), ", " ) << " ";

  return os << "]";
} // operator<<() [ std::map ]

/*----------------------------------------------------------------------------*/
/**
 * \brief operator<<() overload for claw::avl.
 * \param os The stream in which we write.
 * \param that The instance to write.
 */
template<typename Key, typename Compare>
std::ostream&
operator<<( std::ostream& os, const claw::avl<Key, Compare>& that )
{
  os << "{";

  if ( !that.empty() )
    os << " " << output_range( os, that.begin(), that.end(), ", " ) << " ";

  return os << "}";
} // operator<<() [ claw::avl ]

/*----------------------------------------------------------------------------*/
/**
 * \brief operator<<() overload for claw::math::coordinate_2d.
 * \param os The stream in which we write.
 * \param that The instance to write.
 */
template<typename T>
std::ostream&
operator<<( std::ostream& os, const claw::math::coordinate_2d<T>& that )
{
  return os << "x=" << that.x << ", y=" << that.y;
} // operator<<() [ claw::math::coordinate_2d ]

/*----------------------------------------------------------------------------*/
/**
 * \brief operator<<() overload for claw::math::rectangle.
 * \param os The stream in which we write.
 * \param that The instance to write.
 */
template<typename T>
std::ostream&
operator<<( std::ostream& os, const claw::math::rectangle<T>& that )
{
  return os << that.position << ", w=" << that.width << ", h=" << that.height;
} // operator<<() [ claw::math::rectangle ]

/*----------------------------------------------------------------------------*/
/**
 * \brief operator<<() overload for claw::math::box_2d.
 * \param os The stream in which we write.
 * \param that The instance to write.
 */
template<typename T>
std::ostream&
operator<<( std::ostream& os, const claw::math::box_2d<T>& that )
{
  return os << "p1=<" << that.first_point << ">, p2=<" << that.second_point
	    << ">";
} // operator<<() [ claw::math::box_2d ]

/*----------------------------------------------------------------------------*/
/**
 * \brief operator<<() overload for claw::math::line_2d.
 * \param os The stream in which we write.
 * \param that The instance to write.
 */
template<typename T>
std::ostream&
operator<<( std::ostream& os, const claw::math::line_2d<T>& that )
{
  return os << "o=<" << that.origin << ">, d=<" << that.direction << ">";
} // operator<<() [ claw::math::line_2d ]

#endif // __COMMON_ADDITIONAL_STREAM_FORMATTING_HPP__
