/*
    Plee The Bear - Level editor

    Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file bf/code/image_pool.cpp
 * \brief Implementation of the bf::image_pool class.
 * \author Julien Jorge
 */
#include "bf/image_pool.hpp"

#include <claw/assert.hpp>
#include <claw/logger.hpp>
#include "bf/wx_facilities.hpp"

/*----------------------------------------------------------------------------*/
const wxSize bf::image_pool::s_thumb_size( 150, 150 );

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
bf::image_pool::image_pool()
{
  wxInitAllImageHandlers();
} // image_pool::image_pool()

/*----------------------------------------------------------------------------*/
/**
 * \brief Read all item files from a given directory and in its subdirectories.
 * \param dir_path The path to the directory to scan.
 */
void bf::image_pool::scan_directory( const std::string& dir_path )
{
  boost::filesystem::path dir(dir_path, boost::filesystem::native);

  if ( boost::filesystem::exists(dir) )
    {
      m_image.clear();
      m_thumbnail.clear();

      std::string path( dir_path );
      if ( path[path.size() - 1] != '/' )
        path += '/';

      m_root = std_to_wx_string(path);

      scan_sub_directory( path, dir );
    }
} // image_pool::scan_directory()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if we have an item class with a given name.
 * \param image_name The name of the item class to check.
 */
bool bf::image_pool::has_image( const wxString& image_name ) const
{
  return m_image.find(image_name) != m_image.end();
} // image_pool::has_image()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the image with a given name.
 * \param image_name The name of the image we want.
 */
wxBitmap bf::image_pool::get_image( const wxString& image_name ) const
{
  CLAW_PRECOND( has_image(image_name) );

  image_map::const_iterator it = m_image.find(image_name);

  if ( !it->second.IsOk() )
    const_cast<image_pool*>(this)->load_image_data(it->first);

  return it->second;
} // image_pool::get_image()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the thumbnail with a given name.
 * \param image_name The name of the thumbnail we want.
 */
wxBitmap bf::image_pool::get_thumbnail( const wxString& image_name ) const
{
  CLAW_PRECOND( has_image(image_name) );

  return m_thumbnail.find(image_name)->second;
} // image_pool::get_thumbnail()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get a constant iterator on the begining of the pool.
 */
bf::image_pool::const_iterator bf::image_pool::begin() const
{
  return const_iterator( m_image.begin() );
} // image_pool::begin()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get a constant iterator just past the end of the pool.
 */
bf::image_pool::const_iterator bf::image_pool::end() const
{
  return const_iterator( m_image.end() );
} // image_pool::end()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add an image.
 * \param root The path to the directory in which the path started.
 * \param file_path The path to the image to add.
 */
void bf::image_pool::add_image
( const std::string& root, const std::string& file_path )
{
  wxBitmap thumb = load_thumb( file_path );

  if ( thumb.IsOk() )
    {
      const wxString image_name =
        std_to_wx_string( file_path.substr(root.size()) );

      m_image[image_name] = wxBitmap();
      m_thumbnail[image_name] = thumb;
    }
} // image_pool::add_image()

/*----------------------------------------------------------------------------*/
/**
 * \brief Load the thumbnail of an image.
 * \param path The path to the image to load.
 */
wxBitmap bf::image_pool::load_thumb( const std::string& path ) const
{
  wxImage img( std_to_wx_string(path) );

  if ( (img.GetWidth() > s_thumb_size.x) 
       || (img.GetHeight() > s_thumb_size.y) )
    {
      int w, h;

      if( img.GetWidth() > img.GetHeight() )
        {
          w = s_thumb_size.x;
          h = (img.GetHeight() * w) / img.GetWidth();
        }
      else
        {
          h = s_thumb_size.y;
          w = (img.GetWidth() * h) / img.GetHeight();
        }

      img.Rescale(w, h);
    }

  return wxBitmap(img);
} // image_pool::load_thumb()

/*----------------------------------------------------------------------------*/
/**
 * \brief Load and store the data of an image.
 * \param path The path to the image to load.
 */
void bf::image_pool::load_image_data( const wxString& name )
{
  m_image[name] = wxBitmap( wxImage(m_root + name) );
} // image_pool::load_image_data()

/*----------------------------------------------------------------------------*/
/**
 * \brief Read all item files from a given directory and in its subdirectories.
 * \param The path to the root directory where the images are searched.
 * \param dir_path The path to the directory to scan.
 */
void bf::image_pool::scan_sub_directory
( const std::string& root, const boost::filesystem::path& dir )
{
  boost::filesystem::directory_iterator it(dir);
  boost::filesystem::directory_iterator eit;

  for ( ; it!=eit; ++it)
    if ( boost::filesystem::is_directory(*it) )
      scan_sub_directory( root, *it );
    else if (supported_extension( it->string() ))
      add_image( root, it->string() );
} // image_pool::scan_sub_directory()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if a path correspond to a supported extension.
 * \param path The path to check.
 */
bool bf::image_pool::supported_extension( const std::string& path ) const
{
  static char* ext[] = { ".png", ".jpg", ".tga", ".bmp", NULL };
  bool result = false;

  for (char** it=ext; !result && (*it != NULL); ++it)
    {
      const std::string cext(*it);
      result = ( path.rfind(cext) == path.size() - cext.size() );
    }

  return result;
} // image_pool::supported_extension()
