/*
  Plee The Bear - Level editor

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file bf/code/ingame_view.cpp
 * \brief Implementation of the bf::ingame_view class.
 * \author Julien Jorge
 */
#include "bf/ingame_view.hpp"

#include "bf/image_pool.hpp"
#include "bf/ingame_view_frame.hpp"
#include "bf/item_class_pool.hpp"
#include "bf/properties_frame.hpp"
#include "bf/layer_list_frame.hpp"
#include "bf/level_file_xml_writer.hpp"
#include "bf/wx_facilities.hpp"

#include <claw/assert.hpp>
#include <wx/dcbuffer.h>
#include <wx/clipbrd.h>

/*----------------------------------------------------------------------------*/
const wxCoord bf::ingame_view::s_point_size = 10;
const wxCoord bf::ingame_view::s_grip_size = 15;
bf::level_clipboard bf::ingame_view::s_clipboard;

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
bf::ingame_view::drag_info::drag_info()
  : drag_mode(drag_mode_none), picked_item(NULL)
{

} // ingame_view::drag_info::drag_info()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the difference between the initial mouse position and the current
 *        mouse position.
 */
wxSize bf::ingame_view::drag_info::delta() const
{
  return wxSize
    ( mouse_position.x - mouse_origin.x, mouse_position.y - mouse_origin.y );
} // ingame_view::drag_info::delta()



/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param view The view associated with this target.
 */
bf::ingame_view::item_drop_target::item_drop_target( ingame_view& view )
  : m_view(view)
{

} // ingame_view::item_drop_target::item_drop_target()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add an item in the level. The class name of the item is dropped on
 *        the view.
 * \param x The x coordinate of the mouse.
 * \param y The y coordinate of the mouse.
 * \param data The name of the class to instanciate.
 */
bool bf::ingame_view::item_drop_target::OnDropText
( wxCoord x, wxCoord y, const wxString& data )
{
  return m_view.add_item( wx_to_std_string(data), x, y );
} // ingame_view::item_drop_target::OnDropText()



/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param parent The window owning this window.
 * \param lvl The level.
 * \param layout The window layout of the program.
 *
 * The level will be deleted in the destructor.
 */
bf::ingame_view::ingame_view
( ingame_view_frame& parent, level* lvl, windows_layout& layout )
  : super( &parent, wxID_ANY ), m_parent(parent), m_layout(layout),
    m_history(lvl), m_view(0, 0), m_active_layer(0), m_selected_item(NULL),
    m_drag_info(NULL), m_visible(lvl->layers_count(), true),
    m_wireframe_drawing(true), m_graphic_drawing(true), m_display_grid(false)
{
  CLAW_PRECOND(lvl != NULL);
  SetDropTarget( new item_drop_target(*this) );
} // ingame_view::ingame_view()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the level contains a layer.
 */
bool bf::ingame_view::empty() const
{
  return m_history.get_level().layers_count() == 0;
} // ingame_view::empty()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the position of the view in the level.
 */
const wxPoint& bf::ingame_view::get_view_position() const
{
  return m_view;
} // ingame_view::get_view_position()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the position of the view in the level.
 * \param x X-position of the view.
 * \param y Y-position of the view.
 */
void bf::ingame_view::set_view_position( wxCoord x, wxCoord y )
{
  if ( empty() )
    m_view.x = m_view.y = 0;
  else
    {
      m_view.x = x;
      m_view.y = y;

      if ( (unsigned int)(m_view.x + GetSize().x)
           > get_active_layer().get_width() )
        m_view.x = get_active_layer().get_width() - GetSize().x;

      if (m_view.x < 0)
        m_view.x = 0;

      if ((unsigned int)(m_view.y + GetSize().y)
          > get_active_layer().get_height())
        m_view.y = get_active_layer().get_height() - GetSize().y;

      if (m_view.y < 0)
        m_view.y = 0;
    }

  m_parent.adjust_scrollbars();
} // ingame_view::set_view_position()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the index of the active layer.
 */
unsigned int bf::ingame_view::get_active_index() const
{
  return m_active_layer;
} // ingame_view::get_active_index()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the index of the active layer.
 */
void bf::ingame_view::set_active_index( unsigned int index )
{
  CLAW_PRECOND( index < m_history.get_level().layers_count() );

  m_active_layer = index;
  clear_selection();

  m_parent.adjust_scrollbars();
} // ingame_view::get_active_index()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the layer on which we are working.
 */
const bf::layer& bf::ingame_view::get_active_layer() const
{
  return m_history.get_level().get_layer(m_active_layer);
} // ingame_view::get_active_layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the level on which we are working.
 */
const bf::level& bf::ingame_view::get_level() const
{
  return m_history.get_level();
} // ingame_view::get_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the level on which we are working.
 * \remark Call start_change() before this method if you plan to modify the
 *         level.
 */
bf::level& bf::ingame_view::get_level()
{
  return m_history.get_level();
} // ingame_view::get_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if a given layer is visible.
 * \param index The index of the layer to check.
 */
bool bf::ingame_view::get_visible( unsigned int index )
{
  CLAW_PRECOND( index < m_visible.size() );
  return m_visible[index];
} // ingame_view::get_visible()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the visibility of a given layer.
 * \param index The index of the layer to change.
 * \param vis Tell if the layer is visible or not.
 */
void bf::ingame_view::set_visible( unsigned int index, bool vis )
{
  CLAW_PRECOND( index < m_visible.size() );
  m_visible[index] = vis;
} // ingame_view::set_visible()

/*----------------------------------------------------------------------------*/
/**
 * \brief Show/hide the grid.
 * \param v Tell is the grid is visible.
 */
void bf::ingame_view::show_grid( bool v )
{
  m_display_grid = v;
} // ingame_view::show_grid()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the grid is visible or not.
 */
bool bf::ingame_view::grid_visible() const
{
  return m_display_grid;
} // ingame_view::grid_visible()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the grid.
 */
const bf::grid& bf::ingame_view::get_grid() const
{
  return m_grid;
} // ingame_view::get_grid()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the grid.
 */
bf::grid& bf::ingame_view::get_grid()
{
  return m_grid;
} // ingame_view::get_grid()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the grid.
 * \param g The new grid.
 */
void bf::ingame_view::set_grid( const grid& g )
{
  m_grid = g;
} // ingame_view::set_grid()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a layer.
 * \param width The width of the layer.
 * \param height The height of the layer.
 * \param class_name The name of the class of the layer.
 */
void bf::ingame_view::add_layer
( unsigned int width, unsigned int height, const std::string& class_name )
{
  start_change();

  m_history.get_level().add_layer(width, height, class_name);
  m_visible.push_back(true);
  m_parent.adjust_scrollbars();
} // ingame_view::add_layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Remove the current layer.
 */
void bf::ingame_view::remove_layer()
{
  CLAW_PRECOND( !empty() );

  start_change();

  m_visible.erase( m_visible.begin() + m_active_layer );
  m_history.get_level().remove_layer(m_active_layer);

  if ( m_active_layer == m_history.get_level().layers_count() )
    if ( m_history.get_level().layers_count() > 0 )
      m_active_layer = m_history.get_level().layers_count() - 1;

  clear_selection();
} // ingame_view::remove_layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Change the parameters of the active layer.
 * \param x The new width of the layer.
 * \param y The new height of the layer.
 * \param class_name The new name of the class of the layer.
 */
void bf::ingame_view::change_layer
( unsigned int x, unsigned int y, const std::string& class_name )
{
  CLAW_PRECOND( !empty() );

  if ( (x!=get_active_layer().get_width())
       || (y!=get_active_layer().get_height())
       || (class_name!=get_active_layer().get_class_name()) )
    {
      start_change();
      current_layer().resize(x, y);
      current_layer().set_class_name(class_name);
      Refresh();
    }
} // ingame_view::remove_layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Move the current layer toward the background.
 */
void bf::ingame_view::move_backward()
{
  CLAW_PRECOND( !empty() );

  if ( m_active_layer > 0 )
    {
      start_change();

      bool tmp = m_visible[m_active_layer-1];
      m_visible[m_active_layer-1] = m_visible[m_active_layer];
      m_visible[m_active_layer] = tmp;

      m_history.get_level().move_backward(m_active_layer);
      --m_active_layer;
    }
} // ingame_view::move_backward()

/*----------------------------------------------------------------------------*/
/**
 * \brief Move the current layer toward the foreground.
 */
void bf::ingame_view::move_forward()
{
  CLAW_PRECOND( !empty() );

  if ( m_active_layer + 1 < m_history.get_level().layers_count() )
    {
      start_change();

      bool tmp = m_visible[m_active_layer+1];
      m_visible[m_active_layer+1] = m_visible[m_active_layer];
      m_visible[m_active_layer] = tmp;

      m_history.get_level().move_forward(m_active_layer);
      ++m_active_layer;
    }
} // ingame_view::move_forward()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add an item to the current layer.
 * \param class_name The name of the class to instanciate.
 * \param x The x coordinate of the item in the view.
 * \param y The y coordinate of the item in the view.
 * \return true if the item was added.
 */
bool bf::ingame_view::add_item
( const std::string& class_name, wxCoord x, wxCoord y )
{
  bool result = false;

  if ( !empty() )
    if ( item_class_pool::get_instance().has_item_class(class_name) )
      {
        start_change();

        item_instance item( class_name );
        item.set_position(x + m_view.x, y + m_view.y);

        set_selection( &current_layer().add_item(item) );
        result = true;
      }

  return result;
} // ingame_view::item_drop_target::add_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add an item to the current layer.
 * \param class_name The name of the class to instanciate.
 * \return true if the item was added.
 */
bool bf::ingame_view::add_item( const std::string& class_name )
{
  return add_item( class_name, GetSize().x / 2, GetSize().y / 2 );
} // ingame_view::add_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save the current level.
 * \param os The stream to save the level in.
 */
void bf::ingame_view::save( std::ostream& os )
{
  level_file_xml_writer lvl;
  lvl.save( os, m_history.get_level() );
  m_history.set_saved();
} // ingame_view::save()

/*----------------------------------------------------------------------------*/
/**
 * \brief Revert to the last save state.
 * Return True if a saved state is in the history.
 */
bool bf::ingame_view::revert_save()
{
  bool result = true;

  if ( ! m_history.saved_state() ) 
    {
      if ( m_history.saved_state_in_past() ) 
	{
	  while ( ! m_history.saved_state() ) 
	    if ( m_history.can_undo() )
	      undo();
	}
      else if ( m_history.saved_state_in_future() ) 
	{
	  while ( ! m_history.saved_state() ) 
	    if ( m_history.can_redo() )
	      redo();
	}
      else
	result = false;
    }

  return result;
} // ingame_view::revert_save()

/*----------------------------------------------------------------------------*/
/**
 * \brief Put a level state in the history.
 * \param lvl The new level.
 */
void bf::ingame_view::put_level( level* lvl )
{
  m_history.push(lvl);
} // ingame_view::put_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Start a change in the level.
 *
 * Update undo/redo information. Inform the parent frame.
 */
void bf::ingame_view::start_change()
{
  m_history.push();
  m_parent.set_changed();
} // ingame_view::start_change()

/*----------------------------------------------------------------------------*/
/**
 * \brief Undo the last change.
 */
void bf::ingame_view::undo()
{
  if ( m_history.can_undo() )
    {
      clear_selection();

      m_history.undo();

      m_parent.set_changed( !m_history.saved_state() );

      if ( m_active_layer >= m_history.get_level().layers_count() )
        m_active_layer = 0;

      update_layout();
    }
} // ingame_view::undo()

/*----------------------------------------------------------------------------*/
/**
 * \brief Cancel an undo.
 */
void bf::ingame_view::redo()
{
  if ( m_history.can_redo() )
    {
      clear_selection();

      m_history.redo();

      m_parent.set_changed( !m_history.saved_state() );
      update_layout();
    }
} // ingame_view::redo()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the history of the level.
 */
const bf::level_history& bf::ingame_view::get_history() const
{
  return m_history;
} // ingame_view::get_history()

/*----------------------------------------------------------------------------*/
/**
 * \brief Update the content of the windows of the program.
 */
void bf::ingame_view::update_layout()
{
  m_layout.get_layer_list_frame().set_level_view(this);
  m_layout.get_properties_frame().set_item( m_selected_item );

  Refresh();
} // ingame_view::update_layout()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if there is an item selected.
 */
bool bf::ingame_view::has_selection() const
{
  return m_selected_item != NULL;
} // ingame_view::has_selection()

/*----------------------------------------------------------------------------*/
/**
 * \brief Remove of the layer all items in the selection
 */
void bf::ingame_view::delete_selection()
{
  if ( !m_selected_group.empty() )
    {
      start_change();

      std::set<item_instance*>::iterator it;

      for (it=m_selected_group.begin(); it!=m_selected_group.end(); ++it)
        current_layer().remove_item(**it);

      m_selected_group.clear();
      m_selected_item = NULL;
      update_layout();
    }
} // ingame_view::delete_selection()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the selection to nothing.
 */
void bf::ingame_view::clear_selection()
{
  m_selected_group.clear();
  m_selected_item = NULL;
  update_layout();
} // ingame_view::clear_selection()

/*----------------------------------------------------------------------------*/
/**
 * \brief Select all items.
 */
void bf::ingame_view::select_all()
{
  std::list<item_instance*> item;
  pick_item( item,
             wxRect(0, 0, get_level().get_width(), get_level().get_height()) );
  set_selection( item );
} // ingame_view::select_all()

/*----------------------------------------------------------------------------*/
/**
 * \brief Select an item, its layer and center view on the item.
 * \param item The item to select.
 * \param index The index of layer.
 */
void bf::ingame_view::select_item_and_layer( item_instance* item,
					     const unsigned int index)
{
  // select the layer
  set_active_index( index );
  set_visible(index,true);
  
  // select the item
  set_selection(item);
  
  // center the view on the item
  wxCoord x = 
    (int)item->get_pos_x() - GetSize().x/2;
   
  wxCoord y = 
    (int)item->get_pos_y() - GetSize().y/2;
  
  set_view_position(x,y);
  
} // ingame_view::set_selection()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the clipboard.
 */
const bf::level_clipboard& bf::ingame_view::get_clipboard() const
{
  return s_clipboard;
} // ingame_view::get_clipboard()

/*----------------------------------------------------------------------------*/
/**
 * \brief Copy the current selection in the clipboard.
 */
void bf::ingame_view::copy_to_clipboard() const
{
  if ( m_selected_item != NULL )
    {
      s_clipboard.x = m_selected_item->get_pos_x();
      s_clipboard.y = m_selected_item->get_pos_y();

      s_clipboard.items.clear();

      std::set<item_instance*>::iterator it;

      for (it=m_selected_group.begin(); it!=m_selected_group.end(); ++it)
        s_clipboard.items.push_front(**it);
    }
} // ingame_view::copy_to_clipboard()

/*----------------------------------------------------------------------------*/
/**
 * \brief Copy the current selection in the clipboard then remove the selected
 *        items.
 */
void bf::ingame_view::cut_to_clipboard()
{
  copy_to_clipboard();
  delete_selection();
} // ingame_view::cut_to_clipboard()

/*----------------------------------------------------------------------------*/
/**
 * \brief Paste the content of the clipboard.
 * \param add_selection Indicates if the items are added at the selection.
 */
void bf::ingame_view::paste_from_clipboard(bool add_to_selection)
{
  if ( !s_clipboard.items.empty() && !empty() )
    {
      start_change();

      wxPoint center( m_view.x + GetSize().x / 2, m_view.y + GetSize().y / 2 );

      std::list<item_instance>::const_iterator it;
      std::list<item_instance*> selection;
      item_instance* selected;

      for (it=s_clipboard.items.begin(); it!=s_clipboard.items.end(); ++it)
        {
          item_instance* item = &current_layer().add_item(*it);

          if ( (it->get_pos_x() == s_clipboard.x)
               && (it->get_pos_y() == s_clipboard.y) )
            selected = item;

          item->set_position( center.x + (it->get_pos_x() - s_clipboard.x),
                              center.y + (it->get_pos_y() - s_clipboard.y) );

          selection.push_front(item);
        }

      set_selection(selection, selected, add_to_selection);
    }
} // ingame_view::paste_from_clipboard()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create a small view of the full level.
 */
wxBitmap bf::ingame_view::create_thumbnail( const wxSize& s ) const
{
  wxBitmap result(s.x, s.y);
  wxMemoryDC dc(result);

  dc.SetUserScale( (double)s.x / (double)m_history.get_level().get_width(),
                   (double)s.y / (double)m_history.get_level().get_height() );

  dc.SetBackground(*wxBLACK_BRUSH);
  dc.Clear();

  for (unsigned int i=0; i!=m_history.get_level().layers_count(); ++i)
    if ( m_visible[i] )
      render_layer_full(dc, i);

  return result;
} // ingame_view::create_thumbnail()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render all items of the level.
 */
void bf::ingame_view::render()
{
  wxBufferedPaintDC dc( this );

  if( IsShown() )
    {
      dc.SetBackground(*wxGREY_BRUSH);
      dc.Clear();

      render_layer_background(dc);
      render_layers(dc);
      render_grid(dc);
    }
} // ingame_view::render()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render all layers.
 * \param dc The device context for the drawings.
 */
void bf::ingame_view::render_layers( wxDC& dc ) const
{
  for (unsigned int i=0; i!=m_history.get_level().layers_count(); ++i)
    if ( m_visible[i] )
      {
        render_layer(dc, i);

        if ( i == m_active_layer )
          render_drag(dc);
      }
} // ingame_view::render_layers()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render a layer.
 * \param dc The device context for the drawings.
 * \param i The index of the layer to render.
 */
void bf::ingame_view::render_layer( wxDC& dc, unsigned int i ) const
{
  const layer& the_layer = m_history.get_level().get_layer(i);
  layer::const_item_iterator it;
  std::multimap<int, const item_instance*> z_order;

  const wxRect vis_box( m_view, GetSize() );

  for (it=the_layer.item_begin(); it!=the_layer.item_end(); ++it)
    {
      wxRect box = get_presence_box(*it);

      if ( box.Intersects(vis_box) )
        z_order.insert
          ( std::pair<int, const item_instance*>(it->get_pos_z(), &(*it)) );
    }

  std::multimap<int, const item_instance*>::const_iterator zit;
  bool active = (i == m_active_layer);

  for (zit=z_order.begin(); zit!=z_order.end(); ++zit)
    render_item(dc, *zit->second, active );
} // ingame_view::render_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render all items with a sprite in the layer.
 * \param dc The device context for the drawings.
 * \param i The index of the layer to render.
 */
void bf::ingame_view::render_layer_full( wxDC& dc, unsigned int i ) const
{
  const layer& the_layer = m_history.get_level().get_layer(i);
  layer::const_item_iterator it;
  std::multimap<int, const item_instance*> z_order;

  for (it=the_layer.item_begin(); it!=the_layer.item_end(); ++it)
    if ( it->has_sprite() )
      z_order.insert
        ( std::pair<int, const item_instance*>(it->get_pos_z(), &(*it)) );

  std::multimap<int, const item_instance*>::const_iterator zit;

  for (zit=z_order.begin(); zit!=z_order.end(); ++zit)
    {
      wxPoint pos( (wxCoord)zit->second->get_pos_x(),
                   (wxCoord)zit->second->get_pos_y() );
      wxPoint gap( zit->second->get_gap_x(), zit->second->get_gap_y() );
      render_sprite(dc, *zit->second, pos + gap);
    }
} // ingame_view::render_layer_full()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render an item on the screen.
 * \param dc The device context for the drawings.
 * \param item The item to render.
 * \param active Tell if the item is in the active layer.
 */
void bf::ingame_view::render_item
( wxDC& dc, const item_instance& item, bool active ) const
{
  wxPoint pos( (int)item.get_pos_x() - m_view.x,
               (int)item.get_pos_y() - m_view.y );
  render_item(dc, item, pos, active);
} // ingame_view::render_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render an item on the screen at a given position.
 * \param dc The device context for the drawings.
 * \param item The item to render.
 * \param pos The position of the item on the screen.
 * \param active Tell if the item is in the active layer.
 */
void bf::ingame_view::render_item
( wxDC& dc, const item_instance& item, const wxPoint& pos, bool active ) const
{
  wxSize size( (int)item.get_width(), (int)item.get_height() );
  
  if ( (size.x == 0) || (size.y == 0) )
    render_item_as_point( dc, item, pos, active );
  else
    render_item_as_box( dc, item, pos, size, active );

  if ( &item == m_selected_item )
    render_grip(dc);
} // ingame_view::render_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render an item as a point on the screen.
 * \param dc The device context for the drawings.
 * \param item The item to render.
 * \param pos The position of the point on the screen.
 * \param active Tell if the item is in the active layer.
 */
void bf::ingame_view::render_item_as_point
( wxDC& dc, const item_instance& item, const wxPoint& pos, bool active ) const
{
  dc.SetPen( get_display_pen(item, active) );
  dc.SetBrush(*wxTRANSPARENT_BRUSH);

  dc.DrawCircle(pos.x, pos.y, s_point_size);

  wxPoint p[2];

  p[0] = wxPoint( pos.x, pos.y - 2 * s_point_size );
  p[1] = wxPoint( pos.x, pos.y + 2 * s_point_size );

  dc.DrawPolygon(2, p);

  p[0] = wxPoint( pos.x - 2 * s_point_size, pos.y );
  p[1] = wxPoint( pos.x + 2 * s_point_size, pos.y );

  dc.DrawPolygon(2, p);
} // ingame_view::render_item_as_point()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render an item as a box on the screen.
 * \param dc The device context for the drawings.
 * \param item The item to render.
 * \param pos The position of the box.
 * \param size The size of the box.
 * \param active Tell if the item is in the active layer.
 */
void bf::ingame_view::render_item_as_box
( wxDC& dc, const item_instance& item, const wxPoint& pos, const wxSize& size,
  bool active ) const
{
  if ( m_graphic_drawing && item.has_sprite() )
    {
      wxPoint gap( item.get_gap_x(), item.get_gap_y() );
      bool force = !render_sprite(dc, item, pos + gap);

      if ( force || m_wireframe_drawing
           || ( m_selected_group.find((item_instance*)&item)
                != m_selected_group.end() ) )
        render_item_bounding_box(dc, item, pos, size, active);
    }
  else
    render_item_bounding_box(dc, item, pos, size, active);
} // ingame_view::render_item_as_box()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render a sprite on the screen.
 * \param dc The device context for the drawings.
 * \param item The item to render.
 * \param pos The position of the box.
 * \return true if the sprite was rendered.
 */
bool bf::ingame_view::render_sprite
( wxDC& dc, const item_instance& item, const wxPoint& pos ) const
{
  std::pair<wxBitmap, wxPoint> spr( get_item_visual(item, pos) );

  if ( spr.first.IsOk() )
    dc.DrawBitmap( spr.first, spr.second.x, spr.second.y, true );

  return spr.first.IsOk();
} // ingame_view::render_sprite()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render the bounding box of an item on the screen.
 * \param dc The device context for the drawings.
 * \param item The item to render.
 * \param pos The position of the box.
 * \param size The size of the box.
 * \param active Tell if the item is in the active layer.
 */
void bf::ingame_view::render_item_bounding_box
( wxDC& dc, const item_instance& item, const wxPoint& pos, const wxSize& size,
  bool active ) const
{
  dc.SetPen( get_display_pen(item, active) );
  dc.SetBrush(*wxTRANSPARENT_BRUSH);

  wxPoint p[4];

  p[0] = wxPoint( pos.x,  pos.y );
  p[1] = wxPoint( pos.x + size.x - 1, pos.y );
  p[2] = wxPoint( pos.x + size.x - 1, pos.y + size.y - 1 );
  p[3] = wxPoint( pos.x, pos.y + size.y - 1 );
  
  dc.DrawPolygon(4, p);
} // ingame_view::render_item_bounding_box()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render the grip of the selected item.
 * \param dc The device context for the drawings.
 * \param pos The position of the item.
 * \param size The size of the item.
 */
void bf::ingame_view::render_grip( wxDC& dc ) const
{
  CLAW_PRECOND( m_selected_item != NULL );

  wxRect box = get_bounding_box( *m_selected_item );
  box.SetPosition( box.GetPosition() - m_view );

  dc.SetPen(*wxRED_PEN);
  dc.SetBrush(*wxTRANSPARENT_BRUSH);

  wxPoint p[3];

  p[0] = wxPoint( box.x,  box.y );
  p[1] = wxPoint( p[0].x + s_grip_size, p[0].y );
  p[2] = wxPoint( p[0].x, p[0].y + s_grip_size );
  
  dc.DrawPolygon(3, p);

  p[0] = wxPoint( box.x + box.width - 1,  box.y );
  p[1] = wxPoint( p[0].x - s_grip_size, p[0].y );
  p[2] = wxPoint( p[0].x, p[0].y + s_grip_size );
  
  dc.DrawPolygon(3, p);

  p[0] = wxPoint( box.x,  box.y + box.height - 1 );
  p[1] = wxPoint( p[0].x + s_grip_size, p[0].y );
  p[2] = wxPoint( p[0].x, p[0].y - s_grip_size );
  
  dc.DrawPolygon(3, p);

  p[0] = wxPoint( box.x + box.width - 1,  box.y + box.height - 1 );
  p[1] = wxPoint( p[0].x - s_grip_size, p[0].y );
  p[2] = wxPoint( p[0].x, p[0].y - s_grip_size );
  
  dc.DrawPolygon(3, p);
} // ingame_view::render_grip()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render the result of the drag.
 * \param dc The device context for the drawings.
 */
void bf::ingame_view::render_drag( wxDC& dc ) const
{
  if ( m_drag_info != NULL )
    switch(m_drag_info->drag_mode)
      {
      case drag_info::drag_mode_selection:
        render_drag_mode_selection(dc);
        break;
      case drag_info::drag_mode_move:
        render_drag_mode_move(dc);
        break;
      case drag_info::drag_mode_size:
        render_drag_mode_size(dc);
        break;
      default:
        {
          // nothing to do
        }
      }
} // ingame_view::render_drag()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render the result of the drag in a situation of selection.
 * \param dc The device context for the drawings.
 */
void bf::ingame_view::render_drag_mode_selection( wxDC& dc ) const
{
  dc.SetPen( wxPen( *wxRED, 1, wxLONG_DASH ) );
  dc.SetBrush(*wxTRANSPARENT_BRUSH);

  wxPoint p[4];

  p[0] = m_drag_info->mouse_origin - m_view;
  p[1] = wxPoint( p[0].x + m_drag_info->delta().x, p[0].y );
  p[2] = wxPoint( p[1].x, p[0].y + m_drag_info->delta().y );
  p[3] = wxPoint( p[0].x, p[2].y );
  
  dc.DrawPolygon(4, p);
} // ingame_view::render_drag_mode_selection()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render the result of the drag in a situation of selection.
 * \param dc The device context for the drawings.
 */
void bf::ingame_view::render_drag_mode_move( wxDC& dc ) const
{
  std::set<item_instance*>::const_iterator it;

  for (it=m_selected_group.begin(); it!=m_selected_group.end(); ++it)
    {
      wxPoint pos
        ( (wxCoord)((*it)->get_pos_x() + m_drag_info->delta().x - m_view.x),
          (wxCoord)((*it)->get_pos_y() + m_drag_info->delta().y - m_view.y) );

      render_item(dc, **it, pos, true);
    }
} // ingame_view::render_drag_mode_move()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render the result of the drag in a situation of sizing.
 * \param dc The device context for the drawings.
 */
void bf::ingame_view::render_drag_mode_size( wxDC& dc ) const
{
  dc.SetPen( wxPen( *wxRED, 1, wxSHORT_DASH ) );
  dc.SetBrush(*wxTRANSPARENT_BRUSH);

  wxPoint p[4];

  // m_drag_info->mouse_origin is the immobile corner
  // m_drag_info->mouse_position is the moving corner
  p[0] = m_drag_info->mouse_origin - m_view;
  p[2] = m_drag_info->mouse_position - m_view;

  p[1] = wxPoint( p[2].x, p[0].y );
  p[3] = wxPoint( p[0].x, p[2].y );
  
  dc.DrawPolygon(4, p);
} // ingame_view::render_drag_mode_size()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render the background of the layer.
 * \param dc The device context for the drawings.
 */
void bf::ingame_view::render_layer_background( wxDC& dc ) const
{
  if ( !empty() )
    {
      dc.SetPen( *wxBLACK_PEN );
      dc.SetBrush( *wxBLACK_BRUSH );

      wxPoint p[4];

      p[0] = wxPoint( 0, 0 );
      p[1] = wxPoint( get_active_layer().get_width(), p[0].y );
      p[2] = wxPoint( p[1].x, get_active_layer().get_height() );
      p[3] = wxPoint( 0, p[2].y );
  
      dc.DrawPolygon(4, p);
    }
} // ingame_view::render_layer_background()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render the grid on the screen.
 * \param dc The device context for the drawings.
 */
void bf::ingame_view::render_grid( wxDC& dc ) const
{
  if ( !empty() && m_display_grid )
    {
      wxColour color(200,200,200);
      dc.SetPen( wxPen( color, 1, wxSHORT_DASH ) );
      dc.SetBrush(*wxTRANSPARENT_BRUSH);

      int width = get_active_layer().get_width();
      int height = get_active_layer().get_height();
      wxSize offset = m_grid.get_offset();
      wxSize step = m_grid.get_step();     
           
      for ( int column = m_view.x / step.x; 
            (column * step.x) + offset.x < width + m_view.x ; ++column ) 
        if ( (column * step.x) + offset.x > m_view.x ) 
          {
            wxPoint p[2];	  
            p[0] = wxPoint((column * step.x)+offset.x - m_view.x, 0 );
            p[1] = wxPoint( p[0].x, height );
            dc.DrawPolygon(2, p);
          }
             
      for ( int line = m_view.y / step.y; 
            (line * step.y) + offset.y < height + m_view.y ; ++line ) 
        if ( (line * step.y) + offset.y > m_view.y )
          {
            wxPoint p[2];
            p[0] = wxPoint( 0, (line * step.y) + offset.y - m_view.y);
            p[1] = wxPoint( width, p[0].y );
            dc.DrawPolygon(2, p);
          }      
    }
} // ingame_view::render_grid()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the visual (sprite and position) of an item.
 * \param item The item to render.
 * \param pos Reference position.
 * \return The sprite and the position where it has to be rendered.
 */
std::pair<wxBitmap, wxPoint> bf::ingame_view::get_item_visual
( const item_instance& item, const wxPoint& pos ) const
{
  CLAW_PRECOND( item.has_sprite() );

  wxBitmap result_bmp;
  wxPoint result_pos(pos);
  const sprite spr( item.get_sprite() );
  wxString name( std_to_wx_string(spr.get_image_name()) );

  if ( image_pool::get_instance().has_image(name) )
    {
      const wxRect sub_bmp
        ( spr.get_left(), spr.get_top(), 
	  spr.get_clip_width(), spr.get_clip_height() );

      result_bmp = 
	image_pool::get_instance().get_image(name).GetSubBitmap(sub_bmp);

      if ( spr.get_flip_x() || spr.get_flip_y() 
           || (spr.get_clip_width() != spr.get_width())
           || (spr.get_clip_height() != spr.get_height())
	   || (item.get_angle() != 0) )
        {
          wxImage img( result_bmp.ConvertToImage() );

          if ( spr.get_flip_x() )
            img = img.Mirror(true);

          if ( spr.get_flip_y() )
            img = img.Mirror(false);

          if ( (spr.get_clip_width() != spr.get_width())
               || (spr.get_clip_height() != spr.get_height()) )
            img.Rescale( spr.get_width(), spr.get_height() );

	  if ( item.get_angle() != 0 )
	    {
	      wxPoint center(spr.get_width() / 2, spr.get_height() / 2);
	      result_bmp = wxBitmap(img.Rotate(item.get_angle(), center));

              result_pos.x += center.x - result_bmp.GetWidth() / 2;
              result_pos.y += center.y - result_bmp.GetHeight() / 2;
	    }
	  else
	    result_bmp = wxBitmap(img);
        }
    }

  return std::pair<wxBitmap, wxPoint>(result_bmp, result_pos);
} // ingame_view::get_item_visual()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the pen used for displaying an item.
 * \param item The item to display.
 * \param active Tell if the item is in the active layer.
 */
wxPen
bf::ingame_view::get_display_pen( const item_instance& item, bool active ) const
{
  const item_class& item_c =
    item_class_pool::get_instance().get_item_class(item.get_class_name());

  wxPen result( wxColour( std_to_wx_string(item_c.get_color()) ) );

  if ( !active )
    result.SetStyle( wxLONG_DASH );
  else if ( m_selected_group.find((item_instance*)&item)
            != m_selected_group.end() )
    {
      result.SetColour(*wxRED);

      if ( &item != m_selected_item )
        result.SetStyle( wxSHORT_DASH );
    }

  return result;
} // ingame_view::get_display_pen()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the layer on which we are working.
 */
bf::layer& bf::ingame_view::current_layer()
{
  return m_history.get_level().get_layer(m_active_layer);
} // ingame_view::current_layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Pick the first item found near a given point.
 * \param pos The position of the point.
 */
bf::item_instance* bf::ingame_view::pick_item( const wxPoint& pos )
{
  item_instance* result=NULL;
  layer::item_iterator it;

  if ( !empty() )
    for (it=current_layer().item_begin();
         (result==NULL) && it!=current_layer().item_end(); ++it)
      {
        wxRect box = get_bounding_box( *it );

        if ( box.Contains(pos) )
          result = &(*it);
      }

  return result;
} // ingame_view::pick_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Pick all items in the last box dragged.
 * \param item (out) The items selected.
 * \param box The box where to pick the items.
 */
void bf::ingame_view::pick_item
( std::list<item_instance*>& item, const wxRect& box )
{
  layer::item_iterator it;

  if ( !empty() )
    for (it=current_layer().item_begin(); it!=current_layer().item_end(); ++it)
      {
        wxRect item_box = get_bounding_box( *it );

        if ( box.Intersects(item_box) )
          item.push_front( &(*it) );
      }
} // ingame_view::pick_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Change the selection on an item.
 * \param item The item to process.
 * \remark The function does nothing if item == NULL.
 */
void bf::ingame_view::toggle_selection( item_instance* item )
{
  if ( item != NULL )
    {
      if ( m_selected_group.find(item) == m_selected_group.end() )
        {
          m_selected_item = item;
          m_selected_group.insert(item);
        }
      else
        {
          m_selected_group.erase(item);

          if ( m_selected_item == item )
            {
              if ( m_selected_group.empty() )
                m_selected_item = NULL;
              else
                m_selected_item = *m_selected_group.begin();
            }
        }

      update_layout();
    }
} // ingame_view::toggle_selection()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add an item in the selection.
 * \param item The item to select.
 */
void bf::ingame_view::add_selection( item_instance* item )
{
  if ( item != NULL )
    {
      m_selected_group.insert( item );
      m_selected_item = item;
      update_layout();
    }
} // ingame_view::add_selection()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add some items in the selection.
 * \param item The items to select.
 */
void bf::ingame_view::add_selection( const std::list<item_instance*>& item )
{
  if ( !item.empty() )
    {
      m_selected_group.insert( item.begin(), item.end() );
      m_selected_item = item.front();
      update_layout();
    }
} // ingame_view::add_selection()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the selected items.
 * \param item The items to select.
 */
void bf::ingame_view::set_selection( const std::list<item_instance*>& item)
{
  if ( item.empty() )
    clear_selection();
  else
    set_selection( item, item.front() );
} // ingame_view::set_selection()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the selected items.
 * \param item The items to select.
 * \param selected Set this item as the selected item.
 * \param add_selection Indicates if the items are added at the selection.
 */
void bf::ingame_view::set_selection
( const std::list<item_instance*>& item, item_instance* selected,
  bool add_selection )
{
  CLAW_PRECOND( std::find( item.begin(), item.end(), selected ) != 
                item.end() );

  if ( ! add_selection )
    m_selected_group.clear();
    
  m_selected_group.insert( item.begin(), item.end() );
  m_selected_item = selected;
  
  update_layout();
} // ingame_view::set_selection()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the selected item.
 * \param item The item to select.
 */
void bf::ingame_view::set_selection( item_instance* item )
{
  if ( item == NULL )
    clear_selection();
  else
    {
      std::list<item_instance*> item_list;
      item_list.push_front(item);
      set_selection( item_list );
    }
} // ingame_view::set_selection()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the focus box of an item.
 * \param item The item to bound.
 * \param box (out) The bounding box of the item.
 */
wxRect bf::ingame_view::get_bounding_box( const item_instance& item ) const
{
  wxRect result;

  result.x = (int)item.get_pos_x();
  result.y = (int)item.get_pos_y();
  result.width = (int)item.get_width();
  result.height = (int)item.get_height();
  
  if ( result.width == 0 )
    {
      result.x -= s_point_size;
      result.width = 2 * s_point_size;
    }

  if ( result.height == 0 )
    {
      result.y -= s_point_size;
      result.height = 2 * s_point_size;
    }

  return result;
} // ingame_view::get_bounding_box()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the box bounding the bounding box of an item and its sprite, if
 *        any.
 * \param item The item to bound.
 */
wxRect bf::ingame_view::get_presence_box( const item_instance& item ) const
{
  wxRect result = get_bounding_box(item);
  wxPoint gap( item.get_gap_x(), item.get_gap_y() );

  result.SetPosition( result.GetPosition() + gap );
  result.width += std::abs( gap.x );
  result.height += std::abs( gap.y );

  return result;
} // ingame_view::get_presence_box()

/*----------------------------------------------------------------------------*/
/**
 * \brief Copy the selected items at the position where the mouse button was
 *        released.
 * \param add_selection Indicates if the items are added at the selection.
 */
void bf::ingame_view::copy_selection(bool add_selection)
{
  start_change();

  std::set<item_instance*>::iterator it;
  std::set<item_instance*> selection( m_selected_group );

  if ( !add_selection )
    m_selected_group.clear();

  for (it=selection.begin(); it!=selection.end(); ++it)
    {
      double x = (*it)->get_pos_x() + m_drag_info->delta().x;
      double y = (*it)->get_pos_y() + m_drag_info->delta().y;

      item_instance& item = current_layer().add_item(**it);

      item.set_position(x, y);
      m_selected_group.insert(&item);
      
      if ( *it == m_selected_item )
        m_selected_item = &item;
    }

  update_layout();
} // ingame_view::copy_selection()

/*----------------------------------------------------------------------------*/
/**
 * \brief Move the selected items at the position where the mouse button was
 *        released.
 */
void bf::ingame_view::move_selection()
{
  start_change();

  std::set<item_instance*>::iterator it;

  for (it=m_selected_group.begin(); it!=m_selected_group.end(); ++it)
    {
      double x = (*it)->get_pos_x() + m_drag_info->delta().x;
      double y = (*it)->get_pos_y() + m_drag_info->delta().y;

      (*it)->set_position(x, y);
    }

  update_layout();
} // ingame_view::move_selection()

/*----------------------------------------------------------------------------*/
/**
 * \brief Toggle the status of the graphic drawing.
 */
void bf::ingame_view::toggle_graphic_drawing()
{
  m_graphic_drawing = !m_graphic_drawing;
  Refresh();
} // ingame_view::toggle_graphic_drawing()

/*----------------------------------------------------------------------------*/
/**
 * \brief Toggle the status of the wireframe drawing.
 */
void bf::ingame_view::toggle_wireframe_drawing()
{
  m_wireframe_drawing = !m_wireframe_drawing;
  Refresh();
} // ingame_view::toggle_wireframe_drawing()

/*----------------------------------------------------------------------------*/
/**
 * \brief Calculate the gap between the grid and the selected item.
 * \param item_position The coordinate of the selected item.
 * \param offset_grid The oofset of the grid.
 * \param step_grid The step of the grid.
 */
int bf::ingame_view::update_coordinate_magnetism
( unsigned int item_position, unsigned int size_item, unsigned int offset_grid, 
  unsigned int step_grid )
{
  int result;
  int gap[4];
  gap[0] = (item_position - offset_grid) % step_grid;
  gap[1] = step_grid - gap[0];
  gap[2] = (item_position - offset_grid + size_item) % step_grid;
  gap[3] = step_grid - gap[2];

  unsigned int gap_positive;
  if ( gap[0] <= gap[2] )
    gap_positive = gap[0];
  else
    gap_positive = gap[2];

  unsigned int gap_negative;
  if ( gap[1] <= gap[3] )
    gap_negative = gap[1];
  else
    gap_negative = gap[3];

  result = 0;

  if ( gap_positive <= m_grid.get_magnetism_force() )
    {
      if ( gap_positive <= gap_negative )
    	result = -gap_positive;
    }
  else if ( gap_negative <= m_grid.get_magnetism_force() )
    result = gap_negative;

  return result;
} // ingame_view::update_coordinate_magnetism()

/*----------------------------------------------------------------------------*/
/**
 * \brief Update the position according to the magnetism.
 * \param mouse_position The position of the mouse in the level.
 */
void bf::ingame_view::update_mouse_position( const wxPoint& position )
{
  // magnetism
  wxPoint item_position;
  item_position.x = position.x
    + (wxCoord)m_selected_item->get_pos_x() - m_drag_info->mouse_origin.x; 
  item_position.y = position.y
    + (wxCoord)m_selected_item->get_pos_y() - m_drag_info->mouse_origin.y;
  
  wxPoint pos = position;

  pos.x += 
    update_coordinate_magnetism
    ( item_position.x, (unsigned int)m_selected_item->get_width(),
      m_grid.get_offset().x, m_grid.get_step().x );
  pos.y += 
    update_coordinate_magnetism
    ( item_position.y, (unsigned int)m_selected_item->get_height(),
      m_grid.get_offset().y, m_grid.get_step().y );
  
  m_drag_info->mouse_position = pos;
} // ingame_view::update_mouse_position()

/*----------------------------------------------------------------------------*/
/**
 * \brief Move the selection of one pixel.
 * \param  keycode The keycode pressed.
 */
void bf::ingame_view::move_pixel_selection(int keycode)
{
  int d_x = 0;
  int d_y = 0;

  switch( keycode )
    {
    case WXK_LEFT: d_x = -1; break;
    case WXK_UP: d_y = -1; break;
    case WXK_RIGHT: d_x = 1; break;
    case WXK_DOWN: d_y = 1; break;
    }

  start_change();

  std::set<item_instance*>::iterator it;

  for (it=m_selected_group.begin(); it!=m_selected_group.end(); ++it)
    {
      double x = (*it)->get_pos_x() + d_x;
      double y = (*it)->get_pos_y() + d_y;

      (*it)->set_position(x, y);
    }

  update_layout();
  
  Refresh();
} // ingame_view::move_pixel_selection()

/*----------------------------------------------------------------------------*/
/**
 * \brief Event Move the grid of one pixel.
 * \param keycode The keycode pressed.
 */
void bf::ingame_view::move_grid(int keycode)
{
  wxSize new_offset = m_grid.get_offset();
  
  switch( keycode )
    {
    case WXK_LEFT:
      if ( new_offset.x == 0 ) 
	new_offset.x = m_grid.get_step().x - 1;
      else
	--new_offset.x;
      break;
    case WXK_UP: 
      if ( new_offset.y == 0 ) 
	new_offset.y = m_grid.get_step().y - 1;
      else
	--new_offset.y;
      break;
    case WXK_RIGHT: 
      new_offset.x = ( new_offset.x + 1 ) % m_grid.get_step().x;
      break;
    case WXK_DOWN: 
      new_offset.y = ( new_offset.y + 1 ) % m_grid.get_step().y;
      break;
    }
  
  m_grid.set_offset(new_offset);

  Refresh();
} // ingame_view::move_grid()

/*----------------------------------------------------------------------------*/
/**
 * \brief Event sent to a resized window.
 * \param keycode The keycode pressed.
 */
void bf::ingame_view::move_view(int keycode)
{
  switch( keycode )
    {
    case WXK_LEFT:
      set_view_position
	( m_view.x - m_parent.get_h_scrollbar()->GetPageSize(), m_view.y );
      break;
    case WXK_DOWN:
      set_view_position
	( m_view.x, m_view.y + m_parent.get_v_scrollbar()->GetPageSize() ); 
      break;
    case WXK_UP: 
      set_view_position
	( m_view.x, m_view.y - m_parent.get_v_scrollbar()->GetPageSize() ); 
      break;
    case WXK_RIGHT:
      set_view_position
	( m_view.x + m_parent.get_h_scrollbar()->GetPageSize(), m_view.y );
      break;
    }

  Refresh();
} // ingame_view::move_view()

/*----------------------------------------------------------------------------*/
/**
 * \brief Write the mouse position.
 * \param point The position.
 */
void bf::ingame_view::write_mouse_position(const wxPoint& point)
{
  bool size_mode = false;

  if ( m_drag_info != NULL )
    if ( m_drag_info->drag_mode == drag_info::drag_mode_size )
      size_mode = true;

  if ( size_mode )
    {
      wxRect box = get_drag_mode_size_box();

      m_parent.GetStatusBar()->SetStatusText
        ( wxString::Format( _("x=%d"), box.x), 1 );
      m_parent.GetStatusBar()->SetStatusText
        ( wxString::Format( _("y=%d"), box.y), 2 );
      m_parent.GetStatusBar()->SetStatusText
        ( wxString::Format( _("width=%d"), box.width), 3 );
      m_parent.GetStatusBar()->SetStatusText
        ( wxString::Format( _("height=%d"), box.height), 4 );
    }
  else
    {
      m_parent.GetStatusBar()->SetStatusText
        ( wxString::Format( _("x=%d"), point.x), 1 );
      m_parent.GetStatusBar()->SetStatusText
        ( wxString::Format( _("y=%d"), point.y), 2 );

      item_instance* item = pick_item(point);
      wxString class_str, id_str;

      if ( item != NULL )
        {
          class_str = std_to_wx_string(item->get_class_name());

          if ( !item->get_id().empty() )
            id_str = _("id='") + std_to_wx_string(item->get_id()) + wxT("'");
        }

      m_parent.GetStatusBar()->SetStatusText( class_str, 0 );
      m_parent.GetStatusBar()->SetStatusText( id_str, 3 );
      m_parent.GetStatusBar()->SetStatusText( wxT(""), 4 );
    }
} // ingame_view::write_mouse_position()

/*----------------------------------------------------------------------------*/
/**
 * \brief Choose the next layaer as the current layer .
 */
void bf::ingame_view::next_layer()
{
  if ( m_history.get_level().layers_count() > 1 )
    {
      m_active_layer = ( m_active_layer + 1 ) % 
	m_history.get_level().layers_count();

      clear_selection();
    }
} // ingame_view::next_layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Choose the previous layaer as the current layer .
 */
void bf::ingame_view::previous_layer()
{
  if ( m_history.get_level().layers_count() > 1 )
    {
      if ( m_active_layer == 0 )
	m_active_layer = m_history.get_level().layers_count() - 1;
      else
	--m_active_layer;
      
      clear_selection();
    }
} // ingame_view::previous_layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Check if the user tries to resize an item and set the drag mode if it
 *        is the case.
 * \param pos The position of the mouse in the layer.
 * \pre m_drag_info->picked_item is the item on which the user clicked.
 * \pre pos in in the bounding box of m_drag_info->picked_item.
 */
void bf::ingame_view::set_drag_mode_size( const wxPoint& pos )
{
  wxRect box = get_bounding_box( *m_drag_info->picked_item );

  CLAW_PRECOND( box.Contains(pos) );

  const wxPoint top_left( box.x, box.y );
  const wxPoint top_right( box.x + box.width - 1, box.y );
  const wxPoint bottom_left( box.x, box.y + box.height - 1 );
  const wxPoint bottom_right( box.x + box.width - 1, box.y + box.height - 1 );

  bool grip = true;

  if ( (pos.x - top_left.x + pos.y - top_left.y) <= s_grip_size )
    m_drag_info->mouse_origin = bottom_right;
  else if ( (top_right.x - pos.x + pos.y - top_right.y) <= s_grip_size )
    m_drag_info->mouse_origin = bottom_left;
  else if ( (pos.x - bottom_left.x + bottom_left.y - pos.y) <= s_grip_size )
    m_drag_info->mouse_origin = top_right;
  else if ( (bottom_right.x - pos.x + bottom_right.y - pos.y) <= s_grip_size )
    m_drag_info->mouse_origin = top_left;
  else
    grip = false;

  if ( grip )
    {
      m_drag_info->drag_mode = drag_info::drag_mode_size;
      m_drag_info->mouse_position = pos;
    }
} // ingame_view::set_drag_mode_size()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the size of the box when resizing an item.
 */
wxRect bf::ingame_view::get_drag_mode_size_box() const
{
  CLAW_PRECOND( m_drag_info != NULL );
  CLAW_PRECOND( m_drag_info->drag_mode == drag_info::drag_mode_size );
  CLAW_PRECOND( m_drag_info->picked_item != NULL );

  wxRect result;

  result.x =
    std::min( m_drag_info->mouse_position.x, m_drag_info->mouse_origin.x );
  result.y =
    std::min( m_drag_info->mouse_position.y, m_drag_info->mouse_origin.y );
  result.width =
    std::abs( m_drag_info->mouse_position.x - m_drag_info->mouse_origin.x ) + 1;
  result.height =
    std::abs( m_drag_info->mouse_position.y - m_drag_info->mouse_origin.y ) + 1;

  return result;
} // ingame_view::get_drag_mode_size_box()

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the result of a drag in move mode.
 * \param ctrl Control key is pressed.
 * \param shift Shift key is pressed.
 * \param alt Alt key is pressed.
 */
void bf::ingame_view::apply_drag_mode_move( bool ctrl, bool shift, bool alt )
{
  CLAW_PRECOND( m_drag_info->drag_mode == drag_info::drag_mode_move );

  if ( m_drag_info->mouse_origin == m_drag_info->mouse_position )
    {
      if ( alt )
        add_selection(m_drag_info->picked_item);
      else if ( ctrl )
        toggle_selection(m_drag_info->picked_item);
    }
  else
    {
      if ( m_grid.get_magnetism_active() && !shift
           && (m_selected_item != NULL) ) 
        update_mouse_position(m_drag_info->mouse_position);
      
      if ( ctrl )
        copy_selection( alt );
      else
        move_selection();
    }
} // ingame_view::apply_drag_mode_move()

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the result of a drag in selection mode.
 * \param ctrl Control key is pressed.
 * \param alt Alt key is pressed.
 */
void bf::ingame_view::apply_drag_mode_selection( bool ctrl, bool alt )
{
  CLAW_PRECOND( m_drag_info->drag_mode == drag_info::drag_mode_selection );

  std::list<item_instance*> item;
  pick_item
    ( item, wxRect(m_drag_info->mouse_origin, m_drag_info->mouse_position) );

  if ( ctrl || alt )
    add_selection( item );
  else
    set_selection(item);
} // ingame_view::apply_drag_mode_selection()

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the result of a drag in size mode.
 * \param ctrl Control key is pressed.
 * \param alt Alt key is pressed.
 */
void bf::ingame_view::apply_drag_mode_size()
{
  CLAW_PRECOND( m_drag_info != NULL );
  CLAW_PRECOND( m_drag_info->drag_mode == drag_info::drag_mode_size );
  CLAW_PRECOND( m_drag_info->picked_item != NULL );

  wxRect box = get_drag_mode_size_box();

  if ( (box.x != m_drag_info->picked_item->get_pos_x())
       || (box.y != m_drag_info->picked_item->get_pos_y())
       || (box.width != m_drag_info->picked_item->get_width())
       || (box.height != m_drag_info->picked_item->get_height()) )
    {
      start_change();

      m_drag_info->picked_item->set_position(box.x, box.y);
      m_drag_info->picked_item->set_size(box.width, box.height);
    }

  update_layout();
} // ingame_view::apply_drag_mode_size()

/*----------------------------------------------------------------------------*/
/**
 * \brief Event sent to a resized window.
 * \param event The event.
 */
void bf::ingame_view::on_size(wxSizeEvent& event)
{
  Refresh();
} // ingame_view::on_size()

/*----------------------------------------------------------------------------*/
/**
 * \brief Draw the content of the window.
 * \param event The paint event.
 */
void bf::ingame_view::on_paint(wxPaintEvent& event)
{
  render();
} // ingame_view::on_paint()

/*----------------------------------------------------------------------------*/
/**
 * \brief The user start a click in the frame.
 * \param event The mouse event that occured.
 */
void bf::ingame_view::on_mouse_left_down(wxMouseEvent& event)
{
  item_instance* item = pick_item( m_view + event.GetPosition() );

  m_drag_info = new drag_info();
  m_drag_info->mouse_origin = event.GetPosition() + m_view;
  m_drag_info->mouse_position = m_drag_info->mouse_origin;

  if ( item == NULL )
    m_drag_info->drag_mode = drag_info::drag_mode_selection;
  else
    {
      m_drag_info->drag_mode = drag_info::drag_mode_move;

      if ( m_selected_group.find(item) == m_selected_group.end() )
        {
          if ( event.ControlDown() )
            toggle_selection( item );
          else
            set_selection(item);
        }
      else
        {
          m_drag_info->picked_item = item;
          set_drag_mode_size( event.GetPosition() + m_view );
        }
    }
} // ingame_view::on_mouse_left_down()

/*----------------------------------------------------------------------------*/
/**
 * \brief The user clicked in the frame.
 * \param event The mouse event that occured.
 */
void bf::ingame_view::on_mouse_move(wxMouseEvent& event)
{
  if ( event.LeftIsDown() )
    if ( m_drag_info != NULL )
      {
        int x(0), y(0);

        if ( event.GetX() > GetSize().x )
          x = event.GetX() - GetSize().x;
        else if ( event.GetX() < 0 )
          x = event.GetX();

        if ( event.GetY() > GetSize().y )
          y = event.GetY() - GetSize().y;
        else if ( event.GetY() < 0 )
          y = event.GetY();

        if ( (x!=0) || (y!=0) )
          set_view_position( m_view.x + x, m_view.y + y );

	if ( !m_grid.get_magnetism_active() || event.ShiftDown() 
	     || (m_selected_item == NULL) ) 
	  m_drag_info->mouse_position = event.GetPosition() + m_view;
	else
          update_mouse_position( m_view + event.GetPosition() );

        Refresh();
      }

  write_mouse_position( m_view + event.GetPosition() );
} // ingame_view::on_mouse_move()

/*----------------------------------------------------------------------------*/
/**
 * \brief The user click in the frame.
 * \param event The mouse event that occured.
 */
void bf::ingame_view::on_mouse_left_up(wxMouseEvent& event)
{
  if ( m_drag_info == NULL )
    event.Skip();
  else
    {
      m_drag_info->mouse_position = event.GetPosition() + m_view;

      if ( m_drag_info->drag_mode == drag_info::drag_mode_move )
        apply_drag_mode_move
          ( event.ControlDown(), event.ShiftDown(), event.AltDown() );
      else if ( m_drag_info->drag_mode == drag_info::drag_mode_selection )
        apply_drag_mode_selection( event.ControlDown(), event.AltDown() );
      else if ( m_drag_info->drag_mode == drag_info::drag_mode_size )
        apply_drag_mode_size();

      delete m_drag_info;
      m_drag_info = NULL;
    }
} // ingame_view::on_mouse_left_up()

/*----------------------------------------------------------------------------*/
/**
 * \brief The middle button of mouse turn.
 * \param event The mouse event that occured.
 */
void bf::ingame_view::on_mouse_wheel_rotation(wxMouseEvent& event)
{
  int rotation = event.GetWheelRotation();

  if ( rotation < 0 ) 
    if ( event.ShiftDown() )
      set_view_position
	( m_view.x + m_parent.get_h_scrollbar()->GetPageSize()/4, m_view.y );
    else
      set_view_position
	( m_view.x, m_view.y + m_parent.get_v_scrollbar()->GetPageSize()/4 );
  else
    if ( event.ShiftDown() )
      set_view_position
	( m_view.x - m_parent.get_h_scrollbar()->GetPageSize()/4, m_view.y );
    else
      set_view_position
	( m_view.x, m_view.y - m_parent.get_v_scrollbar()->GetPageSize()/4 );

  Refresh();

} // ingame_view::on_mouse_wheel_rotation()

/*----------------------------------------------------------------------------*/
/**
 * \brief The user pressed a keybord key.
 * \param event The keyboard event that occured.
 */
void bf::ingame_view::on_key_down(wxKeyEvent& event)
{
  switch( event.GetKeyCode() )
    {
    case WXK_LEFT:
    case WXK_UP:
    case WXK_RIGHT:
    case WXK_DOWN:
      if ( event.ControlDown() )
        move_pixel_selection(event.GetKeyCode());
      else if ( event.ShiftDown() )
	move_grid(event.GetKeyCode());
      else
	{
	  move_view(event.GetKeyCode());
	  wxPoint point(m_view.x + event.GetX(),m_view.y + event.GetY());
	  write_mouse_position(point);
	}
      break;
    default:
      event.Skip();
    }
} // ingame_view::on_key_down()

/*----------------------------------------------------------------------------*/
/**
 * \brief The user pressed a keybord key.
 * \param event The keyboard event that occured.
 */
void bf::ingame_view::on_key_up(wxKeyEvent& event)
{
  if ( !event.ControlDown() && !event.ShiftDown() && !event.AltDown() )
   switch( event.GetKeyCode() )
    {
    case WXK_PAGEUP: previous_layer(); break;
    case WXK_PAGEDOWN: next_layer(); break;
    case 'G': toggle_graphic_drawing(); break;
    case 'W': toggle_wireframe_drawing(); break;
    default:
      event.Skip();
    }
  else
    event.Skip();
} // ingame_view::on_key_up()

/*----------------------------------------------------------------------------*/
BEGIN_EVENT_TABLE(bf::ingame_view, wxWindow)
  EVT_SIZE( bf::ingame_view::on_size )
  EVT_PAINT( bf::ingame_view::on_paint )
  EVT_LEFT_UP( bf::ingame_view::on_mouse_left_up )
  EVT_LEFT_DOWN( bf::ingame_view::on_mouse_left_down )
  EVT_MOUSEWHEEL( bf::ingame_view::on_mouse_wheel_rotation )
  EVT_MOTION( bf::ingame_view::on_mouse_move )
  EVT_KEY_UP( bf::ingame_view::on_key_up )
  EVT_KEY_DOWN( bf::ingame_view::on_key_down )
END_EVENT_TABLE()
