/*
    Plee The Bear - Level editor

    Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file bf/code/item_class_pool.cpp
 * \brief Implementation of the bf::item_class_pool class.
 * \author Julien Jorge
 */
#include "bf/item_class_pool.hpp"

#include "bf/item_class_xml_parser.hpp"
#include <claw/assert.hpp>
#include <claw/logger.hpp>

/*----------------------------------------------------------------------------*/
/**
 * \brief Read all item files from a given directory and in its subdirectories.
 * \param dir_path The path to the directory to scan.
 */
void bf::item_class_pool::scan_directory( const std::string& dir_path )
{
  boost::filesystem::path dir(dir_path, boost::filesystem::native);

  if ( boost::filesystem::exists(dir) )
    {
      scan_sub_directory( dir );
      field_unicity_test();
      inheritance_test();
    }
} // item_class_pool::scan_directory()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if we have an item class with a given name.
 * \param class_name The name of the item class to check.
 */
bool bf::item_class_pool::has_item_class( const std::string& class_name ) const
{
  return m_item_class.find(class_name) != m_item_class.end();
} // item_class_pool::has_item_class()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the item class with a given name.
 * \param class_name The name of the item class we want.
 */
const bf::item_class&
bf::item_class_pool::get_item_class( const std::string& class_name ) const
{
  CLAW_PRECOND( has_item_class(class_name) );

  return m_item_class.find(class_name)->second;
} // item_class_pool::get_item_class()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get an iterator on the begining of the pool.
 */
bf::item_class_pool::iterator bf::item_class_pool::begin()
{
  return iterator( m_item_class.begin() );
} // item_class_pool::begin()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get an iterator just pas the end of the pool.
 */
bf::item_class_pool::iterator bf::item_class_pool::end()
{
  return iterator( m_item_class.end() );
} // item_class_pool::end()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get a constant iterator on the begining of the pool.
 */
bf::item_class_pool::const_iterator bf::item_class_pool::begin() const
{
  return const_iterator( m_item_class.begin() );
} // item_class_pool::begin()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get a constant iterator just pas the end of the pool.
 */
bf::item_class_pool::const_iterator bf::item_class_pool::end() const
{
  return const_iterator( m_item_class.end() );
} // item_class_pool::end()

/*----------------------------------------------------------------------------*/
/**
 * \brief Read an item file.
 * \param file_path The path to the file to read.
 */
void bf::item_class_pool::parse_file( const std::string& file_path )
{
  try
    {
      item_class_xml_parser item( file_path );
      m_item_class[ item.item.get_class_name() ] = item.item;
    }
  catch( std::exception& e )
    {
      claw::logger << claw::log_error << e.what() << claw::lendl;
    }
} // item_class_pool::parse_file()

/*----------------------------------------------------------------------------*/
/**
 * \brief Read all item files from a given directory and in its subdirectories.
 * \param dir The directory to scan.
 */
void
bf::item_class_pool::scan_sub_directory( const boost::filesystem::path& dir )
{
  boost::filesystem::directory_iterator it(dir);
  boost::filesystem::directory_iterator eit;

  static const std::string ext(".xml");

  for ( ; it!=eit; ++it)
    if ( boost::filesystem::is_directory(*it) )
      scan_sub_directory( *it );
    else if ( it->string().rfind(ext) == it->string().length() - ext.length() )
      parse_file( it->string() );
} // item_class_pool::scan_sub_directory()

/*----------------------------------------------------------------------------*/
/**
 * \brief Test if each super class of the current class are defined.
 * \param current The current class.
 * \param valid_classes Classes known to be valid.
 * \param not_valid_classes Classes known to be invalid.
 * \param inherit_set The classes seen in the scanned hierarchy.
 */
bool bf::item_class_pool::check_super_class
( const std::string& current, std::set<std::string>& valid_classes,
  std::set<std::string>& not_valid_classes,
  std::set<std::string>& inherit_set ) const
{
  bool result = false;
  
  if ( m_item_class.find(current) == m_item_class.end() )
    claw::logger << claw::log_warning << "The super class '" << current
                 << "' is not defined." << claw::lendl;
  else
    {
      result = recursive_super_class_check
        (current, valid_classes, not_valid_classes, inherit_set);

      if ( result )
	valid_classes.insert(current);
      else
	not_valid_classes.insert(current);
    }

  return result;
} // item_class_pool::check_super_class()

/*----------------------------------------------------------------------------*/
/**
 * \brief Call check_super_class on each super class of a given class.
 * \param current The current class.
 * \param valid_classes Classes known to be valid.
 * \param not_valid_classes Classes known to be invalid.
 * \param inherit_set The classes seen in the scanned hierarchy.
 */
bool bf::item_class_pool::recursive_super_class_check
( const std::string& current, std::set<std::string>& valid_classes,
  std::set<std::string>& not_valid_classes, 
  std::set<std::string>& inherit_set ) const
{
  bool result = true;
  std::list<std::string>::const_iterator it;
  const item_class& current_class = m_item_class.find(current)->second;

  for ( it=current_class.get_super_classes().begin();
        (it!=current_class.get_super_classes().end()) && result; ++it ) 
    if ( inherit_set.find(*it) == inherit_set.end() ) 
      {
        inherit_set.insert(*it);

        if ( valid_classes.find(*it) == valid_classes.end() ) 
          {
            if ( not_valid_classes.find(*it) != not_valid_classes.end() ) 
              result = false;
            else
              result = check_super_class
                (*it, valid_classes, not_valid_classes, inherit_set);
          }

        if ( !result )
          claw::logger << claw::log_warning << "Ignoring class '"
                       << current << "': requires a valid class named '"
                       << *it << "'" << claw::lendl;

        inherit_set.erase(*it);
      }
    else
      {
        result = false;
        claw::logger << claw::log_warning << "Ignoring class '" << current 
                     << "' circular inheritance" << claw::lendl;
      }

  return result;
} // item_class_pool::recursive_super_class_check()

/*----------------------------------------------------------------------------*/
/**
 * \brief Test if each super classes are defined.
 */
void bf::item_class_pool::inheritance_test()
{
  item_class_map::const_iterator it;
  std::set<std::string> valid_classes;
  std::set<std::string> not_valid_classes;
  
  for ( it = m_item_class.begin(); it != m_item_class.end(); ++it )
    if ( valid_classes.find(it->first) == valid_classes.end() )
      if ( not_valid_classes.find(it->first) == not_valid_classes.end() )
        {
          std::set<std::string> inherit_set;
          inherit_set.insert(it->first);

          check_super_class
            ( it->first, valid_classes, not_valid_classes, inherit_set );
        }

  std::set<std::string>::const_iterator it2;

  for ( it2=not_valid_classes.begin(); it2!=not_valid_classes.end(); ++it2 )
    m_item_class.erase(*it2);
} // item_class_pool::inheritance_test()

/*----------------------------------------------------------------------------*/
/**
 * \brief Test, for each classe, the unicity of fields.
 */
void bf::item_class_pool::field_unicity_test()
{
  item_class_map::const_iterator it;
  std::set<std::string> not_valid_classes;

  for ( it = m_item_class.begin(); it != m_item_class.end(); ++it )
    {
      std::string error_msg;
      if ( ! it->second.field_unicity_test(error_msg) )
        {  
          claw::logger << claw::log_warning << "Ignoring class '" << it->first 
                       << "' : " << error_msg << claw::lendl;
          
          not_valid_classes.insert(it->first);
        }
    }
      
  std::set<std::string>::const_iterator it2;

  for ( it2=not_valid_classes.begin(); it2!=not_valid_classes.end(); ++it2 )
    m_item_class.erase(*it2);
} // item_class_pool::control_sprite_size()
