/*
  Plee The Bear - Level editor

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file bf/code/item_instance.cpp
 * \brief Implementation of the bf::item_instance class.
 * \author Julien Jorge
 */
#include "bf/item_instance.hpp"
#include "bf/item_class_pool.hpp"

#include <claw/assert.hpp>
#include <sstream>

/*----------------------------------------------------------------------------*/
/**
 * \brief Default constructor.
 */
bf::item_instance::item_instance()
{

} // item_instance::item_instance()

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param class_name The name of the class I am an instance of.
 */
bf::item_instance::item_instance( const std::string& class_name )
  : m_class_name(class_name),
    m_fixed( item_class_pool::get_instance().
             get_item_class(m_class_name).get_fixable() )
{

} // item_instance::item_instance()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the name of the class I am an instance of.
 */
const std::string& bf::item_instance::get_class_name() const
{
  return m_class_name;
} // item_instance::get_class_name()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the the identifier of this item.
 */
const std::string& bf::item_instance::get_id() const
{
  return m_id;
} // item_instance::get_id()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the the identifier of this item.
 * \param id The new identifier of this item.
 */
void bf::item_instance::set_id( const std::string& id )
{
  m_id = id;
} // item_instance::set_id()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the value of a field.
 * \param field_name The name of the field to set.
 * \param v The value of the field.
 */
void bf::item_instance::set_value
( const std::string& field_name, const integer_type& v )
{
  m_int[field_name] = v;
} // item_instance::set_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the value of a field.
 * \param field_name The name of the field to set.
 * \param v The value of the field.
 */
void bf::item_instance::set_value
( const std::string& field_name, const u_integer_type& v )
{
  m_u_int[field_name] = v;
} // item_instance::set_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the value of a field.
 * \param field_name The name of the field to set.
 * \param v The value of the field.
 */
void bf::item_instance::set_value
( const std::string& field_name, const real_type& v )
{
  m_real[field_name] = v;
} // item_instance::set_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the value of a field.
 * \param field_name The name of the field to set.
 * \param v The value of the field.
 */
void bf::item_instance::set_value
( const std::string& field_name, const bool_type& v )
{
  m_bool[field_name] = v;
} // item_instance::set_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the value of a field.
 * \param field_name The name of the field to set.
 * \param v The value of the field.
 */
void bf::item_instance::set_value
( const std::string& field_name, const string_type& v )
{
  m_string[field_name] = v;
} // item_instance::set_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the value of a field.
 * \param field_name The name of the field to set.
 * \param v The value of the field.
 */
void bf::item_instance::set_value
( const std::string& field_name, const sprite& v )
{
  m_sprite[field_name] = v;
} // item_instance::set_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the value of a field.
 * \param field_name The name of the field to set.
 * \param v The value of the field.
 */
void bf::item_instance::set_value
( const std::string& field_name, const animation& v )
{
  m_animation[field_name] = v;
} // item_instance::set_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the value of a field.
 * \param field_name The name of the field to set.
 * \param v The value of the field.
 */
void bf::item_instance::set_value
( const std::string& field_name, const item_reference_type& v )
{
  m_item_reference[field_name] = v;
} // item_instance::set_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the value of a field.
 * \param field_name The name of the field to set.
 * \param v The value of the field.
 */
void bf::item_instance::set_value
( const std::string& field_name, const std::list<integer_type>& v )
{
  m_int_list[field_name] = v;
} // item_instance::set_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the value of a field.
 * \param field_name The name of the field to set.
 * \param v The value of the field.
 */
void bf::item_instance::set_value
( const std::string& field_name, const std::list<u_integer_type>& v )
{
  m_u_int_list[field_name] = v;
} // item_instance::set_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the value of a field.
 * \param field_name The name of the field to set.
 * \param v The value of the field.
 */
void bf::item_instance::set_value
( const std::string& field_name, const std::list<real_type>& v )
{
  m_real_list[field_name] = v;
} // item_instance::set_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the value of a field.
 * \param field_name The name of the field to set.
 * \param v The value of the field.
 */
void bf::item_instance::set_value
( const std::string& field_name, const std::list<bool_type>& v )
{
  m_bool_list[field_name] = v;
} // item_instance::set_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the value of a field.
 * \param field_name The name of the field to set.
 * \param v The value of the field.
 */
void bf::item_instance::set_value
( const std::string& field_name, const std::list<string_type>& v )
{
  m_string_list[field_name] = v;
} // item_instance::set_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the value of a field.
 * \param field_name The name of the field to set.
 * \param v The value of the field.
 */
void bf::item_instance::set_value
( const std::string& field_name, const std::list<sprite>& v )
{
  m_sprite_list[field_name] = v;
} // item_instance::set_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the value of a field.
 * \param field_name The name of the field to set.
 * \param v The value of the field.
 */
void bf::item_instance::set_value
( const std::string& field_name, const std::list<animation>& v )
{
  m_animation_list[field_name] = v;
} // item_instance::set_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the value of a field.
 * \param field_name The name of the field to set.
 * \param v The value of the field.
 */
void bf::item_instance::set_value
( const std::string& field_name, const std::list<item_reference_type>& v )
{
  m_item_reference_list[field_name] = v;
} // item_instance::set_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the value of a field.
 * \param field_name The name of the field to get.
 * \param v (out) The value of the field.
 */
void bf::item_instance::get_value
( const std::string& field_name, integer_type& v ) const
{
  CLAW_PRECOND( m_int.find(field_name) != m_int.end() );

  v = m_int.find(field_name)->second;
} // item_instance::get_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the value of a field.
 * \param field_name The name of the field to get.
 * \param v (out) The value of the field.
 */
void bf::item_instance::get_value
( const std::string& field_name, u_integer_type& v ) const
{
  CLAW_PRECOND( m_u_int.find(field_name) != m_u_int.end() );

  v = m_u_int.find(field_name)->second;
} // item_instance::get_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the value of a field.
 * \param field_name The name of the field to get.
 * \param v (out) The value of the field.
 */
void bf::item_instance::get_value
( const std::string& field_name, real_type& v ) const
{
  CLAW_PRECOND( m_real.find(field_name) != m_real.end() );

  v = m_real.find(field_name)->second;
} // item_instance::get_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the value of a field.
 * \param field_name The name of the field to get.
 * \param v (out) The value of the field.
 */
void bf::item_instance::get_value
( const std::string& field_name, bool_type& v ) const
{
  CLAW_PRECOND( m_bool.find(field_name) != m_bool.end() );

  v = m_bool.find(field_name)->second;
} // item_instance::get_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the value of a field.
 * \param field_name The name of the field to get.
 * \param v (out) The value of the field.
 */
void bf::item_instance::get_value
( const std::string& field_name, string_type& v ) const
{
  CLAW_PRECOND( m_string.find(field_name) != m_string.end() );

  v = m_string.find(field_name)->second;
} // item_instance::get_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the value of a field.
 * \param field_name The name of the field to get.
 * \param v (out) The value of the field.
 */
void bf::item_instance::get_value
( const std::string& field_name, sprite& v ) const
{
  CLAW_PRECOND( m_sprite.find(field_name) != m_sprite.end() );

  v = m_sprite.find(field_name)->second;
} // item_instance::get_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the value of a field.
 * \param field_name The name of the field to get.
 * \param v (out) The value of the field.
 */
void bf::item_instance::get_value
( const std::string& field_name, animation& v ) const
{
  CLAW_PRECOND( m_animation.find(field_name) != m_animation.end() );

  v = m_animation.find(field_name)->second;
} // item_instance::get_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the value of a field.
 * \param field_name The name of the field to get.
 * \param v (out) The value of the field.
 */
void bf::item_instance::get_value
( const std::string& field_name, item_reference_type& v ) const
{
  CLAW_PRECOND( m_item_reference.find(field_name) != m_item_reference.end() );

  v = m_item_reference.find(field_name)->second;
} // item_instance::get_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the value of a field.
 * \param field_name The name of the field to get.
 * \param v (out) The value of the field.
 */
void bf::item_instance::get_value
( const std::string& field_name, std::list<integer_type>& v ) const
{
  CLAW_PRECOND( m_int_list.find(field_name) != m_int_list.end() );

  v = m_int_list.find(field_name)->second;
} // item_instance::get_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the value of a field.
 * \param field_name The name of the field to get.
 * \param v (out) The value of the field.
 */
void bf::item_instance::get_value
( const std::string& field_name, std::list<u_integer_type>& v ) const
{
  CLAW_PRECOND( m_u_int_list.find(field_name) != m_u_int_list.end() );

  v = m_u_int_list.find(field_name)->second;
} // item_instance::get_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the value of a field.
 * \param field_name The name of the field to get.
 * \param v (out) The value of the field.
 */
void bf::item_instance::get_value
( const std::string& field_name, std::list<real_type>& v ) const
{
  CLAW_PRECOND( m_real_list.find(field_name) != m_real_list.end() );

  v = m_real_list.find(field_name)->second;
} // item_instance::get_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the value of a field.
 * \param field_name The name of the field to get.
 * \param v (out) The value of the field.
 */
void bf::item_instance::get_value
( const std::string& field_name, std::list<bool_type>& v ) const
{
  CLAW_PRECOND( m_bool_list.find(field_name) != m_bool_list.end() );

  v = m_bool_list.find(field_name)->second;
} // item_instance::get_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the value of a field.
 * \param field_name The name of the field to get.
 * \param v (out) The value of the field.
 */
void bf::item_instance::get_value
( const std::string& field_name, std::list<string_type>& v ) const
{
  CLAW_PRECOND( m_string_list.find(field_name) != m_string_list.end() );

  v = m_string_list.find(field_name)->second;
} // item_instance::get_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the value of a field.
 * \param field_name The name of the field to get.
 * \param v (out) The value of the field.
 */
void bf::item_instance::get_value
( const std::string& field_name, std::list<sprite>& v ) const
{
  CLAW_PRECOND( m_sprite_list.find(field_name) != m_sprite_list.end() );

  v = m_sprite_list.find(field_name)->second;
} // item_instance::get_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the value of a field.
 * \param field_name The name of the field to get.
 * \param v (out) The value of the field.
 */
void bf::item_instance::get_value
( const std::string& field_name, std::list<animation>& v ) const
{
  CLAW_PRECOND( m_animation_list.find(field_name) != m_animation_list.end() );

  v = m_animation_list.find(field_name)->second;
} // item_instance::get_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the value of a field.
 * \param field_name The name of the field to get.
 * \param v (out) The value of the field.
 */
void bf::item_instance::get_value
( const std::string& field_name, std::list<item_reference_type>& v ) const
{
  CLAW_PRECOND
    ( m_item_reference_list.find(field_name) != m_item_reference_list.end() );

  v = m_item_reference_list.find(field_name)->second;
} // item_instance::get_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if a field has a value.
 * \param f The field to check.
 */
bool bf::item_instance::has_value( const type_field& f ) const
{
    bool result = false;

  if ( f.is_list() )
    switch ( f.get_field_type() )
      {
      case type_field::integer_field_type:
        result = m_int_list.find(f.get_name()) != m_int_list.end();
        break;
      case type_field::u_integer_field_type:
        result = m_u_int_list.find(f.get_name()) != m_u_int_list.end();
        break;
      case type_field::real_field_type:
        result = m_real_list.find(f.get_name()) != m_real_list.end();
        break;
      case type_field::boolean_field_type:
        result = m_bool_list.find(f.get_name()) != m_bool_list.end();
        break;
      case type_field::string_field_type:
        result = m_string_list.find(f.get_name()) != m_string_list.end();
        break;
      case type_field::sprite_field_type:
        result = m_sprite_list.find(f.get_name()) != m_sprite_list.end();
        break;
      case type_field::animation_field_type:
        result = m_animation_list.find(f.get_name()) != m_animation_list.end();
        break;
      case type_field::item_reference_field_type:
        result = ( m_item_reference_list.find(f.get_name())
                   != m_item_reference_list.end() );
        break;
      }
  else
    switch ( f.get_field_type() )
      {
      case type_field::integer_field_type:
        result = m_int.find(f.get_name()) != m_int.end();
        break;
      case type_field::u_integer_field_type:
        result = m_u_int.find(f.get_name()) != m_u_int.end();
        break;
      case type_field::real_field_type:
        result = m_real.find(f.get_name()) != m_real.end();
        break;
      case type_field::boolean_field_type:
        result = m_bool.find(f.get_name()) != m_bool.end();
        break;
      case type_field::string_field_type:
        result = m_string.find(f.get_name()) != m_string.end();
        break;
      case type_field::sprite_field_type:
        result = m_sprite.find(f.get_name()) != m_sprite.end();
        break;
      case type_field::animation_field_type:
        result = m_animation.find(f.get_name()) != m_animation.end();
        break;
      case type_field::item_reference_field_type:
        result = m_item_reference.find(f.get_name()) != m_item_reference.end();
        break;
      }

  return result;
} // item_instance::has_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Delete the value of a field.
 * \param f The field to delete.
 */
void bf::item_instance::delete_value( const type_field& f )
{
  if ( has_value(f) )
    {
      if ( f.is_list() )
        switch ( f.get_field_type() )
          {
          case type_field::integer_field_type:
            m_int_list.erase( f.get_name() );
            break;
          case type_field::u_integer_field_type:
            m_u_int_list.erase( f.get_name() );
            break;
          case type_field::real_field_type:
            m_real_list.erase( f.get_name() );
            break;
          case type_field::boolean_field_type:
            m_bool_list.erase( f.get_name() );
            break;
          case type_field::string_field_type:
            m_string_list.erase( f.get_name() );
            break;
          case type_field::sprite_field_type:
            m_sprite_list.erase( f.get_name() );
            break;
          case type_field::animation_field_type:
            m_animation_list.erase( f.get_name() );
            break;
          case type_field::item_reference_field_type:
            m_item_reference_list.erase( f.get_name() );
            break;
          }
      else
        switch ( f.get_field_type() )
          {
          case type_field::integer_field_type:
            m_int.erase( f.get_name() );
            break;
          case type_field::u_integer_field_type:
            m_u_int.erase( f.get_name() );
            break;
          case type_field::real_field_type:
            m_real.erase( f.get_name() );
            break;
          case type_field::boolean_field_type:
            m_bool.erase( f.get_name() );
            break;
          case type_field::string_field_type:
            m_string.erase( f.get_name() );
            break;
          case type_field::sprite_field_type:
            m_sprite.erase( f.get_name() );
            break;
          case type_field::animation_field_type:
            m_animation.erase( f.get_name() );
            break;
          case type_field::item_reference_field_type:
            m_item_reference.erase( f.get_name() );
            break;
          }
    }
} // item_instance::delete_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the item is fixed.
 */
bool bf::item_instance::get_fixed() const
{
  const item_class& my_class
    ( item_class_pool::get_instance().get_item_class(m_class_name) );

  return m_fixed && my_class.get_fixable();
} // item_instance::get_fixed()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell that this item is fixed.
 * \param b Fixed or not ?
 */
void bf::item_instance::set_fixed( bool b )
{
  m_fixed = b;
} // item_instance::set_fixed()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the x-position of the item, if any.
 */
double bf::item_instance::get_pos_x() const
{
  double result = 0;
  std::map<std::string, real_type>::const_iterator it;

  it = m_real.find("pos_x");

  if ( it!=m_real.end() )
    result = it->second.get_value();

  return result;
} // item_instance::get_pos_x()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the y-position of the item, if any.
 */
double bf::item_instance::get_pos_y() const
{
  double result = 0;
  std::map<std::string, real_type>::const_iterator it;

  it = m_real.find("pos_y");

  if ( it!=m_real.end() )
    result = it->second.get_value();

  return result;
} // item_instance::get_pos_y()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the z-position of the item, if any.
 */
int bf::item_instance::get_pos_z() const
{
  int result = 0;
  std::map<std::string, integer_type>::const_iterator it;

  it = m_int.find("pos_z");

  if ( it!=m_int.end() )
    result = it->second.get_value();

  return result;
} // item_instance::get_pos_z()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the width of the item, if any.
 */
double bf::item_instance::get_width() const
{
  double result = 0;
  std::map<std::string, real_type>::const_iterator it;

  it = m_real.find("width");

  if ( it!=m_real.end() )
    result = it->second.get_value();
  else if ( has_sprite() )
    result = get_sprite().get_width();

  return result;
} // item_instance::get_width()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the height of the item, if any.
 */
double bf::item_instance::get_height() const
{
  double result = 0;
  std::map<std::string, real_type>::const_iterator it;

  it = m_real.find("height");

  if ( it!=m_real.end() )
   result = it->second.get_value();
  else if ( has_sprite() )
    result = get_sprite().get_height();

  return result;
} // item_instance::get_height()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the angle of the item, if any.
 */
double bf::item_instance::get_angle() const
{
  double result = 0;
  std::map<std::string, real_type>::const_iterator it;

  it = m_real.find("angle");

  if ( it!=m_real.end() )
   result = it->second.get_value();

  return result;
} // item_instance::get_angle()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the x-gap of the picture of the item, if any.
 */
int bf::item_instance::get_gap_x() const
{
  int result = 0;
  std::map<std::string, integer_type>::const_iterator it;

  it = m_int.find("gap_x");

  if ( it!=m_int.end() )
    result = it->second.get_value();

  return result;
} // item_instance::get_gap_x()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the y-gap of the picture of the item, if any.
 */
int bf::item_instance::get_gap_y() const
{
  int result = 0;
  std::map<std::string, integer_type>::const_iterator it;

  it = m_int.find("gap_y");

  if ( it!=m_int.end() )
    result = it->second.get_value();

  return result;
} // item_instance::get_gap_y()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if this item has a sprite.
 */
bool bf::item_instance::has_sprite() const
{
  return has_sprite_in_sprites() || has_sprite_in_animations();
} // item_instance::has_sprite()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the default sprite of this item.
 */
const bf::sprite& bf::item_instance::get_sprite() const
{
  CLAW_PRECOND( has_sprite() );

  if ( has_sprite_in_sprites() )
    return get_sprite_in_sprites();
  else // if ( has_sprite_in_animations() )
    return get_sprite_in_animations();
} // item_instance::get_sprite()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the position of the item, if it has this kind of field.
 */
void bf::item_instance::set_position( double x, double y )
{
  const item_class& my_class
    ( item_class_pool::get_instance().get_item_class(m_class_name) );

  if ( my_class.has_field("pos_x", type_field::real_field_type) )
    set_value( "pos_x", real_type(x) );

  if ( my_class.has_field("pos_y", type_field::real_field_type) )
    set_value( "pos_y", real_type(y) );
} // item_instance::set_position()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the size of the item, if it has this kind of field.
 */
void bf::item_instance::set_size( double x, double y )
{
  const item_class& my_class
    ( item_class_pool::get_instance().get_item_class(m_class_name) );

  if ( my_class.has_field("width", type_field::real_field_type) )
    set_value( "width", real_type(x) );

  if ( my_class.has_field("height", type_field::real_field_type) )
    set_value( "height", real_type(y) );
} // item_instance::set_size()

/*----------------------------------------------------------------------------*/
/**
 * \brief Compile the fields of the item.
 * \param f The file in which we compile.
 * \param id_to_int Integer values associated to the items id, to use for
 *        references.
 */
void bf::item_instance::compile
( compiled_file& f, const std::map<std::string, unsigned int>& id_to_int ) const
{
  const item_class& my_class
    ( item_class_pool::get_instance().get_item_class(m_class_name) );

  f << get_fixed();

  std::list<std::string> fields;

  for( sort_fields(fields); !fields.empty(); fields.pop_front() )
    compile_field( f, my_class.get_field(fields.front()), id_to_int );
} // item_instance::compile()

/*----------------------------------------------------------------------------*/
/**
 * \brief Test if the item has a good initialization.
 * Return "true" if there are no problem.
 * Test item references and required fields.
 * \param list_id The set of identifiant.
 * \param error_msg The string of error message.
 */
bool bf::item_instance::check
( std::set<std::string>& list_id, std::string& error_msg) const
{
  bool result = false;
  if ( check_item_references(list_id,error_msg) ) 
    if ( check_required_fields(error_msg) ) 
      result = true;
      
  return result;
} // item_instance::check()

/*----------------------------------------------------------------------------*/
/**
 * \brief Test if each item reference is known.
 * Return "true" if there are no problem.
 * \param list_id The set of identifiant.
 * \param error_msg The string of error message.
 */
bool bf::item_instance::check_item_references
( std::set<std::string>& list_id, std::string& error_msg ) const
{
  bool result = true;  
  std::map<std::string, item_reference_type>::const_iterator it1;
  std::map<std::string, std::list<item_reference_type> >::const_iterator it2;
  
  for ( it1 = m_item_reference.begin(); 
	( it1 != m_item_reference.end() ) && result ; it1++ )
    if( !it1->second.get_value().empty() ) 
      if ( list_id.find(it1->second.get_value()) == list_id.end() )
	{
	  result = false;
	  error_msg = "Error item " + get_class_name() + 
	    " : field '" + it1->first + "' : " + 
	    it1->second.get_value() + " is not a identifier" ;
	}
    
  for ( it2 = m_item_reference_list.begin(); 
	( it2 != m_item_reference_list.end() ) && result ; it2++ )
    {
      std::list<item_reference_type>::const_iterator it;
      
      for ( it = it2->second.begin(); 
	    ( it != it2->second.end() ) && result ; it++ )
	if( !it->get_value().empty() ) 
	  if ( list_id.find(it->get_value()) == list_id.end() )
	    {
	      error_msg = "Error item " + get_class_name() + 
		" : field '" + it2->first + "' : " + 
		it->get_value() + " is not a identifier" ;
	      result = false;
	    }
    }

  return result;
} // item_instance::check_item_references()
   
/*----------------------------------------------------------------------------*/
/**
 * \brief Test if each required fields are setted.
 * Return "true" if there are no problem.
 * \param error_msg The string of error message.
 */
bool bf::item_instance::check_required_fields(std::string& error_msg) const
{
  bool result = true;

  const item_class& my_class
    ( item_class_pool::get_instance().get_item_class(m_class_name) );

  item_class::field_iterator it;
  for ( it = my_class.field_begin(); 
	( it != my_class.field_end() ) && result; ++it) 
    if( it->get_required() ) 
      if( ! has_value(*it) )
	{
	  result = false;
	  error_msg = "Error item " + get_class_name() + 
	    " : field '" + it->get_name() + "' is required";
	}

  return result;
} // item_instance::check_required_fields()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if this item has a sprite in the sprites.
 */
bool bf::item_instance::has_sprite_in_sprites() const
{
  return !m_sprite.empty();
} // item_instance::has_sprite_in_sprites()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if this item has a sprite in the animations.
 */
bool bf::item_instance::has_sprite_in_animations() const
{
  bool result = false;

  std::map<std::string, animation>::const_iterator it;

  for (it=m_animation.begin(); !result && (it!=m_animation.end()); ++it)
    result = !it->second.empty();

  return result;
} // item_instance::has_sprite_in_animations()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the default sprite among the sprites of this item.
 */
const bf::sprite& bf::item_instance::get_sprite_in_sprites() const
{
  CLAW_PRECOND( has_sprite_in_sprites() );

  const sprite* result;

  std::map<std::string, sprite>::const_iterator it;
  it = m_sprite.find("sprite");

  if ( it!=m_sprite.end() )
    result = &it->second;
  else
    result = &m_sprite.begin()->second;

  return *result;
} // item_instance::get_sprite_in_sprites()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the default sprite among the animations of this item.
 */
const bf::sprite& bf::item_instance::get_sprite_in_animations() const
{
  CLAW_PRECOND( has_sprite_in_animations() );

  const sprite* result = NULL;

  std::map<std::string, animation>::const_iterator it;
  it = m_animation.find("animation");

  if ( it!=m_animation.end() )
    if ( !it->second.empty() )
      result = &it->second[0].get_sprite();

  for (it=m_animation.begin(); (result==NULL) && (it!=m_animation.end()); ++it)
    if ( !it->second.empty() )
      result = &it->second[0].get_sprite();

  return *result;
} // item_instance::get_sprite_in_animations()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the name of the fields, ordered so that fields depending on other
 *        fields are after them in the list.
 * \param fields (out) The name of the fields.
 */
void bf::item_instance::sort_fields( std::list<std::string>& fields ) const
{
  std::set<std::string> all_fields;

  copy_field_names( m_int, all_fields );
  copy_field_names( m_u_int, all_fields );
  copy_field_names( m_real, all_fields );
  copy_field_names( m_bool, all_fields );
  copy_field_names( m_string, all_fields );
  copy_field_names( m_sprite, all_fields );
  copy_field_names( m_animation, all_fields );
  copy_field_names( m_item_reference, all_fields );
  copy_field_names( m_int_list, all_fields );
  copy_field_names( m_u_int_list, all_fields );
  copy_field_names( m_real_list, all_fields );
  copy_field_names( m_bool_list, all_fields );
  copy_field_names( m_string_list, all_fields );
  copy_field_names( m_sprite_list, all_fields );
  copy_field_names( m_animation_list, all_fields );
  copy_field_names( m_item_reference_list, all_fields );

  while ( !all_fields.empty() )
    insert_field( *all_fields.begin(), fields, all_fields );
} // item_instance::sort_fields()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the name of the fields, ordered so that fields depending on other
 *        fields are after them in the list.
 * \param field_name The name of the field to insert.
 * \param fields (out) The name of the fields.
 * \param all_fields (in/out) All the remaining fields to insert.
 */
void bf::item_instance::insert_field
( const std::string& field_name, std::list<std::string>& fields,
  std::set<std::string>& all_fields ) const
{
  if ( all_fields.find(field_name) != all_fields.end() )
    {
      all_fields.erase(field_name);

      const item_class& my_class
        ( item_class_pool::get_instance().get_item_class(m_class_name) );
      std::set<std::string>::const_iterator it;

      for ( it=my_class.get_field(field_name).get_preceding().begin();
            it!=my_class.get_field(field_name).get_preceding().end(); ++it )
        insert_field(*it, fields, all_fields);

      fields.push_back( field_name );
    }
} // item_instance::insert_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Compile a field.
 * \param f The file in which we compile.
 * \param field The field to save.
 * \param id_to_int Integer values associated to the items id, to use for
 *        references.
 */
void bf::item_instance::compile_field
( compiled_file& f, const type_field& field,
  const std::map<std::string, unsigned int>& id_to_int ) const
{
  CLAW_PRECOND( has_value(field) );

  if ( field.is_list() )
    f << bear::level_code_value::field_list;

  f << get_code_value(field) << field.get_name();

  if ( field.is_list() )
    compile_field_list(f, field, id_to_int);
  else
    compile_field_single(f ,field, id_to_int);
} // item_instance::compile_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Compile a field made of a single values.
 * \param f The file in which we compile.
 * \param field The field to save.
 * \param id_to_int Integer values associated to the items id, to use for
 *        references.
 */
void bf::item_instance::compile_field_single
( compiled_file& f, const type_field& field,
  const std::map<std::string, unsigned int>& id_to_int ) const
{
  CLAW_PRECOND( has_value(field) );
  const std::string& field_name = field.get_name();

  switch ( field.get_field_type() )
    {
    case type_field::integer_field_type:
      m_int.find(field_name)->second.compile(f);
      break;
    case type_field::u_integer_field_type:
      m_u_int.find(field_name)->second.compile(f);
      break;
    case type_field::real_field_type:
      m_real.find(field_name)->second.compile(f);
      break;
    case type_field::boolean_field_type:
      m_bool.find(field_name)->second.compile(f);
      break;
    case type_field::string_field_type:
      m_string.find(field_name)->second.compile(f);
      break;
    case type_field::sprite_field_type:
      m_sprite.find(field_name)->second.compile(f);
      break;
    case type_field::animation_field_type:
      m_animation.find(field_name)->second.compile(f);
      break;
    case type_field::item_reference_field_type:
      f << id_to_int.find
        ( m_item_reference.find(field_name)->second.get_value())->second;
      break;
    }
} // item_instance::compile_field_single()

/*----------------------------------------------------------------------------*/
/**
 * \brief Compile a field made of a list of values.
 * \param f The file in which we compile.
 * \param field The field to save.
 * \param id_to_int Integer values associated to the items id, to use for
 *        references.
 */
void bf::item_instance::compile_field_list
( compiled_file& f, const type_field& field,
  const std::map<std::string, unsigned int>& id_to_int ) const
{
  CLAW_PRECOND( has_value(field) );
  const std::string& field_name = field.get_name();

  switch ( field.get_field_type() )
    {
    case type_field::integer_field_type:
      compile_list( f, m_int_list.find(field_name)->second );
      break;
    case type_field::u_integer_field_type:
      compile_list( f, m_u_int_list.find(field_name)->second );
      break;
    case type_field::real_field_type:
      compile_list( f, m_real_list.find(field_name)->second );
      break;
    case type_field::boolean_field_type:
      compile_list( f, m_bool_list.find(field_name)->second );
      break;
    case type_field::string_field_type:
      compile_list( f, m_string_list.find(field_name)->second );
      break;
    case type_field::sprite_field_type:
      compile_list( f, m_sprite_list.find(field_name)->second );
      break;
    case type_field::animation_field_type:
      compile_list( f, m_animation_list.find(field_name)->second );
      break;
    case type_field::item_reference_field_type:
      {
        f << m_item_reference_list.find(field_name)->second.size();

        std::list<item_reference_type>::const_iterator it =
          m_item_reference_list.find(field_name)->second.begin();
        const std::list<item_reference_type>::const_iterator eit =
          m_item_reference_list.find(field_name)->second.end();

        for ( ; it!=eit; ++it)
          f << id_to_int.find(it->get_value())->second;

        break;
      }
    }
} // item_instance::compile_field_list()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the compiler code value for type of field.
 * \param field The field forwhich we want the value.
 */
bear::level_code_value::value_type
bf::item_instance::get_code_value( const type_field& field ) const
{
  bear::level_code_value::value_type result;

  switch ( field.get_field_type() )
    {
    case type_field::integer_field_type:
      result = bear::level_code_value::field_int;
      break;
    case type_field::u_integer_field_type:
      result = bear::level_code_value::field_u_int;
      break;
    case type_field::real_field_type:
      result = bear::level_code_value::field_real;
      break;
    case type_field::boolean_field_type:
      result = bear::level_code_value::field_bool;
      break;
    case type_field::string_field_type:
      result = bear::level_code_value::field_string;
      break;
    case type_field::sprite_field_type:
      result = bear::level_code_value::field_sprite;
      break;
    case type_field::animation_field_type:
      result = bear::level_code_value::field_animation;
      break;
    case type_field::item_reference_field_type:
      result = bear::level_code_value::field_item;
      break;
    }

  return result;
} // item_instance::compile_fields()
