/*
    Plee The Bear - Level editor

    Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file bf/code/layer.cpp
 * \brief Implementation of the bf::layer class.
 * \author Julien Jorge
 */
#include "bf/layer.hpp"

#include "level_code_value.hpp"

#include <claw/assert.hpp>

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param width The width of the layer.
 * \param height The height of the layer.
 * \param index The position of the layer in the stack.
 * \param layer_type The type of the layer.
 */
bf::layer::layer
( unsigned int width, unsigned int height, const std::string& layer_type )
  : m_width(width), m_height(height), m_layer_type(layer_type)
{
  CLAW_PRECOND( width > 0 );
  CLAW_PRECOND( height > 0 );
} // layer::layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Copy constructor.
 * \param that The layer to copy from.
 */
bf::layer::layer( const layer& that )
{
  assign(that);
} // layer::layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
bf::layer::~layer()
{
  clear();
} // layer::~layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Change the size of the layer.
 * \param width The new width of the layer.
 * \param height The new height of the layer.
 */
void bf::layer::resize( unsigned int width, unsigned int height )
{
  CLAW_PRECOND( width > 0 );
  CLAW_PRECOND( height > 0 );

  m_width = width;
  m_height = height;
} // layer::resize()

/*----------------------------------------------------------------------------*/
/**
 * \brief Change the name of the class of the layer.
 * \param class_name The new name of the class.
 */
void bf::layer::set_class_name( const std::string& class_name )
{
  CLAW_PRECOND( !class_name.empty() );

  m_layer_type = class_name;
} // layer::set_class_name()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add an item in the layer.
 * \param item The item to add.
 * \remark The layer stores and returns a copy of \a item.
 */
bf::item_instance& bf::layer::add_item( const item_instance& item )
{
  item_instance* result = new item_instance(item);
  m_item.insert(result);
  return *result;
} // layer::add_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Remove an item from the layer.
 * \param item The item to remove.
 * \post item is no more valid.
 */
void bf::layer::remove_item( item_instance& item )
{
  CLAW_PRECOND( m_item.find(&item) != m_item.end() );

  m_item.erase(&item);
  delete &item;
} // layer::remove_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the type of the layer.
 */
const std::string& bf::layer::get_class_name() const
{
  return m_layer_type;
} // layer::get_class_name()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the width of the layer.
 */
const unsigned int bf::layer::get_width() const
{
  return m_width;
} // layer::get_width()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the height of the layer.
 */
const unsigned int bf::layer::get_height() const
{
  return m_height;
} // layer::get_height()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if there is no item in the layer.
 */
bool bf::layer::empty() const
{
  return m_item.empty();
} // layer::empty()

/*----------------------------------------------------------------------------*/
/**
 * \brief Test if the layer is valid.
 * Return the bad item (NULL if layer is valid).
 * \param error_msg The error message.
 */
bf::item_instance* bf::layer::check( std::string& error_msg )
{
  item_instance* item = NULL;
  item_set_type::const_iterator it;
  
  std::set<std::string> list_id;
  for (it=m_item.begin(); (it!=m_item.end()); ++it)
    if ( !(*it)->get_id().empty() )
      list_id.insert((*it)->get_id());
  
  for (it=m_item.begin(); (it!=m_item.end()) && (item == NULL); ++it)
    if ( ! (*it)->check(list_id,error_msg) )
	item = *it;
  
  return item;
} // layer::check()

/*----------------------------------------------------------------------------*/
/**
 * \brief Compile the layer.
 * \param f The file in which we compile.
 */
void bf::layer::compile( compiled_file& f ) const
{
  f << m_layer_type << m_width << m_height;

  std::map<std::string, unsigned int> id_to_int;
  std::list<item_instance*> referenced, not_referenced;
  std::list<item_instance*>::const_iterator iti;
  unsigned int index = 0;
  item_set_type::const_iterator it;

  for (it=m_item.begin(); it!=m_item.end(); ++it)
    if ( (*it)->get_id().empty() )
      not_referenced.push_front(*it);
    else
      {
        id_to_int[(*it)->get_id()] = index;
        ++index;
        referenced.push_back(*it);
      }

  f << m_item.size();

  // declaration of referenced items
  if ( !referenced.empty() )
    {
      f << bear::level_code_value::item_declaration << referenced.size();

      for (iti=referenced.begin(); iti!=referenced.end(); ++iti)
        f << (*iti)->get_class_name();

      // definition of referenced items
      for (iti=referenced.begin(); iti!=referenced.end(); ++iti)
        {
          f << bear::level_code_value::item_definition;
          (*iti)->compile( f, id_to_int );
        }
    }

  // not referenced items
  for (iti=not_referenced.begin(); iti!=not_referenced.end(); ++iti)
    {
      f << bear::level_code_value::base_item << (*iti)->get_class_name();
      (*iti)->compile( f, id_to_int );
    }
} // layer::compile()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get an iterator on the set of items.
 */
bf::layer::item_iterator bf::layer::item_begin()
{
  return m_item.begin();
} // layer::item_begin()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get an iterator past the end of the set of items.
 */
bf::layer::item_iterator bf::layer::item_end()
{
  return m_item.end();
} // layer::item_end()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get an iterator on the set of items.
 */
bf::layer::const_item_iterator bf::layer::item_begin() const
{
  return m_item.begin();
} // layer::item_begin()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get an iterator past the end of the set of items.
 */
bf::layer::const_item_iterator bf::layer::item_end() const
{
  return m_item.end();
} // layer::item_end()

/*----------------------------------------------------------------------------*/
/**
 * \brief Assignment operator.
 * \param that The instance to copy from.
 */
bf::layer& bf::layer::operator=( const layer& that )
{
  if ( &that != this )
    {
      clear();
      assign( that );
    }

  return *this;
} // layer::operator=()

/*----------------------------------------------------------------------------*/
/**
 * \brief Remove all items.
 */
void bf::layer::clear()
{
  item_set_type::iterator it;

  for (it=m_item.begin(); it!=m_item.end(); ++it)
    delete *it;

  m_item.clear();
} // layer::clear()

/*----------------------------------------------------------------------------*/
/**
 * \brief Assign a layer to this one.
 * \param that The instance to copy from.
 */
void bf::layer::assign( const layer& that )
{
  CLAW_PRECOND( m_item.empty() );
  CLAW_PRECOND( &that != this );

  m_width = that.m_width;
  m_height = that.m_height;
  m_layer_type = that. m_layer_type;

  item_set_type::const_iterator it;

  for (it=that.m_item.begin(); it!=that.m_item.end(); ++it)
    m_item.insert( new item_instance( **it ) );
} // layer::assign()
