/*
    Plee The Bear - Level editor

    Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file bf/sprite_edit.cpp
 * \brief Implementation of the bf::sprite_edit class.
 * \author Sebastien Angibaud
 */
#include "bf/sprite_edit.hpp"

#include "bf/image_selection_dialog.hpp"
#include "bf/wx_facilities.hpp"
#include "bf/image_pool.hpp"
#include <limits>

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param parent The window owning this window.
 * \param sprite The initial sprite.
 */
bf::sprite_edit::sprite_edit( wxWindow& parent, const sprite& spr )
  : wxPanel(&parent, wxID_ANY), base_edit<sprite>(spr)
{
  create_controls();
  Fit();
} // sprite_edit::sprite_edit()

/*----------------------------------------------------------------------------*/
/**
 * \brief Check if the displayed value is correct and, if it is, set the
 *        value according to the display.
 */
bool bf::sprite_edit::validate()
{
  sprite spr;

  spr.set_left( m_left_text->GetValue() );
  spr.set_top( m_top_text->GetValue() );
  spr.set_clip_width( m_clip_width_text->GetValue() );
  spr.set_clip_height( m_clip_height_text->GetValue() );
  spr.set_width( m_width_text->GetValue() );
  spr.set_height( m_height_text->GetValue() );
  spr.set_flip_x( m_flip_x_box->IsChecked() );
  spr.set_flip_y( m_flip_y_box->IsChecked() );
  spr.set_alpha( m_alpha_spin->GetValue() );
    
  spr.set_image_name( wx_to_std_string(m_image_name_text->GetValue()) );

  set_value(spr);

  return true;
} // sprite_edit::validate()

/*----------------------------------------------------------------------------*/
/**
 * \brief Method called after changing the value by set_value().
 */
void bf::sprite_edit::value_updated()
{
  fill_controls();
} // sprite_edit::value_updated()

/*----------------------------------------------------------------------------*/
/**
 * \brief Fill the controls with the values of the sprite.
 */
void bf::sprite_edit::fill_controls()
{
  sprite spr = get_value();

  m_left_text->SetValue( wxString::Format(wxT("%d"), spr.get_left()) );
  m_top_text->SetValue( wxString::Format(wxT("%d"), spr.get_top()) );
  m_clip_width_text->SetValue
    ( wxString::Format(wxT("%d"), spr.get_clip_width()) );
  m_clip_height_text->SetValue
    ( wxString::Format(wxT("%d"), spr.get_clip_height()) );
  m_width_text->SetValue( wxString::Format(wxT("%d"), spr.get_width()) );
  m_height_text->SetValue( wxString::Format(wxT("%d"), spr.get_height()) );
  m_image_name_text->SetValue( std_to_wx_string(spr.get_image_name()) );
  m_flip_x_box->SetValue( spr.get_flip_x() );
  m_flip_y_box->SetValue( spr.get_flip_y() );
  m_alpha_spin->SetValue( spr.get_alpha() );

  control_sprite_size();
} // sprite_edit::fill_controls()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the controls of the window.
 */
void bf::sprite_edit::create_controls()
{
  create_sizer_controls();
  fill_controls();
} // sprite_edit::create_controls()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the controls and add them in sizers.
 */
void bf::sprite_edit::create_sizer_controls()
{
  wxBoxSizer* sizer = new wxBoxSizer( wxVERTICAL );

  sizer->Add( create_position_sizer(), 0, wxEXPAND );
  sizer->Add( create_size_sizer(), 0, wxEXPAND );
  sizer->Add( create_options_sizer(), 0, wxEXPAND );
  sizer->Add( create_image_name_sizer(), 0, wxEXPAND );

  SetSizer(sizer);
} // sprite_edit::create_sizer_controls()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the controls and the sizer for the position part.
 */
wxSizer* bf::sprite_edit::create_position_sizer()
{
  wxStaticBoxSizer* result =
    new wxStaticBoxSizer( wxVERTICAL, this, _("Rectangle in image source") );
  
  m_left_text = new wxSpinCtrl( this, IDC_RECTANGLE_CLIP_CHANGE );
  m_top_text = new wxSpinCtrl( this, IDC_RECTANGLE_CLIP_CHANGE );
  m_clip_width_text = new wxSpinCtrl( this, IDC_RECTANGLE_CLIP_CHANGE );
  m_clip_height_text = new wxSpinCtrl( this, IDC_RECTANGLE_CLIP_CHANGE );
  m_clip_width_text->SetRange( 0, std::numeric_limits<int>::max() );
  m_clip_height_text->SetRange( 0, std::numeric_limits<int>::max() );

  wxBoxSizer* s_sizer = new wxBoxSizer( wxHORIZONTAL );
  s_sizer->Add
    ( new wxStaticText(this, wxID_ANY, _("X:")),
      1, wxALIGN_CENTRE_VERTICAL | wxALL, 5 );
  s_sizer->Add( m_left_text, 0, wxALL, 5 );
  s_sizer->Add
    ( new wxStaticText(this, wxID_ANY, _("Y:")),
      1, wxALIGN_CENTRE_VERTICAL | wxALL, 5 );
  s_sizer->Add( m_top_text, 0, wxALL, 5 );
  result->Add( s_sizer, 0, wxEXPAND, 0 );

  s_sizer = new wxBoxSizer( wxHORIZONTAL );
  s_sizer->Add
    ( new wxStaticText(this, wxID_ANY, _("Width:")),
      1, wxALIGN_CENTRE_VERTICAL | wxALL, 5 );
  s_sizer->Add( m_clip_width_text, 0, wxALL, 5 );
  s_sizer->Add
    ( new wxStaticText(this, wxID_ANY, _("Height:")),
      1, wxALIGN_CENTRE_VERTICAL | wxALL, 5 );
  s_sizer->Add( m_clip_height_text, 0, wxALL, 5 );
  result->Add( s_sizer, 0, wxEXPAND, 0 );

  return result;
} // sprite_edit::create_position_sizer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the controls and the sizer for the size part.
 */
wxSizer* bf::sprite_edit::create_size_sizer()
{
  wxStaticBoxSizer* result =
    new wxStaticBoxSizer( wxHORIZONTAL, this, _("Size on the screen") );

  m_width_text = new wxSpinCtrl( this, wxID_ANY );
  m_height_text = new wxSpinCtrl( this, wxID_ANY );
  m_width_text->SetRange( 0, std::numeric_limits<int>::max() );
  m_height_text->SetRange( 0, std::numeric_limits<int>::max() );

  result->Add
    ( new wxStaticText(this, wxID_ANY, _("Width:")),
      1, wxALIGN_CENTRE_VERTICAL | wxALL, 5 );
  result->Add( m_width_text, 0, wxALL, 5 );

  result->Add
    ( new wxStaticText(this, wxID_ANY, _("Height:")),
      1, wxALIGN_CENTRE_VERTICAL | wxALL, 5 );
  result->Add( m_height_text, 0, wxALL, 5 );

  return result;
} // sprite_edit::create_size_sizer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the controls and the sizer for the options part.
 */
wxSizer* bf::sprite_edit::create_options_sizer()
{
  wxStaticBoxSizer* result = 
    new wxStaticBoxSizer( wxHORIZONTAL, this, _("Options") );

  m_flip_x_box = new wxCheckBox( this, wxID_ANY, _("Flip X") );
  m_flip_y_box = new wxCheckBox( this, wxID_ANY, _("Flip Y") );
  m_alpha_spin = 
    new spin_ctrl<double>( this, wxID_ANY,wxDefaultPosition,
			   wxDefaultSize, wxEXPAND,0,1,1,0.01 );

  result->Add( m_flip_x_box, 0, wxEXPAND , 0 );
  result->Add( m_flip_y_box, 0, wxEXPAND , 0 );
  result->Add
    ( new wxStaticText(this, wxID_ANY, _("Transparency:")),
      1, wxALIGN_CENTRE_VERTICAL | wxALL, 5 );

  result->Add( m_alpha_spin, 1, wxEXPAND , 0 );
  
  return result;
} // sprite_edit::create_options_sizer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the controls and the sizer for the image name.
 */
wxSizer* bf::sprite_edit::create_image_name_sizer()
{
  wxBoxSizer* result = new wxBoxSizer( wxHORIZONTAL );

  m_image_name_text = new wxTextCtrl( this, wxID_ANY );
  m_image_name_browse = new wxButton
    ( this, IDC_IMAGE_SELECT, wxT("..."), wxDefaultPosition, wxSize(30, -1) );

  result->Add
    ( new wxStaticText(this, wxID_ANY, _("Image:")),
      0, wxALIGN_CENTRE_VERTICAL | wxALL, 5 );
  result->Add( m_image_name_text, 1, wxEXPAND | wxALL, 5 );
  result->Add( m_image_name_browse, 0, wxALL, 5 );

  return result;
} // sprite_edit::create_image_name_sizer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Test validity of the sprite size.
 */
void bf::sprite_edit::control_sprite_size()
{
  if( image_pool::get_instance().has_image(m_image_name_text->GetValue()) ) 
    {
      wxBitmap bmp = 
	image_pool::get_instance().get_image(m_image_name_text->GetValue());
     
      m_left_text->SetRange( 0, bmp.GetWidth() - 1 );
      m_top_text->SetRange( 0, bmp.GetHeight() - 1 );
      m_clip_width_text->SetRange
	( 0, bmp.GetWidth() - m_left_text->GetValue() );
      m_clip_height_text->SetRange
	( 0, bmp.GetHeight() - m_top_text->GetValue() );
    }
} // sprite_edit::control_sprite_size()

/*----------------------------------------------------------------------------*/
/**
 * \brief The user clicked on the button to select an image.
 * \param event This event occured.
 */
void bf::sprite_edit::on_image_select( wxCommandEvent& event )
{
  image_selection_dialog dlg(*this, m_image_name_text->GetValue());

  if ( dlg.ShowModal() == wxID_OK )
    {
      m_image_name_text->SetValue( dlg.get_image_name() );
      
      if ( ( m_width_text->GetValue() == 0 ) || 
           ( m_height_text->GetValue() == 0 ) ) 
        {
          wxBitmap bmp = 
            image_pool::get_instance().get_image(dlg.get_image_name());

	  m_width_text->SetValue(bmp.GetWidth());
          m_height_text->SetValue(bmp.GetHeight());
	}
      
      if ( ( m_clip_width_text->GetValue() == 0 ) || 
           ( m_clip_height_text->GetValue() == 0 ) ) 
        {
          wxBitmap bmp = 
            image_pool::get_instance().get_image(dlg.get_image_name());
	  
	  m_clip_width_text->SetValue(bmp.GetWidth());
          m_clip_height_text->SetValue(bmp.GetHeight());
        }
      
      control_sprite_size();
    }
} // sprite_edit::on_image_select()

/*----------------------------------------------------------------------------*/
/**
 * \brief The user change a parameter of rectangle clip.
 * \param event The event.
 */
void bf::sprite_edit::on_rectangle_clip_change( wxSpinEvent& event )
{
  control_sprite_size();
} // sprite_edit::on_rectangle_clip_change()

/*----------------------------------------------------------------------------*/
BEGIN_EVENT_TABLE(bf::sprite_edit, wxPanel)
  EVT_BUTTON( bf::sprite_edit::IDC_IMAGE_SELECT,
              bf::sprite_edit::on_image_select )
  EVT_SPINCTRL( bf::sprite_edit::IDC_RECTANGLE_CLIP_CHANGE,
                bf::sprite_edit::on_rectangle_clip_change )
END_EVENT_TABLE()
