/*
    Plee The Bear - Level editor

    Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file bf/impl/level_file_xml_reader.tpp
 * \brief Implementation of the template methods of the
 *        bf::level_file_xml_reader class.
 * \author Julien Jorge
 */

#include "bf/stream_conv.hpp"
#include "bf/xml_exception.hpp"
#include "bf/wx_facilities.hpp"

#include <claw/assert.hpp>
#include <claw/logger.hpp>
#include <sstream>

/*----------------------------------------------------------------------------*/
/**
 * \brief Read the value from a xml value node.
 * \param v (out) The value we have read.
 * \param node The node from which we read the value.
 */
template<typename Type>
void bf::xml_to_value<Type>::operator()
  ( Type& v, wxXmlNode* node ) const
{
  CLAW_PRECOND( node != NULL );
  wxString val;

  if ( !node->GetPropVal( wxT("value"), &val ) )
    throw xml_missing_property( "value" );

  const std::string std_val( wx_to_std_string(val) );
  std::istringstream iss( std_val );

  if ( !stream_conv<Type>::read(iss, v) )
    throw xml_bad_value( wx_to_std_string(node->GetName()), std_val );
} // xml_to_value::operator()()




/*----------------------------------------------------------------------------*/
/**
 * \brief Load the value of a field of simple type.
 * \param item The item in which we save the value.
 * \param field_name The name of the field to load.
 * \param node_name The name of the xml node.
 * \param node The node to parse.
 */
template<typename Type>
void bf::level_file_xml_reader::load_value
( item_instance& item, const std::string& field_name,
  const std::string& node_name, wxXmlNode* node ) const
{
  bool found = false;
  Type v;
  const wxString wx_node_name = std_to_wx_string(node_name);

  if ( node == NULL )
    throw xml_missing_node( node_name );
  
  xml_to_value<Type> xml_conv;

  // we set all values found so the field has the value of the last one.
  for ( ; node!=NULL; node=node->GetNext() )
    if ( node->GetName() == wx_node_name )
      {
        found = true;
        xml_conv( v, node );
      }
    else
      claw::logger << claw::log_warning << "Ignored node '"
                   << wx_to_std_string(node->GetName()) << "'" << claw::lendl;

  if ( !found )
    throw xml_missing_node( node_name );

  item.set_value( field_name, v );
} // level_file_xml_reader::load_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Load the value of a field of type list of simple type.
 * \param item The item in which we save the value.
 * \param field_name The name of the field to load.
 * \param node_name The name of the xml node.
 * \param node The node to parse.
 */
template<typename Type>
void bf::level_file_xml_reader::load_value_list
( item_instance& item, const std::string& field_name,
  const std::string& node_name, wxXmlNode* node ) const
{
  std::list<Type> v;
  const wxString wx_node_name = std_to_wx_string(node_name);

  xml_to_value<Type> xml_conv;

  for ( ; node!=NULL; node=node->GetNext() )
    if ( node->GetName() == wx_node_name )
      {
        Type tmp;
        xml_conv( tmp, node );
        v.push_back(tmp);
      }
    else
      claw::logger << claw::log_warning << "Ignored node '"
                   << wx_to_std_string(node->GetName()) << "'" << claw::lendl;

  item.set_value( field_name, v );
} // level_file_xml_reader::load_value_list()
